package com.transsion.search.fragment

import android.net.Network
import android.net.NetworkCapabilities
import android.net.Uri
import android.os.Bundle
import android.text.Editable
import android.text.TextUtils
import android.text.TextWatcher
import android.view.Gravity
import android.view.KeyEvent
import android.view.LayoutInflater
import android.view.View
import android.view.inputmethod.EditorInfo
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.activity.OnBackPressedCallback
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import androidx.fragment.app.commit
import androidx.fragment.app.commitNow
import androidx.lifecycle.LifecycleOwner
import com.alibaba.android.arouter.launcher.ARouter
import com.blankj.utilcode.util.KeyboardUtils
import com.tn.lib.ext.gone
import com.tn.lib.ext.visible
import com.tn.lib.util.networkinfo.NetworkUtil
import com.tn.lib.util.networkinfo.OnNetworkStatusChangedListener
import com.tn.lib.widget.toast.TRToast
import com.tn.lib.widget.toast.core.ToastUtils
import com.transsion.baselib.helper.DeeplinkHelper
import com.transsion.baseui.fragment.BaseFragment
import com.transsion.moviedetailapi.MovieDetailConstants
import com.transsion.moviedetailapi.MovieDetailPageIdentity
import com.transsion.moviedetailapi.SubjectType
import com.transsion.search.R
import com.transsion.search.SearchManager
import com.transsion.search.activity.SearchManagerActivity.Companion.log
import com.transsion.search.bean.SuggestEntity
import com.transsion.search.constant.SearchConstants
import com.transsion.search.databinding.FragmentSearchSubjectBinding
import com.transsion.search.fragment.hot.SearchHotFragment
import com.transsion.search.fragment.result.SearchResultFragment
import com.transsion.search.fragment.suggest.SearchSuggestFragment
import com.transsion.search.fragment.suggest.SearchSuggestMultiAdapter
import com.transsion.search.speech.SRLogger
import com.transsion.search.widget.SearchReportHelper
import com.transsnet.downloader.DownloadManagerApi

/**
 *  首页入口用 ，主搜
 */
class SearchSubjectFragment : BaseFragment<FragmentSearchSubjectBinding>() {
    private var page: Int = 1

    private var mKeyword: String = ""
    private var hintSearchWord: String? = null
    private var hintSearchIndex: Int = 0
    private var backExitSearch: Boolean = false

    private var currentPage: Fragment? = null
    private var hotFragment: SearchHotFragment? = null
    private var suggestFragment: SearchSuggestFragment? = null
    private var searchResultFragment: SearchResultFragment? = null

    /***suggest不响应输入框变化*/
    private var ignoreEditChanged = false

    /***推荐点击过来的*/
    private var isSuggestClick = false

    /***上一次搜索推荐的 keyword*/
    private var mKeyWord = ""

    override fun getViewBinding(inflater: LayoutInflater): FragmentSearchSubjectBinding {
        return FragmentSearchSubjectBinding.inflate(inflater)
    }

    override fun initView(view: View, savedInstanceState: Bundle?) {
        //每次重新进入搜索页面，重置searchId
        BaseSearchMainFragment.resetSearchId()
        initEditHint()

        // 这里监听了网络的状态
        netListener = object : OnNetworkStatusChangedListener {
            override fun onDisconnected() {
                //
            }

            override fun onConnected(network: Network, networkCapabilities: NetworkCapabilities) {
                if (currentPage is SearchHotFragment) {
                    // 重新请求
                    (currentPage as SearchHotFragment).onConnected()
                }
            }
        }

        //语音搜索结果 回调
        mViewBinding?.sRView?.setCallback {
            if (TextUtils.isEmpty(it)) {
                // 拉起键盘
                mViewBinding?.commentInputEditText?.postDelayed({
                    mViewBinding?.commentInputEditText?.requestFocus()
                    mViewBinding?.commentInputEditText?.let { KeyboardUtils.showSoftInput(it) }
                }, 500)
                SRLogger.logW("${javaClass.simpleName} --> mViewBinding?.sRView?.setCallback{} --> onResult = $it --> 拉起键盘")
            } else {
                setText(it ?: "")
                // 右上角搜索按钮
                searchJob(SearchReportHelper.SEARCH_FROM_VOICE)
                SRLogger.logW("${javaClass.simpleName} --> mViewBinding?.sRView?.setCallback{} --> onResult = $it --> 隐藏键盘")
            }
        }

        // 提前初始化
        addHotFragment()
        addSearchSuggestFragment()
        addSearchValuesFragment()
        setCurrentPage(hotFragment)
    }

    fun setNextHint() {
        if (SearchConstants.searchHotWords.isNullOrEmpty().not()) {
            val size = SearchConstants.searchHotWords?.size?: 0
            val currentHint = SearchConstants.searchHotWords?.getOrNull(hintSearchIndex++ % size)
            if (currentHint.isNullOrEmpty().not()) {
                hintSearchWord = currentHint
            }
            mViewBinding?.commentInputEditText?.apply {
                hint = hintSearchWord
            }
        }
    }

    private fun initEditHint() {
        hintSearchWord = arguments?.getString(SearchConstants.HOT_SEARCH_WORD)

        if (hintSearchWord?.isNotEmpty() == true) {
            mViewBinding?.commentInputEditText?.apply {
                hint = hintSearchWord
            }
        }
    }

    private fun addHotFragment() {
        if (null == hotFragment) {
            childFragmentManager.commit(allowStateLoss = true) {
                hotFragment = SearchHotFragment.newInstance(hintSearchWord).apply {
                    add(R.id.search_fragment_container, this)
                }
            }
        }
    }


    fun clickKeyWordSearch(keyWord: String, searchFrom: String?, callback: (String) -> Unit) {
        activity?.let { KeyboardUtils.hideSoftInput(it) }
        mKeyword = keyWord
        ignoreEditChanged = true
        mViewBinding?.commentInputEditText?.setText(mKeyword)
        mViewBinding?.commentInputEditText?.setSelection(mKeyword.length)
        page = 1
        searchJob(searchFrom)
        SearchManager.INSTANCE.addSearch(mKeyword)
        callback.invoke(keyWord)
    }

    override fun initData(view: View, savedInstanceState: Bundle?) {
        super.initData(view, savedInstanceState)
        val keyWord = arguments?.getString(SearchConstants.SEARCH_KEY_WORD) ?: ""
        backExitSearch = arguments?.getBoolean(SearchConstants.SEARCH_KEY_EXIT_WHEN_BACK) ?: false
        setListener(keyWord.isNotEmpty())
        //带词搜索
        if (keyWord.isNotEmpty()) {
            mKeyword = keyWord
            mViewBinding?.commentInputEditText?.setText(mKeyword)
            mViewBinding?.searchEditClear?.visible()
            searchJob(SearchReportHelper.SEARCH_FROM_DEEPLINK)
        }
    }

    private fun setListener(withoutKeyboard: Boolean) {
        mViewBinding?.commentInputEditText?.setOnClickListener {
            mViewBinding?.tvSearch?.visible()
        }

        mViewBinding?.tvSearch?.setOnClickListener {
            // 右上角搜索按钮
            searchJob(SearchReportHelper.SEARCH_FROM_INPUT)
        }

        mViewBinding?.commentInputEditText?.addTextChangedListener(object : TextWatcher {
            override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
                mViewBinding?.tvSearch?.visible()
            }

            override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
                //
            }

            override fun afterTextChanged(s: Editable?) {
                // 拦截掉首页热搜词点击search的场景，避免推荐和search同时出现
                if (ignoreEditChanged) {
                    ignoreEditChanged = false
                    return
                }

                mKeyword = s.toString()
                showSuggestSearchFragment(mKeyword)
            }
        })
        mViewBinding?.commentInputEditText?.setOnEditorActionListener(object :
            TextView.OnEditorActionListener {
            override fun onEditorAction(tv: TextView?, i: Int, keyEvent: KeyEvent?): Boolean {
                if (i == EditorInfo.IME_ACTION_SEARCH || i == EditorInfo.IME_ACTION_UNSPECIFIED) {
                    mKeyword = tv?.text.toString()
                    // 输入框搜索按钮
                    searchJob(SearchReportHelper.SEARCH_FROM_INPUT)
                    return true
                }
                return false
            }
        })

        //监听焦点变化，处理页面切换
        mViewBinding?.commentInputEditText?.setOnFocusChangeListener { v, hasFocus ->
            if (hasFocus) {
                mViewBinding?.tvSearch?.visible()
                mViewBinding?.searchEditClear?.gone()
                mViewBinding?.commentInputEditText?.layoutParams?.let {
                    if (it is RelativeLayout.LayoutParams) {
                        it.addRule(RelativeLayout.START_OF, R.id.tv_search)
                    }
                }

                //在搜索结果页重新获取焦点时处理页面切换
                if (currentPage is SearchResultFragment) {
                    val text = mViewBinding?.commentInputEditText?.text
                    if ((text?.length?: 0) > 0) {
                        showSuggestSearchFragment(text?.toString() ?: "")
                    } else {
                        setCurrentPage(hotFragment)
                    }
                }
            } else {
                mViewBinding?.tvSearch?.gone()
                mViewBinding?.searchEditClear?.visible()
                mViewBinding?.commentInputEditText?.layoutParams?.let {
                    if (it is RelativeLayout.LayoutParams) {
                        it.addRule(RelativeLayout.START_OF, R.id.search_edit_clear)
                    }
                }
            }
        }

        //清除按钮
        mViewBinding?.searchEditClear?.setOnClickListener {
            mViewBinding?.commentInputEditText?.setText("")
        }

        mViewBinding?.ivCancel?.setOnClickListener {
            handleBack()
        }

        if (withoutKeyboard.not()) {
            mViewBinding?.commentInputEditText?.requestFocus()
        }

        logViewConfig?.isLoadSuccess = true
        addOnBackPressed(this) { onBackPressed() }
    }

    private fun handleBack() {
        if (currentPage is SearchHotFragment) {
            activity?.finish()

        } else if (!isSuggestClick || currentPage is SearchSuggestFragment) {
            isSuggestClick = false
            mViewBinding?.commentInputEditText?.setText("")

        } else if (currentPage is SearchResultFragment) {
            setCurrentPage(suggestFragment)
        }
    }

    /**
     * 页面管理唯一入口
     */
    private fun setCurrentPage(fragment: Fragment?) {
        if (fragment == null || currentPage == fragment) {
            return
        }
        if (!isAdded || isDetached || isRemoving) {
            return
        }
        currentPage = fragment

        childFragmentManager
            .beginTransaction().apply {
                hotFragment?.let {
                    if (currentPage != it) {
                        hide(it)
                    }
                }

                suggestFragment?.let {
                    if (currentPage != it) {
                        hide(it)
                    }
                }

                searchResultFragment?.let {
                    if (currentPage != it) {
                        hide(it)
                    }
                }

                show(fragment)
            }.commitAllowingStateLoss()

        //输入框焦点处理统一收口在这边
        if (fragment == searchResultFragment) {
            mViewBinding?.commentInputEditText?.let {
                it.clearFocus()
                KeyboardUtils.hideSoftInput(it)
            }
        } else {
            mViewBinding?.commentInputEditText?.let {
                it.requestFocus()
                KeyboardUtils.showSoftInput(it)
            }
        }
    }


    /**
     * 开始执行搜索操作
     */
    fun searchJob(searchFrom: String?) {
        if (!NetworkUtil.hasCapabilityAsync()) {
            TRToast.show(com.tn.lib.widget.R.string.no_network_toast)
        }
        var finialSearchFrom = searchFrom

        // 首次进入search，hint为热搜时，点击search直接使用热搜词去搜索
        if (mKeyword.isEmpty()) {
            mViewBinding?.commentInputEditText?.apply {
                if (hint == hintSearchWord) {
                    mKeyword = hintSearchWord.toString()
                    ignoreEditChanged = true  // 避免搜索结果页点击back，出现的白色中间页，还要再点击一次back才能返回搜索作品页
                    setText(hintSearchWord.toString())
                    setSelection(text.length)
                    finialSearchFrom = SearchReportHelper.SEARCH_FROM_HINT
                }
            }
        }

        if (mKeyword.isEmpty()) {
            ToastUtils.setGravity(Gravity.CENTER_VERTICAL)
            ToastUtils.show(getString(R.string.tips_movie))
            return
        }

        hotFragment?.addHistoryWord(mKeyword)

        //触发搜索
        searchResultFragment?.startSearch(mKeyword, finialSearchFrom)
        setCurrentPage(searchResultFragment)
    }

    companion object {
        const val PAGE_NAME = "searchpage"
        fun newInstance(
            hotSearchWord: String?,
            keyWord: String?,
            backExitSearch: Boolean,
        ): SearchSubjectFragment {
            val fragment = SearchSubjectFragment()
            val bundle = Bundle()
            bundle.putString(SearchConstants.HOT_SEARCH_WORD, hotSearchWord)
            bundle.putString(SearchConstants.SEARCH_KEY_WORD, keyWord)
            bundle.putBoolean(SearchConstants.SEARCH_KEY_EXIT_WHEN_BACK, backExitSearch)
            fragment.arguments = bundle
            return fragment
        }
    }

    private fun addOnBackPressed(owner: LifecycleOwner, onBackPressed: () -> Boolean): OnBackPressedCallback {
        val callback = object : OnBackPressedCallback(true) {
            override fun handleOnBackPressed() {
                if (!onBackPressed()) {
                    isEnabled = false
                    requireActivity().onBackPressedDispatcher.onBackPressed()
                    isEnabled = true
                }
            }
        }
        requireActivity().onBackPressedDispatcher.addCallback(owner, callback)
        return callback
    }


    private fun onBackPressed(): Boolean {
        handleBack()
        return true
    }

    override fun lazyLoadData() {
        //
    }


    // =============================================================================================

    /***自定义搜索内容*/
    private fun setText(keyWord: String) {
        mViewBinding?.commentInputEditText?.setText(keyWord)
        mViewBinding?.commentInputEditText?.setSelection(keyWord.length)
    }

    /**
     * 隐藏 系统的输入弹窗
     */
    fun hideSoftInput() {
        // 收起输入框
        mViewBinding?.commentInputEditText?.let {
            KeyboardUtils.hideSoftInput(it)
        }
    }

    /**
     * 初始化
     */
    private fun addSearchSuggestFragment() {
        // 如果已经创建了就直接使用
        if (null == suggestFragment) {
            childFragmentManager.commit(allowStateLoss = true) {
                suggestFragment = SearchSuggestFragment.newInstance().apply {
                    add(R.id.search_fragment_container, this)
                }
            }

            // 点击事件回调
            suggestFragment?.setOnSuggestClickListener(object :
                SearchSuggestFragment.OnSuggestClickListener {
                override fun onItemCLick(data: SuggestEntity?, position: Int, keyWord: String, mOps: String) {
                    // 收起输入框
                    mViewBinding?.commentInputEditText?.let {
                        KeyboardUtils.hideSoftInput(it)
                    }

                    when (data?.type) {
                        SearchSuggestMultiAdapter.SEARCH_SUGGEST_PROVIDER_LIST_COVER -> {
                            data.verticalRank?.deeplink?.let {
                                val uri = DeeplinkHelper.toRouteUri(Uri.parse(it + "&ops=${mOps}"))
                                if (uri != null) {
                                    ARouter.getInstance().build(uri).navigation()
                                }
                            }
                        }

                        SearchSuggestMultiAdapter.SEARCH_SUGGEST_PROVIDER_MOVIE -> {
                            try {
                                if (data.subject?.subjectType == SubjectType.SHORT_TV.value) {
                                    DownloadManagerApi.INSTANCE.downloadShortTV(
                                        activity = context as FragmentActivity,
                                        page_from = PAGE_NAME,
                                        groupId = "",
                                        opss = data.subject?.ops,
                                        moduleName = "download_subject",
                                        scroll2Download = false,
                                        subject = data.subject
                                    )
                                } else {
                                    ARouter.getInstance()
                                        .build(MovieDetailPageIdentity.MOVIE_DETAIL)
                                        .withInt(
                                            MovieDetailConstants.SUBJECT_TYPE,
                                            data.subject?.subjectType ?: SubjectType.MOVIE.value
                                        )
                                        .withString(
                                            MovieDetailConstants.SUBJECT_ID,
                                            data.subject?.subjectId ?: ""
                                        ).withString(
                                            MovieDetailConstants.OPS,
                                            mOps
                                        ).navigation()
                                }
                            } catch (e: Exception) {
                                //
                            }
                        }

                        SearchSuggestMultiAdapter.SEARCH_SUGGEST_PROVIDER_COPY_WRITING -> {
                            mKeyWord = keyWord
                            isSuggestClick = true
                            setText(data.word ?: "")
                            searchJob(SearchReportHelper.SEARCH_FROM_SUGGEST)
                        }

                        SearchSuggestMultiAdapter.SEARCH_SUGGEST_PROVIDER_STAFF -> {
                            data.staff?.let {
                                ARouter.getInstance().build(MovieDetailPageIdentity.MOVIE_STAFF)
                                    .withSerializable("staff", it).navigation()
                            }
                        }
                    }
                }
            })
        }
    }

    /***展示隐藏联动搜索界面*/
    private fun showSuggestSearchFragment(keyword: String) {
        if (keyword.isEmpty()) {
            setCurrentPage(hotFragment)
            return
        }

        // SearchLinkageFragment --> 如果没有网络直接返回
        if (!NetworkUtil.hasCapabilityAsync()) {
            log("showLinkageSearchFragment --> currently no network does not use linkage")
            return
        }

        // 执行搜索事件
        if (!TextUtils.isEmpty(keyword)) {
            suggestFragment?.search(keyword)
        }

        setCurrentPage(suggestFragment)
    }


    // =============================================================================================

    /**
     * 初始化的时候就 创建一个Fragment -- 用于展示搜索结果的页面
     */
    private fun addSearchValuesFragment() {
        // 搜索结果初始化
        if (null == searchResultFragment) {
            childFragmentManager.commitNow(allowStateLoss = true) {
                searchResultFragment = SearchResultFragment().apply {
                    add(R.id.search_fragment_container, this)
                }
            }
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        currentPage = null
        hotFragment = null
        suggestFragment = null
        searchResultFragment = null

        netListener = null

        mViewBinding = null
    }

    override fun onDestroy() {
        super.onDestroy()
        currentPage = null
        hotFragment = null
        suggestFragment = null
        searchResultFragment = null
    }
}