package com.transsion.search.fragment.hot.adapter

import android.app.Activity
import android.text.SpannableStringBuilder
import android.text.TextUtils
import android.text.style.ForegroundColorSpan
import android.view.Gravity
import android.view.View
import android.widget.TextView
import androidx.fragment.app.FragmentActivity
import com.alibaba.android.arouter.launcher.ARouter
import com.blankj.utilcode.util.ColorUtils
import com.blankj.utilcode.util.KeyboardUtils
import com.blankj.utilcode.util.SizeUtils
import com.chad.library.adapter.base.BaseQuickAdapter
import com.chad.library.adapter.base.module.LoadMoreModule
import com.chad.library.adapter.base.viewholder.BaseViewHolder
import com.transsion.baselib.utils.DeviceInfo
import com.transsion.baseui.image.ImageHelper
import com.transsion.moviedetailapi.MovieDetailConstants
import com.transsion.moviedetailapi.MovieDetailPageIdentity
import com.transsion.moviedetailapi.SubjectType
import com.transsion.moviedetailapi.bean.Subject
import com.transsion.moviedetailapi.getSubjectIconId
import com.transsion.search.R
import com.transsion.search.bean.HotSubject
import com.transsion.search.fragment.hot.SearchHotFragment
import com.transsion.search.getImageSpan
import com.transsion.search.getResultSplitSpan
import com.transsion.search.widget.SearchReportHelper
import com.transsnet.downloader.DownloadManagerApi
import java.util.Locale

/**
 * Created by gaoxin.liu on 2024/5/20.
 * Description: 搜索首页热榜子view
 */
class SearchRankItemAdapter(
    private val parentCategory: String,
    private val parentIndex: Int,
    private val isTabLayout: Boolean,
) : BaseQuickAdapter<HotSubject, BaseViewHolder>(R.layout.item_search_hot_rank_inner), LoadMoreModule {

    private val mIsLowDevice: Boolean = DeviceInfo.lowMemoryDevice()

    override fun convert(holder: BaseViewHolder, item: HotSubject) {
        val position = getItemPosition(item)

        //音乐改为横版封面
        if (item.isMusicType()) {
            holder.getView<View>(R.id.search_hot_rank_item_image_linear).apply {
                val scale = if (isTabLayout) 2 else 1
                layoutParams.width = SizeUtils.dp2px(80f * scale )
                layoutParams.height = SizeUtils.dp2px(45f * scale)
                layoutParams = layoutParams
            }

            holder.getView<TextView>(R.id.search_hot_rank_item_title_text).apply {
                maxLines = 1
            }
        }


        var bgRes = R.drawable.bg_hot_subject_no4
        when (position) {
            0 ->
                bgRes = R.drawable.bg_hot_subject_no1

            1 ->
                bgRes = R.drawable.bg_hot_subject_no2

            2 ->
                bgRes = R.drawable.bg_hot_subject_no3
        }
        holder.setText(R.id.search_hot_rank_item_index_text, (position + 1).toString())
        holder.setBackgroundResource(R.id.search_hot_rank_item_index_text, bgRes)

        if (mIsLowDevice) {
            holder.setGone(R.id.search_hot_rank_item_image_linear, true)
        } else {
            ImageHelper.loadNet(
                context,
                holder.getView(R.id.search_hot_rank_item_image),
                item.cover?.url ?: "",
                width = SizeUtils.dp2px(50f),
                thumbnail = item.cover?.thumbnail ?: ""
            )
        }

        holder.setText(R.id.search_hot_rank_item_title_text, item.title)

        //适配左右布局
        holder.getView<TextView>(R.id.search_hot_rank_item_title_text).gravity =
            getTextGravityByRTL()
        holder.getView<TextView>(R.id.search_hot_rank_item_subtitle_text).gravity =
            getTextGravityByRTL()

        holder.itemView.setOnClickListener {
            onItemClick(item, position)
        }

        //角标
        holder.setGone(R.id.search_hot_rank_item_corner, item.corner.isNullOrEmpty())
        holder.setText(R.id.search_hot_rank_item_corner, item.corner ?: "")

        /*---------------------------------描述----------------------------------------*/
        val sb = SpannableStringBuilder()

        //1. icon
        val iconSpan = getImageSpan(context, getSubjectIconId(item.subjectType))
        sb.append("\uFFFC")
        sb.setSpan(iconSpan, 0, 1, SpannableStringBuilder.SPAN_EXCLUSIVE_EXCLUSIVE)
        //2. 分隔符
        sb.append("\uFFFC")
        sb.setSpan(
            getResultSplitSpan(context), sb.length - 1, sb.length,
            SpannableStringBuilder.SPAN_EXCLUSIVE_EXCLUSIVE
        )

        //3. 评分icon
        if (item.imdbRatingValue != null && !item.isMusicType()) {
            val starSpan = getImageSpan(context, R.drawable.ic_search_result_star)
            sb.append("\uFFFC")
            sb.setSpan(
                starSpan, sb.length - 1, sb.length,
                SpannableStringBuilder.SPAN_EXCLUSIVE_EXCLUSIVE
            )

            sb.append(" ")
            sb.append(item.imdbRatingValue)
            val foregroundColorSpan =
                ForegroundColorSpan(ColorUtils.getColor(com.tn.lib.widget.R.color.yellow_60))
            val starLength = item.imdbRatingValue?.length ?: 0
            sb.setSpan(
                foregroundColorSpan, sb.length - starLength, sb.length,
                SpannableStringBuilder.SPAN_EXCLUSIVE_EXCLUSIVE
            )

            sb.append("\uFFFC")
            sb.setSpan(
                getResultSplitSpan(context), sb.length - 1, sb.length,
                SpannableStringBuilder.SPAN_EXCLUSIVE_EXCLUSIVE
            )
        }

        //4. 年份
        if (item.releaseDate != null) {
            val releaseDate = item.releaseDate ?: ""
            val year = if (releaseDate.length >= 4) {
                releaseDate.substring(0, 4)
            } else {
                releaseDate
            }

            sb.append(year)
            sb.append("\uFFFC")
            sb.setSpan(
                getResultSplitSpan(context), sb.length - 1, sb.length,
                SpannableStringBuilder.SPAN_EXCLUSIVE_EXCLUSIVE
            )
        }

        //5.国家
        if (item.countryName != null) {
            sb.append(item.countryName)
            sb.append("\uFFFC")
            sb.setSpan(
                getResultSplitSpan(context), sb.length - 1, sb.length,
                SpannableStringBuilder.SPAN_EXCLUSIVE_EXCLUSIVE
            )
        }

        //6. genre
        if (item.genre != null) {
            val genreList = item.genre?.split(",") ?: listOf()
            //只显示一个
            if (genreList.isNotEmpty()) {
                sb.append(genreList[0])
            }
        }
        holder.getView<TextView>(R.id.search_hot_rank_item_subtitle_text).text = sb
    }

    private fun onItemClick(subject: HotSubject, position: Int) {
        SearchReportHelper.reportRankSubjectClick(subject, position, parentCategory, parentIndex, mIsLowDevice)

        if (context is Activity) {
            if (KeyboardUtils.isSoftInputVisible(context as Activity)) {
                KeyboardUtils.hideSoftInput(context as Activity)
            }
        }

        if (subject.subjectType == SubjectType.SHORT_TV.value) {
            DownloadManagerApi.INSTANCE.downloadShortTV(
                activity = context as FragmentActivity,
                page_from = SearchHotFragment.PAGE_NAME,
                groupId = "",
                opss = subject.ops,
                moduleName = "download_subject",
                scroll2Download = false,
                subjectId = subject.subjectId
            )
        } else {
            ARouter.getInstance().build(MovieDetailPageIdentity.MOVIE_DETAIL)
                .withInt(
                    MovieDetailConstants.SUBJECT_TYPE,
                    subject.subjectType ?: SubjectType.MOVIE.value
                )
                .withString(MovieDetailConstants.SUBJECT_ID, subject.subjectId)
                .withString(MovieDetailConstants.OPS, subject.ops)
                .navigation()
        }
    }

    private fun getTextGravityByRTL(): Int {
        return if (TextUtils.getLayoutDirectionFromLocale(Locale.getDefault()) == View.LAYOUT_DIRECTION_RTL)
            Gravity.END
        else
            Gravity.START
    }
}