package com.transsion.search.fragment.result

import android.app.Application
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import com.google.gson.JsonObject
import com.tn.lib.logger.Logger
import com.tn.lib.net.base.BaseObserver
import com.tn.lib.net.dns.or.CacheIpPool.getHost
import com.tn.lib.net.env.Host
import com.tn.lib.net.manager.NetServiceGenerator
import com.transsion.publish.model.PostEntity
import com.transsion.search.bean.SearchList
import com.transsion.search.bean.SearchResultEntity
import com.transsion.search.fragment.values.SearchValuesFragment
import com.transsion.search.net.RequestSearchResultEntity
import com.transsion.search.net.SearchApi
import io.reactivex.rxjava3.schedulers.Schedulers
import kotlinx.coroutines.launch
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.RequestBody.Companion.toRequestBody

/**
 * Created by gaoxin.liu on 2025/3/18.
 * Description:
 */
class SearchResultViewModel(application: Application) : AndroidViewModel(application) {
    private val service: SearchApi by lazy {
        NetServiceGenerator.instance.getService(SearchApi::class.java)
    }

    val searchResultLiveData = MutableLiveData<SearchResultEntity?>()
    val searchMoreLiveData = MutableLiveData<SearchResultEntity?>()
    val requestMovieResult = MutableLiveData<PostEntity?>()
    private var page = 1

    /**
     * 初始化请求
     */
    fun requestSearchResult(keyword: String, tabId: String? = null, isLoadMore: Boolean = false) {
        viewModelScope.launch {
            val request = RequestSearchResultEntity(page, SearchResultTabFragment.PER_PAGE_COUNT, keyword)

            val json = JsonObject()
            json.addProperty("page", request.page)
            json.addProperty("perPage", request.perPage)
            json.addProperty("keyword", request.keyword)
            if (tabId.isNullOrEmpty().not()) {
                json.addProperty("tabId", tabId)
            }

            val body = json.toString().toRequestBody("application/json".toMediaTypeOrNull())

            runCatching {
                val result = service.searchSubjectV2(getHost(), body)
                result.data?.convertData = convertRenderData(result.data?.results)

                if (isLoadMore) {
                    searchMoreLiveData.postValue(result.data)
                } else {
                    searchResultLiveData.postValue(result.data)
                }
                page++

            }.onFailure {
                searchResultLiveData.postValue(null)
                Logger.e("SearchResultViewModel", "requestSearchResult error: ${it.message}")
            }
        }
    }

    //特殊情况设置
    fun setNextPage(nextPage: Int) {
        page = nextPage
    }

    /**
     * 将下发数据转为渲染数据
     */
    private fun convertRenderData(list: List<SearchList>?): List<ResultWrapData> {
        val result = mutableListOf<ResultWrapData>()
        if (list.isNullOrEmpty()) {
            return result
        }

        for (searchList in list) {
            //title标题
            if (searchList.title.isNullOrEmpty().not()) {
                result.add(ResultWrapData(ResultType.TITLE, title = searchList.title))
            }

            //遍历添加几个子类型
            searchList.subjects?.forEach {
                result.add(ResultWrapData(ResultType.SUBJECT, subject = it))
            }
            searchList.staffs?.forEach {
                result.add(ResultWrapData(ResultType.STAFF, staff = it))
            }
            searchList.groups?.forEach {
                result.add(ResultWrapData(ResultType.GROUP, groups = it))
            }
            searchList.verticalRanks?.forEach {
                result.add(ResultWrapData(ResultType.VERTICAL_RANK, verticalRank = it))
            }

            //跳转更多
            if (searchList.showMore == true) {
                result.add(ResultWrapData(ResultType.MORE, moreTabId = searchList.moreTabId))
            }
        }

        return result
    }

    fun postRequestResource(keyword: String, before: Boolean = false) {
        val jsonObject = JsonObject()
        jsonObject.addProperty("q", keyword)
        jsonObject.addProperty("before", before)

        val params = jsonObject.toString().toRequestBody("application/json".toMediaTypeOrNull())

        service.postRequestResource(Host.getHost(), params).subscribeOn(Schedulers.io())
            .subscribe(object : BaseObserver<PostEntity?>() {
                override fun onFailure(code: String?, message: String?) {
                    Logger.e(SearchValuesFragment.TAG, "SearchValuesFragment --> postRequestResource --> failed to post request resources $message")
                    requestMovieResult.postValue(null)
                }

                override fun onSuccess(data: PostEntity?) {
                    data?.let {
                        requestMovieResult.postValue(data)
                    }
                }
            })
    }
}