package com.transsion.search.fragment.suggest

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.LinearLayoutManager
import com.blankj.utilcode.util.Utils
import com.tn.lib.util.expose.RecyclerViewExposeUtil
import com.transsion.baselib.report.LogViewConfig
import com.transsion.baselib.report.Report
import com.transsion.search.bean.SuggestEntity
import com.transsion.search.databinding.FragmentSearchSuggestLayoutBinding
import com.transsion.search.fragment.BaseSearchMainFragment
import com.transsion.search.viewmodel.SearchViewModel
import com.transsion.search.widget.SearchReportHelper

/**
 * @author xinbing.zhang
 * @date :2023/2/8 16:51
 * @description:联动搜索界面 无网络的时候是进不到这里来的
 */
class SearchSuggestFragment : BaseSearchMainFragment<FragmentSearchSuggestLayoutBinding>() {

    companion object {
        const val PAGE_NAME_SEARCH_SUGGEST_FRAGMENT = "search_suggest"

        fun newInstance(): SearchSuggestFragment {
            return SearchSuggestFragment()
        }

        const val TAG = "suggest"
    }

    /**
     * 点击事件回调出去 SearchSubjectFragment 处理这个点击事件
     */
    interface OnSuggestClickListener {
        fun onItemCLick(data: SuggestEntity?, position: Int, keyWord: String, mOps: String)
    }


    // =============================================================================================


    /***点击事件回调*/
    private var mClickListener: OnSuggestClickListener? = null

    private val mViewModel by lazy {
        SearchViewModel(Utils.getApp())
    }

    private var mAdapter: SearchSuggestMultiAdapter? = null

    /***当前正在搜索的关键字*/
    private var mKeyWord: String = ""

    private var mOps: String? = null

    /**
     * 曝光组件
     */
    private val exposeUtil = RecyclerViewExposeUtil()


    // =============================================================================================


    override fun newLogViewConfig(): LogViewConfig {
        return LogViewConfig(PAGE_NAME_SEARCH_SUGGEST_FRAGMENT).apply {
            needPv = true
            paramsMap["search_id"] = sSearchId
        }
    }

    override fun getViewBinding(inflater: LayoutInflater): FragmentSearchSuggestLayoutBinding =
        FragmentSearchSuggestLayoutBinding.inflate(inflater)

    override fun initView(view: View, savedInstanceState: Bundle?)  {
        mViewBinding?.rvLinkage?.apply {
            layoutManager = LinearLayoutManager(context)
            mAdapter = SearchSuggestMultiAdapter().apply {
                setDiffCallback(object : DiffUtil.ItemCallback<SuggestEntity>() {
                    override fun areItemsTheSame(
                        oldItem: SuggestEntity, newItem: SuggestEntity
                    ): Boolean {
                        return oldItem.hashCode() == newItem.hashCode()
                    }

                    override fun areContentsTheSame(
                        oldItem: SuggestEntity, newItem: SuggestEntity
                    ): Boolean {
                        return oldItem.hashCode() == newItem.hashCode()
                    }
                })
            }
            adapter = mAdapter

            // 列表点击事件
            mAdapter?.setOnItemClickListener { _, _, position ->
                val get = mAdapter?.data?.get(position)
                SearchReportHelper.reportSuggestClick(get, position, mKeyWord, mOps)
                mClickListener?.onItemCLick(get, position, mKeyWord, mOps ?: "")
            }
        }
    }

    override fun initListener() {
        // 列表曝光
        exposeUtil.setRecyclerItemExposeListener(mViewBinding?.rvLinkage,
            object : RecyclerViewExposeUtil.OnItemExposeListener {
                override fun onItemViewVisible(visible: Boolean, position: Int) {
                    // 曝光事件上报
                    val get = mAdapter?.data?.getOrNull(position)
                    SearchReportHelper.reportSuggestBrowse(get, position, mKeyWord, mOps)
                }

                override fun getTotalSize(): Int {
                    return mAdapter?.data?.size ?: 0
                }
            })

        mViewModel.suggestLiveData.observe(this) { keyword ->
            // 无论成功与失败都从缓存里面获取
            val suggestList = mViewModel.suggestMap[keyword]
            if (null == suggestList || suggestList.items?.isEmpty() == true) {
                // 暂不处理 mAdapter?.setList(null)
            } else {
                // 展示数据
                mOps = suggestList.ops ?: ""
                mAdapter?.setKeyWord(mKeyWord)
                // 数据不一致才更新
                if (checkSame(suggestList.items, mAdapter?.data).not()) {
                    // 重置埋点数据
                    exposeUtil.reset()
                    mAdapter?.setList(suggestList.items)
                    exposeUtil.manualCallReporting()

                } else { // 数据一致 更新高亮词
                    mAdapter?.notifyDataSetChanged()
                }
            }
        }
    }

    override fun lazyLoadData() {
        //do nothing
    }


    // =============================================================================================



    private fun checkSame(newList: List<SuggestEntity>?, oldList: List<SuggestEntity>?): Boolean {
        // 内容完全相同可以不用更新
        val oldSize = oldList?.size ?: 0
        val newSize = newList?.size ?: 0
        if (newSize > 0 && oldSize == newSize) {
            newList?.forEachIndexed { index, suggestEntity ->
                // 如果是作品对比作品id
                if (suggestEntity.type == 1 && suggestEntity.subject?.subjectId != oldList?.get(
                        index
                    )?.subject?.subjectId
                ) {
                    return false
                } else {
                    if (suggestEntity.type != 1 && suggestEntity.hashCode() != oldList?.get(index)
                            .hashCode()
                    ) {
                        return false
                    }
                }
            }
            return true
        }
        return false
    }

    /**
     * 设置点击事件回调
     */
    fun setOnSuggestClickListener(listener: OnSuggestClickListener) {
        mClickListener = listener
    }

    /**
     * 用户搜索的词条
     *
     * 这个方法是输入框回调的地方调用的
     */
    fun search(keyWord: String) {
        mKeyWord = keyWord
        mViewModel.getSearchSuggest(keyWord)

        //pt 埋点上报最后一次keyword
        logViewConfig?.paramsMap?.put("keyword", keyWord)
    }

}