package com.transsion.search.speech

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.pm.ResolveInfo
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.speech.RecognitionService
import android.speech.SpeechRecognizer
import android.text.TextUtils
import com.blankj.utilcode.util.Utils
import com.transsion.baselib.report.launch.RoomAppMMKV

/**
 * @author: zhangxinbing
 * @date : 2024/11/4 20:21
 * @description: 不同的手机需要不同的初始化方法
 */
class InitializeStrategy {

    companion object {
        private const val MMKV_KEY_INITIALIZE_STRATEGY = "initialize_strategy"
        private const val MMKV_KEY_INITIALIZE_STRATEGY_ONE = "initialize_one"
        private const val MMKV_KEY_INITIALIZE_STRATEGY_TWO = "initialize_two"

        private const val MMKV_KEY_INITIALIZE_STRATEGY_AFFIRM = "initialize_strategy_affirm"
        private const val MMKV_KEY_INITIALIZE_STRATEGY_ONE_AFFIRM = "initialize_one_affirm"
    }

    private fun getClassTag(): String = javaClass.simpleName

    private var mCallback: (() -> Unit?)? = null

    private val mHandler by lazy {
        Handler(Looper.getMainLooper())
    }

    /**
     * 第一种策略 SpeechRecognizer.createSpeechRecognizer(context)
     */
    private fun getSpeechRecognizerOneParameterStrategy(context: Context?): SpeechRecognizer? {
        SRLogger.logD("${getClassTag()} --> getSpeechRecognizerOneParameterStrategy() --> deviceModel = ${Build.MODEL} --> 采用第一种策略")
        return SpeechRecognizer.createSpeechRecognizer(context)
    }

    /**
     * 第二种策略  SpeechRecognizer.createSpeechRecognizer(context, currentRecognitionCmp)
     */
    private fun getSpeechRecognizerTwoParameterStrategy(context: Context?): SpeechRecognizer? {
        return kotlin.runCatching {
            SRLogger.logD("${getClassTag()} --> deviceModel = ${Build.MODEL} --> 采用第二种策略")
            val list: List<ResolveInfo> = Utils.getApp().packageManager.queryIntentServices(
                Intent(RecognitionService.SERVICE_INTERFACE), 0
            )
            val info = list[0]
            SRLogger.logD("${getClassTag()} --> getSpeechRecognizerTwoParameterStrategy() --> packageName = ${info.serviceInfo.packageName} --> name = info.serviceInfo.name --> info = $info --> deviceModel = ${Build.MODEL}")
            val currentRecognitionCmp =
                ComponentName(info.serviceInfo.packageName, info.serviceInfo.name)
            SpeechRecognizer.createSpeechRecognizer(context, currentRecognitionCmp)
        }.getOrElse {
            null
        }
    }


    // =============================================================================================


    /**
     * 当第一种策略不支持的时候，回调告诉一下修改为第二种策略
     */
    fun setCallback(callback: () -> Unit) {
        mCallback = callback
    }

    /**
     * 启动了
     */
    fun startListening() {
        val strategy = RoomAppMMKV.mmkv.getString(
            MMKV_KEY_INITIALIZE_STRATEGY, MMKV_KEY_INITIALIZE_STRATEGY_ONE
        )
        // 如果已经标记使用第一种策略了，那就不需要在此再执行了
        if (TextUtils.equals(strategy, MMKV_KEY_INITIALIZE_STRATEGY_TWO)) {
            SRLogger.logD("${getClassTag()} --> startListening() --> 当前策略已经是第二种策略了")
            return
        }

        mHandler.removeCallbacksAndMessages(null)
        mHandler.postDelayed({
            // 标记当前不支持第一种策略
            RoomAppMMKV.mmkv.putString(
                MMKV_KEY_INITIALIZE_STRATEGY, MMKV_KEY_INITIALIZE_STRATEGY_TWO
            )
            SRLogger.logD("${getClassTag()} --> startListening() --> 标记当前不支持第一种策略 --> 回调给场景重新加载对象")
            mCallback?.invoke()
            SrReport.useSecondStrategy()
        }, 3000)
        SRLogger.logD("${getClassTag()} --> startListening() --> 开启倒计时")
    }

    /**
     * 当语音识别准备好时调用
     */
    fun onReadyForSpeech() {
        mHandler.removeCallbacksAndMessages(null)
        SRLogger.logD("${getClassTag()} --> onReadyForSpeech() --> 当前策略起作用了")
        RoomAppMMKV.mmkv.putString(
            MMKV_KEY_INITIALIZE_STRATEGY_AFFIRM, MMKV_KEY_INITIALIZE_STRATEGY_ONE_AFFIRM
        )
    }

    /**
     * 获取对象 SpeechRecognizer
     */
    fun getSpeechRecognizer(context: Context?): SpeechRecognizer? {
        // 确定了支持第一种策略，直接返回
        if (TextUtils.isEmpty(
                RoomAppMMKV.mmkv.getString(
                    MMKV_KEY_INITIALIZE_STRATEGY_ONE_AFFIRM, ""
                )
            ).not()
        ) {
            SRLogger.logD("${getClassTag()} --> getSpeechRecognizer() --> 确定了支持第一种策略，直接返回")
            return getSpeechRecognizerOneParameterStrategy(context)
        }

        // 这个手机是已知有问题的手机，所以使用第二种策略
        if (TextUtils.equals("TECNO AC8", Build.MODEL)) {
            return getSpeechRecognizerTwoParameterStrategy(context)
        }

        val strategy = RoomAppMMKV.mmkv.getString(
            MMKV_KEY_INITIALIZE_STRATEGY, MMKV_KEY_INITIALIZE_STRATEGY_ONE
        )
        return when (strategy) {
            MMKV_KEY_INITIALIZE_STRATEGY_ONE -> getSpeechRecognizerOneParameterStrategy(context)
            MMKV_KEY_INITIALIZE_STRATEGY_TWO -> getSpeechRecognizerTwoParameterStrategy(context)
            else -> {
                getSpeechRecognizerOneParameterStrategy(context)
            }
        }
    }

}