package com.transsion.search.speech

import android.content.ComponentName
import android.content.DialogInterface
import android.content.Intent
import android.content.pm.ResolveInfo
import android.graphics.Color
import android.os.Build
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.speech.RecognitionListener
import android.speech.RecognitionService
import android.speech.RecognizerIntent
import android.speech.SpeechRecognizer
import android.text.TextUtils
import android.view.Gravity
import android.view.View
import android.view.WindowManager
import androidx.lifecycle.lifecycleScope
import com.blankj.utilcode.util.Utils
import com.transsion.baselib.report.LogViewConfig
import com.transsion.baseui.dialog.BaseDialog
import com.transsion.search.R
import com.transsion.search.databinding.DialogSpeechRecognizerLayoutBinding
import com.transsion.search.speech.SpeechRecognizerVolumeCircleView.MicState
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.util.Locale


/**
 * @author: zhangxinbing
 * @date : 2024/10/11 19:41
 * @description: 语音识别弹窗
 * https://transsioner.feishu.cn/docx/G38LdH9UwoSSAvx7uRjcG5InnAg
 */
class SpeechRecognizerDialog : BaseDialog(R.layout.dialog_speech_recognizer_layout) {

    private var initializeStrategy: InitializeStrategy? = null
    private var mPartialResults: String? = ""
    private var bind: DialogSpeechRecognizerLayoutBinding? = null
    private var mCallback: ((result: String?) -> Unit?)? = null

    /**
     * 语音识别对象
     */
    private var mSpeechRecognizer: SpeechRecognizer? = null

    /**
     * 标记当前是否启动了 语音识别
     */
    private var isWorking = false

    private val mHandler = Handler(Looper.getMainLooper())


    // =============================================================================================


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setStyle(STYLE_NORMAL, com.transsion.baseui.R.style.BottomDialogTheme)
        SRLogger.logD("${getClassTag()} --> onCreate() --> 展示弹窗组件")
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        dialog?.setCancelable(false)
        dialog?.window?.let {
            it.setGravity(Gravity.BOTTOM)
            it.setDimAmount(0.5f)
            it.setBackgroundDrawable(null)
            it.setWindowAnimations(com.transsion.baseui.R.style.BaseBottomDialogAnimation)
            it.setLayout(
                WindowManager.LayoutParams.MATCH_PARENT, WindowManager.LayoutParams.WRAP_CONTENT
            )

            val lp = it.attributes ?: WindowManager.LayoutParams()
            lp.height = getPeekHeight()
            //lp.height = SizeUtils.dp2px(300f)
        }

        bind = DialogSpeechRecognizerLayoutBinding.bind(view)
        initView()
        setListener()

        // 初始化语音策略
        initializeStrategy = InitializeStrategy()
        initializeStrategy?.setCallback {
            isWorking = false
            SRLogger.logD("${getClassTag()} --> onViewCreated() --> setCallback{} --> 重新获取语音识别对象 --> initSpeechRecognizer()")
            // 重新获取
            initSpeechRecognizer()
        }

        // 初始化
        initSpeechRecognizer()

        //
        SRLogger.logW("${getClassTag()} --> onViewCreated() --> 语音识别是否可用 = ${SRUtil.isRecognitionAvailable()}")
    }

    override fun onPause() {
        super.onPause()
        mSpeechRecognizer?.cancel()
    }

    override fun onDismiss(dialog: DialogInterface) {
        super.onDismiss(dialog)
        // 页面销毁的时候报错
        // 这个错误通常是由于在未注册服务的情况下尝试解绑服务引起的。
        // https://console.firebase.google.com/u/0/project/oneroom-77885/crashlytics/app/android:com.community.mbox.affr/issues/f40fb9d0f95dc484ed42038bbaf6493a?time=last-seven-days&types=crash&versions=2.0.81.1202.02%20(50020021)&sessionEventKey=6757971101D7000164BCFE45DB2FF75C_2025042098096306150
        // 使用try-catch块: 尝试在unbindService()调用周围使用try-catch块来捕获可能的异常，以防止应用崩溃。
        kotlin.runCatching {
            mSpeechRecognizer?.destroy()
        }

        mCallback?.invoke(null)
        mCallback = null
        SRLogger.logD("${getClassTag()} --> onDismiss() --> 语音识别弹窗关闭 释放资源")
    }

    override val logViewConfig: LogViewConfig = LogViewConfig("sr_dialog")


    // =============================================================================================

    /**
     * 设置语音超时时间
     */
    private fun setSpeechTimeout(timeout: Long = 5000) {
        resetTimeOut()
        mHandler.postDelayed({
            mSpeechRecognizer?.stopListening()
            SRLogger.logD("${getClassTag()} --> setSpeechTimeout() --> 语音识别超时")
        }, timeout)
    }

    private fun resetTimeOut() {
        mHandler.removeCallbacksAndMessages(null)
    }

    private fun initView() {
        showResult(result = Utils.getApp().resources.getString(R.string.search_sr_speak_now), isComplete = false)
        bind?.volumeCircleView?.setMicState(MicState.ACTIVE)
        bind?.tvTip?.text = Utils.getApp().resources.getString(R.string.search_sr_tap_try_tip_avatar)
        bind?.volumeCircleView?.setActiveCircleColor(Utils.getApp().resources.getColor(com.transsion.wrapperad.R.color.white_40))
        bind?.volumeCircleView?.setLoadingGradientColors(
            intArrayOf(
                Color.parseColor("#2166E5"), Color.parseColor("#1DD171")
            )
        )
    }

    private fun setListener() {
        bind?.ivClose?.setOnClickListener {
            SrReport.clickCancelBtn()
            dismissAllowingStateLoss()
        }

        bind?.volumeCircleView?.setOnClickListener {
            if (isWorking) {
                mSpeechRecognizer?.stopListening()
                resetTimeOut()
            } else {
                val intent = Intent(RecognizerIntent.ACTION_RECOGNIZE_SPEECH)
                intent.putExtra(
                    RecognizerIntent.EXTRA_LANGUAGE_MODEL, RecognizerIntent.LANGUAGE_MODEL_FREE_FORM
                )
                intent.putExtra(
                    RecognizerIntent.EXTRA_LANGUAGE, Locale.getDefault().toLanguageTag()
                )
                intent.putExtra(RecognizerIntent.EXTRA_PARTIAL_RESULTS, true)

                runCatching {
                    // 开启语音识别
                    mSpeechRecognizer?.startListening(intent)
                }
                // 设置超时
                setSpeechTimeout()
                // 展示UI
                initView()
            }
            isWorking = isWorking.not()
        }
    }

    private fun showResult(result: String, isComplete: Boolean = true) {
        if (isComplete) {
            isWorking = false
        }
        if (TextUtils.isEmpty(result)) {
            return
        }
        bind?.tvResult?.text = result
    }

    private fun onResult(resultStr: String?) {
        SrReport.srSuccess()

        // 停留 500ms 自动搜索结果
        lifecycleScope.launch {
            bind?.volumeCircleView?.setMicState(MicState.LOADING)
            bind?.tvTip?.text = ""
            delay(1500)
            mCallback?.invoke(resultStr)
            mCallback = null
            dismissAllowingStateLoss()
        }
    }

    /**
     * 初始化 SpeechRecognizer
     */
    private fun initSpeechRecognizer() {

        // 回收之前的
        kotlin.runCatching {
            mSpeechRecognizer?.cancel()
            mSpeechRecognizer?.destroy()
        }

        mSpeechRecognizer = initializeStrategy?.getSpeechRecognizer(context)

        mSpeechRecognizer?.setRecognitionListener(object : RecognitionListener {
            override fun onReadyForSpeech(params: Bundle?) {
                SRLogger.logD("${getClassTag()} --> onReadyForSpeech() --> 当语音识别准备好时调用 --> params = $params")
                SrReport.srStart()
                initializeStrategy?.onReadyForSpeech()
            }

            override fun onBeginningOfSpeech() {
                SRLogger.logD("${getClassTag()} --> onBeginningOfSpeech() --> 当开始说话时调用")
            }

            override fun onRmsChanged(rmsdB: Float) {
                // 刷新UI
                //bind?.volumeCircleView?.setVolumeLevel(rmsdB)
                //SRLogger.logD("${getClassTag()} --> onRmsChanged() --> 当麦克风音量变化时调用 --> rmsdB = $rmsdB")
            }

            override fun onBufferReceived(buffer: ByteArray?) {
                SRLogger.logD("${getClassTag()} --> onBufferReceived() --> 当接收到音频数据时调用")
            }

            override fun onEndOfSpeech() {
                SRLogger.logD("${getClassTag()} --> onEndOfSpeech() --> 当说话结束时调用")
            }

            override fun onError(error: Int) {
                // 如果已经识别到部分结果了，那就直接返回
                if (TextUtils.isEmpty(mPartialResults).not()) {
                    onResult(mPartialResults)
                    return
                }

                resetTimeOut()
                val errorMsg = SRErrorUtil.getErrorMsg(error)
                SRLogger.logD("${getClassTag()} --> onError() --> 当发生错误时调用 --> error = $error -- msg = $errorMsg")
                showResult(errorMsg, isComplete = true)
                if (isWorking.not()) {
                    SrReport.srFail(errorMsg, error)
                } else {
                    SrReport.srFail(
                        SRErrorUtil.getErrorMsg(SRErrorUtil.ERROR_CANCEL), SRErrorUtil.ERROR_CANCEL
                    )
                }
                bind?.volumeCircleView?.setMicState(MicState.NORMAL)
                bind?.tvTip?.text = Utils.getApp().resources.getString(R.string.search_sr_tap_microphone_try_again)
            }

            override fun onResults(results: Bundle?) {
                results?.getStringArrayList(SpeechRecognizer.RESULTS_RECOGNITION)?.let {
                    if (it.isNotEmpty()) {
                        val resultStr = it[0]
                        showResult(resultStr)
                        SRLogger.logD("${getClassTag()} --> onResults() --> 当获得语音识别结果时调用 = $resultStr")
                        // 处理结果
                        onResult(resultStr)
                    }
                }
            }

            override fun onPartialResults(partialResults: Bundle?) {
                // 将 获得部分语音识别结果 展示
                partialResults?.getStringArrayList(SpeechRecognizer.RESULTS_RECOGNITION)?.let {
                    //SRLogger.logW("str = ${it[0]}")
                    if (it.isNotEmpty()) {
                        mPartialResults = it[0]
                        showResult(it[0])
                    }

                    setSpeechTimeout()
                }
            }

            override fun onEvent(eventType: Int, params: Bundle?) {
                SRLogger.logD("${getClassTag()} --> onEvent() --> 当发生其他事件时调用 --> eventType = $eventType --> params = $params")
            }
        })

        // 默认开启识别
        bind?.volumeCircleView?.let {
            it.post {
                initializeStrategy?.startListening()
                it.performClick()
            }
        }
    }

    /**
     * 弹窗高度，默认为屏幕高度的四分之三
     * 子类可重写该方法返回peekHeight
     *
     * @return height
     */
    private fun getPeekHeight(): Int {
        val peekHeight: Int = Utils.getApp().resources.displayMetrics.heightPixels
        //设置弹窗高度为屏幕高度的2/3
        return peekHeight - peekHeight / 3
    }

    fun analysisBundle(bundle: Bundle?): String {
        return if (bundle != null) {
            val stringBuilder = StringBuilder()
            for (key in bundle.keySet()) {
                val value = bundle.get(key)
                stringBuilder.append("$key: ${value?.toString() ?: "null"}\n")
                SRLogger.logW("key = $key -- value = $value")
            }
            stringBuilder.toString()
        } else {
            "Bundle is null"
        }
    }

    // =============================================================================================


    /**
     * 将结果导出
     */
    fun setCallback(callback: ((result: String?) -> Unit?)? = null): SpeechRecognizerDialog {
        mCallback = callback
        return this
    }

}