package com.transsion.search.speech

import android.animation.ValueAnimator
import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.util.AttributeSet
import android.view.View
import android.view.animation.AccelerateDecelerateInterpolator
import com.transsion.search.R
import android.util.TypedValue
import android.graphics.*

/**
 * @author: zhangxinbing
 * @date : 2024/10/11 14:28
 * @description: 圆形语音输入View
 */
class SpeechRecognizerVolumeCircleView @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null
) : View(context, attrs) {

    private var mMicState: MicState = MicState.NORMAL // 当前View有三种状态
    private var mActiveAnnulusRadius: Float = 0f // active 状态下 圆的半径
    private var mLoadingAngle: Float = 0f // loading 状态下 圆圈的旋转角度
    private var mActiveAnimationDuration: Long = 1000 // 默认1秒
    private var mMicIconSize: Int = dpToPx(80) // 默认80dp
    private var mScaledMicIconNormal: Bitmap // 自定义的麦克风图标
    private var mScaledMicIconActive: Bitmap // 自定义的麦克风图标

    private var mLoadingGradientColors: IntArray =
        intArrayOf(Color.parseColor("#2166E5"), Color.parseColor("#1DD171")) // Loading 默认的渐变颜色

    private val mMicIconNormal: Bitmap =
        BitmapFactory.decodeResource(context.resources, R.mipmap.sr_icon_normal) // 默认状态下的图标
    private val mMicIconActive: Bitmap =
        BitmapFactory.decodeResource(context.resources, R.mipmap.sr_icon_active) // 语音识别状态下的图标

    /**
     * active 状态下的画笔
     */
    private val mActivePaint = Paint().apply {
        color = Color.RED // 默认颜色为红色
        style = Paint.Style.STROKE
        strokeWidth = context.resources.displayMetrics.density // 默认1dp
        isAntiAlias = true
    }

    /**
     * loading 状态下的画笔
     */
    private val mLoadingPaint = Paint().apply {
        style = Paint.Style.STROKE
        strokeWidth = dpToPx(2).toFloat() // 默认2dp
        isAntiAlias = true
    }

    /**
     * active状态下的圆动画
     */
    private val mActiveRadiusAnimator = ValueAnimator().apply {
        interpolator = AccelerateDecelerateInterpolator()
        addUpdateListener { animation ->
            mActiveAnnulusRadius = animation.animatedValue as Float
            //SRLogger.logW("SpeechRecognizerVolumeCircleView --> mActiveAnnulusRadius = ${animation.animatedValue}")
            // 计算透明度的渐变
            val fraction = animation.animatedFraction
            mActivePaint.alpha = when {
                fraction <= 0.5 -> (255 * (1 - 0.5 * fraction)).toInt() // 从40%到20%
                else -> (255 * (0.5 - 0.5 * (fraction - 0.5))).toInt() // 从20%到0%
            }
            invalidate()
        }
    }


    // =============================================================================================


    init {
        mScaledMicIconNormal =
            Bitmap.createScaledBitmap(mMicIconNormal, mMicIconSize, mMicIconSize, true)
        mScaledMicIconActive =
            Bitmap.createScaledBitmap(mMicIconActive, mMicIconSize, mMicIconSize, true)
        setLoadingGradientColors(mLoadingGradientColors)
    }


    enum class MicState {
        NORMAL, ACTIVE, LOADING
    }

    /**
     * 单位转换
     */
    private fun dpToPx(dp: Int): Int {
        return TypedValue.applyDimension(
            TypedValue.COMPLEX_UNIT_DIP, dp.toFloat(), context.resources.displayMetrics
        ).toInt()
    }

    /**
     * 开启 Active状态下 圆环的动画
     */
    private fun startActiveRadiusAnimation() {
        val iconRadius = (mMicIconSize / 2).toFloat()
        val maxRadius = (minOf(width, height) / 2).toFloat() - mActivePaint.strokeWidth / 2
        mActiveRadiusAnimator.setFloatValues(iconRadius, maxRadius)
        mActiveRadiusAnimator.duration = mActiveAnimationDuration
        mActiveRadiusAnimator.repeatCount = ValueAnimator.INFINITE
        mActiveRadiusAnimator.start()
    }


    // =============================================================================================


    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)

        // 根据不同的状态绘制不同的内容
        when (mMicState) {
            MicState.LOADING -> {
                val cx = (width / 2).toFloat()
                val cy = (height / 2).toFloat()
                val loadingRadius = (mMicIconSize / 2).toFloat()

                // 绘制四分之三的圆形进度条
                canvas.drawArc(
                    cx - loadingRadius,
                    cy - loadingRadius,
                    cx + loadingRadius,
                    cy + loadingRadius,
                    mLoadingAngle,
                    270f, // 四分之三的圆
                    false,
                    mLoadingPaint
                )

                // 更新进度条的角度 并刷新视图
                mLoadingAngle += 5f
                if (mLoadingAngle > 360f) mLoadingAngle = 0f
                postInvalidateOnAnimation()
            }

            MicState.ACTIVE -> {
                val micIcon = mScaledMicIconActive

                // 计算图标的中心点
                val iconLeft = (width - mMicIconSize) / 2
                val iconTop = (height - mMicIconSize) / 2
                val cx = (iconLeft + mMicIconSize / 2).toFloat()
                val cy = (iconTop + mMicIconSize / 2).toFloat()

                // 绘制圆
                canvas.drawCircle(cx, cy, mActiveAnnulusRadius, mActivePaint)

                // 绘制麦克风图标
                canvas.drawBitmap(micIcon, iconLeft.toFloat(), iconTop.toFloat(), null)
            }

            else -> {
                val micIcon = mScaledMicIconNormal

                // 计算图标的中心点
                val iconLeft = (width - mMicIconSize) / 2
                val iconTop = (height - mMicIconSize) / 2

                // 绘制麦克风图标
                canvas.drawBitmap(micIcon, iconLeft.toFloat(), iconTop.toFloat(), null)
            }
        }
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        mActiveRadiusAnimator.cancel()
    }


    // =============================================================================================


    /**
     * 设置当前的状态
     */
    fun setMicState(state: MicState) {
        mMicState = state
        if (mMicState == MicState.ACTIVE) {
            startActiveRadiusAnimation()
        } else {
            mActiveRadiusAnimator.cancel()
        }
        invalidate() // 刷新视图
    }

    /**
     * 设置active状态下圆的颜色
     */
    fun setActiveCircleColor(color: Int) {
        mActivePaint.color = color
        invalidate()
    }

    /**
     * 设置active状态下圆的宽度
     */
    fun setActiveStrokeWidth(widthDp: Int) {
        mActivePaint.strokeWidth = dpToPx(widthDp).toFloat()
        invalidate()
    }

    /**
     * 设置动画时间
     */
    fun setActiveAnimationDuration(duration: Long) {
        mActiveAnimationDuration = duration
    }

    /**
     * 设置麦克风图标的大小
     */
    fun setMicIconSize(size: Int) {
        mMicIconSize = dpToPx(size)
        mScaledMicIconNormal =
            Bitmap.createScaledBitmap(mMicIconNormal, mMicIconSize, mMicIconSize, true)
        mScaledMicIconActive =
            Bitmap.createScaledBitmap(mMicIconActive, mMicIconSize, mMicIconSize, true)
        invalidate()
    }

    /**
     * 设置loading状态下的渐变色
     */
    fun setLoadingGradientColors(colors: IntArray) {
        mLoadingGradientColors = colors
        val cx = (width / 2).toFloat()
        val cy = (height / 2).toFloat()

        // 创建带有起始和结束透明颜色的渐变
        mLoadingPaint.shader = SweepGradient(
            cx, cy, intArrayOf(
                colors[0], colors[1], colors[1], colors[0]
            ), floatArrayOf(0f, 0.5f, 0.75f, 1f)
        )
        invalidate()
    }

    /**
     * 设置loading状态下的圆宽度
     */
    fun setLoadingStrokeWidth(widthDp: Int) {
        mLoadingPaint.strokeWidth = dpToPx(widthDp).toFloat()
        invalidate()
    }
}
