package com.transsion.search.widget

import android.content.Context
import android.os.Build
import android.text.Spannable
import android.text.SpannableStringBuilder
import android.text.style.DynamicDrawableSpan
import android.text.style.ImageSpan
import android.util.AttributeSet
import android.view.ViewGroup
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.recyclerview.widget.GridLayoutManager
import com.alibaba.android.arouter.launcher.ARouter
import com.blankj.utilcode.util.RegexUtils
import com.blankj.utilcode.util.SizeUtils
import com.blankj.utilcode.util.TimeUtils
import com.blankj.utilcode.util.Utils
import com.chad.library.adapter.base.util.getItemView
import com.tn.lib.decoration.GridItemDecoration
import com.tn.lib.ext.invisible
import com.tn.lib.ext.visible
import com.transsion.baselib.utils.PlayModeUtil
import com.transsion.baseui.image.ImageHelper
import com.transsion.baseui.util.formatSecondsToHHMM
import com.transsion.moviedetailapi.MovieDetailConstants
import com.transsion.moviedetailapi.MovieDetailPageIdentity
import com.transsion.moviedetailapi.SubjectType
import com.transsion.moviedetailapi.bean.Subject
import com.transsion.moviedetailapi.getSubjectIconId
import com.transsion.search.R
import com.transsion.search.adapter.SeachAccurateSeasonsAdapter
import com.transsion.search.bean.AccurateSubject
import com.transsion.search.bean.SearchSubject
import com.transsion.search.bean.Season
import com.transsion.search.databinding.ItemSearchValuesAccurateLayoutBinding
import com.transsion.search.fragment.values.SearchValuesFragment
import com.transsion.search.fragment.values.SearchValuesFragment.Companion.SEARCH_VALUES_ITEM_ACCURATE_SUBJECT
import java.util.Calendar

/**
 * Transsion MI
 * Created By Liupeng
 * On 2024/7/18 12:10
 *  此view 是超出recyclerview 宽度的view。2边间距由内部设置
 *
 *
 * Desc：相关作品
 */
class AccurateSubjectView : ConstraintLayout {

    private var seasonsAdapter: SeachAccurateSeasonsAdapter? = null

    private var mViewBinding: ItemSearchValuesAccurateLayoutBinding? = null

    constructor(context: Context) : this(context, null)

    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)

    constructor(context: Context, attrs: AttributeSet?, res: Int) : super(context, attrs, res) {
        initView()
    }

    private fun initView() {
        val container = inflate(context, R.layout.item_search_values_accurate_layout, this)
        mViewBinding = ItemSearchValuesAccurateLayoutBinding.bind(container)
        //设置支持子view超出范围
        this.clipChildren = false
    }


    private val movieDesignW = SizeUtils.dp2px(104f)        // 三列时设计宽
    private val movieDesignH = SizeUtils.dp2px(149f)        // 三列时设计高    // 三列时设计高
    private val size48 = SizeUtils.dp2px(48f)       // 3列是间隔16+8+8+16
    private val screenW = com.tn.lib.utils.ScreenUtils.getWidth(Utils.getApp())                   // movie、tv实际高l
    private val size16 = SizeUtils.dp2px(16f)
    private val showMaxSeasonsSize = 6

    fun setData(accurateSubject: AccurateSubject) {
        //处理支持 分割线 不同颜色
        val marginLayoutParams = this.layoutParams as ViewGroup.MarginLayoutParams
        marginLayoutParams.setMargins(-size16, 0, -size16, 0)
        //处理支持 分割线 不同颜色
        val item = accurateSubject.subject
        val ivCover = mViewBinding?.ivCover
        val coverObj = item?.cover
        if (coverObj != null) {
            coverObj.url?.let { url ->
                ImageHelper.loadNet(
                    context,
                    ivCover ?: return@let,
                    url,
                    thumbnail = coverObj.thumbnail ?: "",
                    thumbScaleType = false
                )
            }
            mViewBinding?.ivSearchCorner?.setTextWithType(item.corner, item.subjectType)
            mViewBinding?.tvSubject?.setText(item.title)
            mViewBinding?.tvSubjectScore?.setText(item.imdbRate)

            // 作品的简介
            mViewBinding?.tvSubjectYear?.apply {
                val span = SpannableStringBuilder()
                span.append("  ")
                // 年份
                if (item.releaseDate.isNullOrEmpty().not()) {
                    val releaseDate = TimeUtils.string2Date(item.releaseDate, "yyyy-MM-dd")
                    val calendar = Calendar.getInstance().apply { time = releaseDate }
                    span.append(calendar.get(Calendar.YEAR).toString())
                }
                // 国家
                if (item.countryName.isNullOrEmpty().not()) {
                    appendSplit(span)
                    span.append(item.countryName)
                }
                // 时长 电影时长（电视剧不展示）
                if ((item.durationSeconds ?: 0) > 0 && !item.isEPType()) {
                    appendSplit(span)
                    span.append(formatSecondsToHHMM(item.durationSeconds))
                }
                // 主演
                var maxShow = 2
                item.staffList?.forEach{
                    if (maxShow == 0) {
                        return@forEach
                    }
                    maxShow--
                    appendSplit(span)
                    span.append(it.name)
                }
                if (item.duration.isNullOrEmpty().not() && !item.isSeries()) {
                    appendSplit(span)
                    span.append(item.duration)
                }

                val subjectImageId = getSubjectIconId(item.subjectType)
                val imageSpan: ImageSpan = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    ImageSpan(context, subjectImageId, DynamicDrawableSpan.ALIGN_CENTER)
                } else {
                    ImageSpan(context, subjectImageId, DynamicDrawableSpan.ALIGN_BASELINE)
                }
                span.setSpan(imageSpan, 0, 1, Spannable.SPAN_EXCLUSIVE_INCLUSIVE)
                text = span
            }

            // 如果没有资源将隐藏下载按钮
            mViewBinding?.llDownload?.let {
                if (item.subjectType == SubjectType.SHORT_TV.value) {
                    it.visible()
                }
                if (item.hasResource == true) {
                    it.visible()
                    if (PlayModeUtil.isStreamMode()) {
                        it.setShowPlayType()
                    } else {
                        item.resourceDetectors?.let {
                            if (it.isNotEmpty()) {
                                mViewBinding?.llDownload?.apply {
                                    setShowType(
                                        subjectId = item.subjectId,
                                        resourceId = it[0].resourceId,
                                        it[0].type == 1
                                    )
                                }
                            }
                        }
                    }
                } else {
                    it.invisible()
                }
            }

        }
        val length = accurateSubject.seasons?.size ?: 0
        val seasonsList = accurateSubject.seasons?.take(6)
        // 标签
        bindTagView(mViewBinding?.tagContentLL, accurateSubject.subject)
        //分季数据
        if (seasonsList?.isNotEmpty() == true && seasonsList.size > 1) {
            mViewBinding?.rvSeasons?.visibility = VISIBLE
            mViewBinding?.rvSeasons?.apply {
                layoutManager = GridLayoutManager(context, 3)
                seasonsAdapter = SeachAccurateSeasonsAdapter().apply {

                    setOnItemClickListener { _, _, position ->
                        val season = seasonsAdapter?.getItem(position)
                        season?.let { onItemCLick(position,item, it) }
                    }

                }
                adapter = seasonsAdapter
                seasonsAdapter?.setList(seasonsList)
                seasonsAdapter?.needShowMore(length >= 7)
                // 添加分割线 -- 只有第一个需要处理 top 8dp
                addItemDecoration(
                    GridItemDecoration(
                        SizeUtils.dp2px(4f),
                        SizeUtils.dp2px(8 / 2f),
                        SizeUtils.dp2px(8f),
                        SizeUtils.dp2px(8f)
                    )
                )
            }
        } else {
            mViewBinding?.rvSeasons?.visibility = GONE
        }

        SearchValuesFragment.reportShow(
            group_id = "",
            subject_id = item?.subjectId,
            staffId = "",
            ops = item?.ops,
            0,
            SEARCH_VALUES_ITEM_ACCURATE_SUBJECT
        )


    }

    private fun appendSplit(span:SpannableStringBuilder) {
        if (span.isNotEmpty()) {
            span.append(" / ")
        }
    }

    /*
     * 绑定标签
     */
    private fun bindTagView(root: ViewGroup?, subject: Subject?) {
        root ?: return
        subject?.let {
            val tags = RegexUtils.getSplits(it.genre, ",")
            tags?.forEach { item->
                val view = getItemView(R.layout.item_search_subject_tag)
                view.findViewById<TextView>(R.id.tagTV).text = item
                root.addView(view)
            }
        }
    }

    private fun onItemCLick(position: Int, subject: SearchSubject?, item: Season) {
        var se = item.se ?: 0
        if (position == showMaxSeasonsSize - 1) {
            se = 0
        }
        //跳转进作品详情页 调起分季列表
        ARouter.getInstance().build(MovieDetailPageIdentity.MOVIE_DETAIL)
            .withInt(MovieDetailConstants.SUBJECT_TYPE, subject?.subjectType ?: SubjectType.MOVIE.value)
            .withString(MovieDetailConstants.SUBJECT_ID, subject?.subjectId)
            .withInt(MovieDetailConstants.SEASON, se)
            .withString(MovieDetailConstants.OPS, subject?.ops)
            .navigation()
        SearchValuesFragment.reportClick(
            "",
            subject?.subjectId,
            "",
            subject?.ops,
            position,
            SEARCH_VALUES_ITEM_ACCURATE_SUBJECT,
            SearchValuesFragment.trid,
            season = item.se.toString()
        )
    }
}
