package com.transsion.search.ad

import android.animation.ObjectAnimator
import android.graphics.drawable.TransitionDrawable
import android.view.Gravity
import android.widget.FrameLayout
import android.widget.ImageView
import androidx.appcompat.widget.AppCompatTextView
import androidx.core.content.ContextCompat
import androidx.core.content.res.ResourcesCompat
import androidx.lifecycle.LifecycleCoroutineScope
import com.blankj.utilcode.util.SizeUtils
import com.hisavana.common.bean.TAdErrorCode
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.bidding.base.BiddingIntermediateMaterialBean
import com.transsion.ad.bidding.nativead.BiddingNativeManager
import com.transsion.ad.log.AdLogger
import com.transsion.ad.view.native_ad.NativeWrapperAdView
import com.transsion.wrapperad.R
import com.transsion.wrapperad.SceneId
import com.transsion.wrapperad.ad_layout.BiddingAdLayoutFactory
import com.transsion.wrapperad.ad_layout.SearchHotBiddingHiNativeSspAdLayout
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * @author Connor
 * @date : 2025/9/1
 * @description: 搜索页广告，含程序化电商广告和bidding广告
 * 优先程序化电商广告，无电商广告时，加载bidding广告
 */
class SearchHotAdHelper(private val lifecycleScope: LifecycleCoroutineScope) {

    companion object {
        const val TAG = "SearchHotAdHelper"
        const val ANIM_DELAY = 3000L
        const val ANIM_DURATION = 300
    }

    private var hiNativeAd: BiddingNativeManager? = null
    private var nativeViewList: List<NativeWrapperAdView>? = null
    private var nativeInfoList: List<BiddingIntermediateMaterialBean>? = null

    private var biddingNativeAd: BiddingNativeManager? = null
    private var biddingNativeAdView: NativeWrapperAdView? = null

    private var adContainerView: SearchHotAdContainerView? = null
    private var showAdCallback: ((canShow: Boolean) -> Unit)? = null
    private var isAdLoaded = false

    private var textColorAnimator: ObjectAnimator? = null
    private var transitionRunnable: Runnable? = null

    private val adHiListener = object : BiddingTAdditionalListener() {

        override fun onBiddingLoad(maxEcpmObject: List<BiddingIntermediateMaterialBean>?) {
            super.onBiddingLoad(maxEcpmObject)

            bindHiAd(maxEcpmObject)
        }

        override fun onBiddingError(p0: TAdErrorCode?) {
            super.onBiddingError(p0)
            // 无电商广告，加载bidding广告
            loadBiddingAd()
        }
    }

    private val adBiddingListener = object : BiddingTAdditionalListener() {

        override fun onBiddingLoad(maxEcpmObject: BiddingIntermediateMaterialBean?) {
            super.onBiddingLoad(maxEcpmObject)

            bindBiddingAd(maxEcpmObject)
        }
    }

    // ================================ Hisavana SSP Native ========================================

    /**
     * 加载程序化电商广告
     */
    private fun loadHiAd() {
        destroyHiAd()
        lifecycleScope.launch(Dispatchers.IO) {
            hiNativeAd = BiddingNativeManager()
            hiNativeAd?.apply {
                setSceneId(SceneId.SCENE_SEARCH_HOT_NATIVE)
                setListener(adHiListener)
                setFetchInFull(true)
                loadAd(fetchCount = 3, onlyHi = true)
            }
        }
    }

    private fun bindHiAd(maxEcpmObject: List<BiddingIntermediateMaterialBean>?) {
        val container = adContainerView ?: return
        val views = container.getHiNativeAdViews().also { nativeViewList = it }
        val infos = maxEcpmObject?.also { nativeInfoList = it }

        if (views.isNullOrEmpty() || infos.isNullOrEmpty()) {
            return
        }

        val adManager = hiNativeAd ?: return
        val itemCount = minOf(views.size, infos.size)

        for (index in 0 until itemCount) {
            val sspProvider = SearchHotBiddingHiNativeSspAdLayout(container.context)
            views[index].bindNativeView(adManager, sspLayoutProvider = sspProvider, maxEcpmObject = infos[index])
        }

        showAd()
    }

    private fun destroyHiAd() {
        nativeViewList?.forEach {
            it.removeAllViews()
        }
        nativeViewList = null

        hiNativeAd?.destroy()
        hiNativeAd = null
    }

    // ================================ Bidding Native ========================================

    /**
     * 无程序化电商广告时，加载Bidding Native广告
     */
    private fun loadBiddingAd() {
        destroyBiddingAd()
        lifecycleScope.launch(Dispatchers.IO) {
            biddingNativeAd = BiddingNativeManager()
            biddingNativeAd?.apply {
                setSceneId(SceneId.SCENE_SEARCH_HOT_NATIVE)
                setListener(adBiddingListener)
                loadAd()
            }
        }
    }

    private fun bindBiddingAd(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        adContainerView?.getNativeWrapperAdView()?.let {
            biddingNativeAdView = it
            val nativeViewBinder = BiddingAdLayoutFactory.getAdLayout(
                it.context, SceneId.SCENE_SEARCH_HOT_NATIVE)
            it.bindNativeView(biddingNativeAd, adLayoutProvider = nativeViewBinder, maxEcpmObject = maxEcpmObject)

            showAd()
            setElementLayout()
        }
    }

    private fun setElementLayout() {
        // 声音图标
        setVolumeImageLayout()

        // 按钮动画
        actionBtnAnimator()
    }

    private fun setVolumeImageLayout() {
        val adVolumeImage = biddingNativeAdView?.findViewWithTag<ImageView>("adVolumeImage")
        adVolumeImage?.let {
            (it.layoutParams as FrameLayout.LayoutParams).apply {
                gravity = Gravity.START or Gravity.BOTTOM
                val margin = SizeUtils.dp2px(8.0f)
                marginStart = margin
                bottomMargin = margin
                val size = SizeUtils.dp2px(26.0f)
                width = size
                height = size
            }
        }
    }

    private fun actionBtnAnimator() {
        transitionRunnable = Runnable {
            val actionView = biddingNativeAdView?.findViewById<AppCompatTextView>(R.id.native_ad_action)
            actionView?.let {
                val targetTextColor =
                    ContextCompat.getColor(it.context, com.transsion.ad.R.color.gray_dark_00)
                val textColorAnimator = ObjectAnimator.ofArgb(
                    it,
                    "textColor",
                    it.currentTextColor,
                    targetTextColor
                ).setDuration(ANIM_DURATION.toLong())

                // 2. 背景过渡动画, 交叉淡入
                val originalBg = it.background ?: ResourcesCompat.getDrawable(
                    it.context.resources,
                    android.R.color.transparent,
                    it.context.theme
                )
                val targetBg = ResourcesCompat.getDrawable(
                    it.context.resources,
                    R.drawable.bg_gradient_brand_4,
                    it.context.theme
                )

                val transitionBg = TransitionDrawable(arrayOf(originalBg, targetBg))
                it.background = transitionBg

                // 3. 同步启动背景和文字动画
                transitionBg.startTransition(ANIM_DURATION)
                textColorAnimator.start()
            }
        }

        biddingNativeAdView?.postDelayed(transitionRunnable, ANIM_DELAY)
    }

    private fun destroyBiddingAd() {
        biddingNativeAd?.destroy()
        biddingNativeAd = null

        destroyAnimation()
    }

    private fun destroyAnimation() {
        textColorAnimator?.cancel()
        textColorAnimator = null

        transitionRunnable?.let {
            biddingNativeAdView?.removeCallbacks(it)
        }
        transitionRunnable = null
    }

    // ================================= API =========================================

    private fun showAd() {
        AdLogger.logSdk(TAG, "-----> showAd")
        isAdLoaded = true
        showAdCallback?.invoke(true)
    }

    fun bindAdData(view: SearchHotAdContainerView?) {
        AdLogger.logSdk(TAG, "-----> bindAdData")
        adContainerView = view

        loadHiAd()
    }

    fun showAdContainer(callback: (canShow: Boolean) -> Unit) {
        showAdCallback = callback
        if (isAdLoaded) {
            AdLogger.logSdk(TAG, "-----> showAdContainer true")
            callback.invoke(true)
        }
    }

    fun destroy() {
        AdLogger.logSdk(TAG, "-----> destroy")
        destroyHiAd()
        destroyBiddingAd()
        showAdCallback = null
        adContainerView?.destroy()
        adContainerView = null
    }
}