package com.transsion.search.adapter.provider

import android.content.Context
import android.os.Build
import android.text.Spannable
import android.text.SpannableStringBuilder
import android.text.style.DynamicDrawableSpan
import android.text.style.ImageSpan
import android.widget.ImageView
import androidx.appcompat.widget.AppCompatTextView
import androidx.constraintlayout.widget.ConstraintLayout
import com.blankj.utilcode.util.RegexUtils
import com.blankj.utilcode.util.SizeUtils
import com.blankj.utilcode.util.TimeUtils
import com.chad.library.adapter.base.provider.BaseItemProvider
import com.chad.library.adapter.base.viewholder.BaseViewHolder
import com.tn.lib.utils.ScreenUtils
import com.tn.lib.view.CornerTextView
import com.transsion.baseui.image.ImageHelper
import com.transsion.moviedetailapi.getSubjectIconId
import com.transsion.search.R
import com.transsion.search.adapter.SearchWorkAdapter
import com.transsion.search.bean.SearchSubject
import java.util.Calendar

class SearchWorkProvider(context: Context) : BaseItemProvider<SearchSubject>() {
    // 动态计算 图片的宽高
    private val movieDesignW = SizeUtils.dp2px(104f)        // 三列时设计宽
    private val movieDesignH = SizeUtils.dp2px(149f)        // 三列时设计高
    private val size48 = SizeUtils.dp2px(48f)       // 3列是间隔16+8+8+16
    private val screenW = ScreenUtils.getWidth(context)
    private val movieW = (screenW - size48) / 3       // movie、tv实际宽
    private val movieH = (movieW * (movieDesignH * 1.0f / movieDesignW)).toInt()                    // movie、tv实际高

    override val layoutId: Int
        get() = R.layout.item_works_layout_vertical

    override val itemViewType: Int
        get() = SearchWorkAdapter.SEARCH_WORK_TYPE

    override fun convert(helper: BaseViewHolder, item: SearchSubject) {
        val context = helper.itemView.context
        val ivAvatar = helper.itemView.findViewById<ImageView>(R.id.ivCover)
        val layoutParams = ivAvatar.layoutParams as ConstraintLayout.LayoutParams
        layoutParams.height = movieH
        layoutParams.width = movieW

        // 获取图片宽高比
        item.cover?.apply {
            url?.let {
                ImageHelper.loadNet(
                    context,
                    ivAvatar,
                    it,
                    width = movieW,
                    height = movieH,
                    thumbnail = item.cover?.thumbnail ?: "",
                    thumbScaleType = false
                )
            }
        }

        helper.setText(R.id.tvSubject, item.title)

        // 作品的简介
        helper.getViewOrNull<AppCompatTextView>(R.id.tvSubjectYear)?.apply {
            val span = SpannableStringBuilder()
            span.append("  ")
            if (item.releaseDate.isNullOrEmpty().not()) {
                val releaseDate = TimeUtils.string2Date(item.releaseDate, "yyyy-MM-dd")
                val calendar = Calendar.getInstance().apply { time = releaseDate }
                span.append(calendar.get(Calendar.YEAR).toString())
            }
            val strings = RegexUtils.getSplits(item.genre, ",")
            strings?.forEach {
                span.append(" / ").append(it)
            }
            if (item.countryName.isNullOrEmpty().not()) {
                span.append(" / ").append(item.countryName)
            }

            val subjectImageId = getSubjectIconId(item.subjectType)
            val imageSpan: ImageSpan = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                ImageSpan(context, subjectImageId, DynamicDrawableSpan.ALIGN_CENTER)
            } else {
                ImageSpan(context, subjectImageId, DynamicDrawableSpan.ALIGN_BASELINE)
            }
            span.setSpan(imageSpan, 0, 1, Spannable.SPAN_EXCLUSIVE_INCLUSIVE)
            text = span
        }

        //季数展示
        val seNum = item.seNum ?: 0
        helper.setGone(R.id.tvSeasons,seNum < 2)
        helper.setText(R.id.tvSeasons,context.getString(R.string.seasons, seNum.toString()))

        //角标
        val cornerTextView = helper.getView<CornerTextView>(R.id.ivSearchCorner)
        cornerTextView.setTextWithType(item.corner, item.subjectType)
    }

}