package com.transsion.search.fragment.hot

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.os.Bundle
import android.text.TextUtils
import android.view.Gravity
import android.view.LayoutInflater
import android.view.View
import android.view.animation.AccelerateInterpolator
import android.view.animation.DecelerateInterpolator
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.content.ContextCompat
import androidx.core.os.bundleOf
import androidx.core.widget.NestedScrollView
import androidx.fragment.app.Fragment
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.lifecycleScope
import androidx.viewpager2.adapter.FragmentStateAdapter
import androidx.viewpager2.widget.ViewPager2
import com.blankj.utilcode.util.KeyboardUtils
import com.blankj.utilcode.util.SizeUtils
import com.google.android.flexbox.FlexDirection
import com.google.android.flexbox.FlexWrap
import com.google.android.flexbox.JustifyContent
import com.tn.lib.ext.gone
import com.tn.lib.ext.invisible
import com.tn.lib.ext.visible
import com.tn.lib.logger.Logger
import com.tn.lib.view.GradientLinePagerIndicator
import com.tn.lib.widget.dialog.TRDialog
import com.tn.lib.widget.dialog.TRDialogListener
import com.transsion.baselib.report.LogViewConfig
import com.transsion.baseui.ext.dp
import com.transsion.baseui.util.MultipleClickListener
import com.transsion.baseui.widget.CustomPagerTitleView
import com.transsion.mb.config.manager.ConfigManager
import com.transsion.publish.view.SoftInputUtil.hideSoftInput
import com.transsion.search.ISearchCallback
import com.transsion.search.R
import com.transsion.search.SearchManager
import com.transsion.search.ad.SearchHotAdHelper
import com.transsion.search.bean.HotSearchKeyWord
import com.transsion.search.bean.HotSubjectEntity
import com.transsion.search.constant.SearchConstants
import com.transsion.search.constant.SearchConstants.Companion.SEARCH_MOVIE
import com.transsion.search.databinding.FragmentHotSearchBinding
import com.transsion.search.fragment.BaseSearchMainFragment
import com.transsion.search.fragment.SearchSubjectFragment
import com.transsion.search.fragment.hot.adapter.HistoryVal
import com.transsion.search.fragment.hot.adapter.SearchHistoryAdapter
import com.transsion.search.fragment.hot.adapter.SearchHistoryAdapter.Companion.TYPE_HISTORY_VAL
import com.transsion.search.fragment.hot.adapter.SearchHotKeywordAdapter
import com.transsion.search.viewmodel.SearchViewModel
import com.transsion.search.widget.LinesFlexBoxLayoutManager
import com.transsion.search.widget.SearchReportHelper
import net.lucode.hackware.magicindicator.buildins.commonnavigator.CommonNavigator
import net.lucode.hackware.magicindicator.buildins.commonnavigator.abs.CommonNavigatorAdapter
import net.lucode.hackware.magicindicator.buildins.commonnavigator.abs.IPagerIndicator
import net.lucode.hackware.magicindicator.buildins.commonnavigator.abs.IPagerTitleView
import net.lucode.hackware.magicindicator.buildins.commonnavigator.indicators.LinePagerIndicator

class SearchHotFragment : BaseSearchMainFragment<FragmentHotSearchBinding>(), TRDialogListener {
    companion object {
        fun newInstance(hotSearchWord: String?): SearchHotFragment {
            val fragment = SearchHotFragment()
            val bundle = bundleOf(SearchConstants.HOT_SEARCH_WORD to hotSearchWord)
            fragment.arguments = bundle
            return fragment
        }

        const val PAGE_NAME = "search_hot_subject"
    }

    //历史词
    private var mSearchHistoryAdapter: SearchHistoryAdapter? = null
    private var expand: Boolean = false
    private var historyExpandLine = 2
    private var historyCollapseLine = 1

    //private var showMore: Boolean = true
    private var hotSearchWord: String? = null
    private var mHistoryList = mutableListOf<String>()

    //热词
    private var everyOneSearchAdapter: SearchHotKeywordAdapter? = null
    private var mHotSubjectEntity: HotSubjectEntity? = null

    private var mSearchViewModel: SearchViewModel? = null

    private var adHelper: SearchHotAdHelper? = null

    override fun getViewBinding(inflater: LayoutInflater): FragmentHotSearchBinding {
        return FragmentHotSearchBinding.inflate(inflater)
    }

    override fun initView(view: View, savedInstanceState: Bundle?) {
        initViewModel()
        initHistory()
        initEveryoneSearch()
        initViewPager()
        adHelper = SearchHotAdHelper(lifecycleScope)
        mViewBinding?.adContainerView?.let { containerView ->
            adHelper?.bindAdData(containerView)
        }

        mViewBinding?.searchHotRootScroll?.setOnScrollChangeListener(NestedScrollView.OnScrollChangeListener { _, _, _, _, _ ->
            activity?.apply {
                if (KeyboardUtils.isSoftInputVisible(this)) {
                    hideSoftInput(mViewBinding?.searchHotRootScroll)
                }
            }
        })
    }

    override fun onDestroyView() {
        super.onDestroyView()
        mViewBinding?.rvHistory?.adapter = null
        mSearchHistoryAdapter = null
        adHelper?.destroy()
        adHelper = null
    }

    override fun lazyLoadData() {
        mSearchViewModel?.getHotSubject()
        mViewBinding?.progressBar?.visibility = View.VISIBLE
        mViewBinding?.searchHotRankIndicatorLinear?.visibility = View.INVISIBLE
    }


    override fun newLogViewConfig(): LogViewConfig {
        return LogViewConfig(pageName = PAGE_NAME).apply {
            needPv = true
            paramsMap["search_id"] = sSearchId
        }
    }

    fun onConnected() {
        if (mHotSubjectEntity == null) {
            lazyLoadData()
        }
    }

    fun addHistoryWord(keyword: String) {
        if (!mHistoryList.contains(keyword)) {
            SearchManager.INSTANCE.addSearch(keyword)
            mHistoryList.add(0, keyword)
            if (mHistoryList.size > SearchManager.MAX_HISTORY_COUNT) {
                mHistoryList.removeAt(mHistoryList.size - 1)
            }
        }
    }

    private fun initViewModel() {
        mSearchViewModel = ViewModelProvider(requireActivity())[SearchViewModel::class.java].apply {
            hotSubjectLiveData.observe(this@SearchHotFragment) { 
                mViewBinding?.progressBar?.visibility = View.GONE
                mViewBinding?.searchHotRankIndicatorLinear?.visibility = View.VISIBLE
                adHelper?.showAdContainer { canShow ->
                    if (canShow) {
                            mViewBinding?.nativeAdContainer?.visible()
                        }
                    }
                logViewConfig?.isLoadSuccess = true
                initHotWordWithData(it)
                initRankWithData(it)
            }
        }
    }

    private fun initHistory() {
        mViewBinding?.tvClear?.setOnClickListener {
            clearTipsDialog()
            if (KeyboardUtils.isSoftInputVisible(it.context as Activity)) KeyboardUtils.hideSoftInput(
                it.context as Activity
            )
        }
        mSearchHistoryAdapter = SearchHistoryAdapter()
        mSearchHistoryAdapter?.addChildClickViewIds(R.id.search_history_text)
        mSearchHistoryAdapter?.setOnItemChildClickListener { _, _, position ->
            val parent = parentFragment
            if (parent is SearchSubjectFragment) {
                val keyWord = kotlin.runCatching {
                        mSearchHistoryAdapter?.getItem(position)?.strVal
                    }.getOrNull()
                keyWord ?: return@setOnItemChildClickListener
                parent.clickKeyWordSearch(keyWord, SearchReportHelper.SEARCH_FROM_HISTORY) {
                    mSearchHistoryAdapter?.addData(HistoryVal(TYPE_HISTORY_VAL, it))
                }

               SearchReportHelper.reportHistoryWordClick(keyWord, position)
            }
        }

        loadHistoryData()
    }

    private fun loadHistoryData() {
        SearchManager.INSTANCE.loadSearch(SEARCH_MOVIE, object : ISearchCallback {
            override fun onSearchData(list: MutableList<String>) {
                activity?.runOnUiThread {
                    val showList = mutableListOf<String>()
                    showList.addAll(list)
                    showList.reverse()
                    mHistoryList = showList
                    showHistoryList()
                    mViewBinding?.rlTips?.visibility = if (list.isNotEmpty()) View.VISIBLE else View.GONE
                }
            }
        })

        if (parentFragment is SearchSubjectFragment) {
            (parentFragment as SearchSubjectFragment).setNextHint()
        }
    }


    @SuppressLint("NotifyDataSetChanged")
    private fun expandHistoryLine(isExpand: Boolean) {
        mViewBinding?.rvHistory?.post {
            if (mViewBinding?.rvHistory?.layoutManager !is LinesFlexBoxLayoutManager)  {
                return@post
            }

            val flexboxLayoutManager = mViewBinding?.rvHistory?.layoutManager as LinesFlexBoxLayoutManager
            val lineCount = flexboxLayoutManager.getActualLineCount()
            if (lineCount <= 0) return@post
            flexboxLayoutManager.setFlowLine(if (isExpand) historyExpandLine else historyCollapseLine)

            val showMore = if (isExpand) true else lineCount > historyCollapseLine
            val moreIndex = flexboxLayoutManager.getRenderedItemCount() - 1
            mSearchHistoryAdapter?.setShowMoreMode(
                moreShow = showMore,
                isExpand,
                moreIndex
            )
            mSearchHistoryAdapter?.notifyDataSetChanged()

            //避免没有箭头的情况，第三次渲染
            if (showMore) {
                mViewBinding?.rvHistory?.post {
                    //渲染数量不到moreIndex展示的位置，再次调整
                    if (flexboxLayoutManager.childCount <= moreIndex) {
                        mSearchHistoryAdapter?.setShowMoreMode(
                            true,
                            isExpand,
                            moreIndex - 1
                        )
                    }
                    mSearchHistoryAdapter?.notifyDataSetChanged()
                }
            }
        }
    }

    override fun onHiddenChanged(hidden: Boolean) {
        super.onHiddenChanged(hidden)
        if (hidden.not()) {
            loadHistoryData()
        }
    }

    private fun showHistoryList() {
        if (!isAdded || isDetached || isRemoving) {
            return
        }

        //历史词展开行数AB
        val showDoubleLines = ConfigManager.instance.getConfig(SearchConstants.SEARCH_HISTORY_LINES_DOUBLE, false)?.value?.toBooleanStrictOrNull()
        if (showDoubleLines == true) {
            historyExpandLine = 4
            historyCollapseLine = 2
        }

        //展开收起能力设置
        mSearchHistoryAdapter?.setMoreClickListener {
            expand = !expand
            expandHistoryLine(expand)
            hideSoftInput(mViewBinding?.rvHistory)
        }
        mViewBinding?.rvHistory?.layoutManager = LinesFlexBoxLayoutManager(requireContext()).apply {
            flexDirection = FlexDirection.ROW
            flexWrap = FlexWrap.WRAP
            justifyContent = JustifyContent.FLEX_START
            setFlowLine(if (expand) historyExpandLine else historyCollapseLine)
        }
        mViewBinding?.rvHistory?.adapter = mSearchHistoryAdapter
        mViewBinding?.rvHistory?.itemAnimator = null
        val list = mHistoryList.map { HistoryVal(TYPE_HISTORY_VAL, it) }.toMutableList()
        mSearchHistoryAdapter?.setNewInstance(list)

        mViewBinding?.rvHistory?.post {
            expandHistoryLine(expand)
        }
    }

    private fun initHotWordWithData(subjectEntity: HotSubjectEntity?) {
        mViewBinding?.searchHotEveryoneLinear?.apply {
            // everyone is search 下发7个热搜词，剔除一个会在搜索框出现热搜词，展示剩余的7个
            val size = subjectEntity?.everyoneSearch?.size ?: 0
            if (size > 0) {
                visible()
                // 剔除搜索框中出现的热搜词
                val hotSearchList = mutableListOf<HotSearchKeyWord>()
                subjectEntity?.everyoneSearch?.forEach { hotSearchKeyWord ->
                    if (hotSearchKeyWord.title != hotSearchWord) {
                        hotSearchList.add(hotSearchKeyWord)
                    }
                }

                everyOneSearchAdapter?.setList(hotSearchList)
                mViewBinding?.everyoneSearchRv?.post {
                    val layoutManager = mViewBinding?.everyoneSearchRv?.layoutManager
                    if (layoutManager is LinesFlexBoxLayoutManager) {
                        for (i in 0 until layoutManager.getRenderedItemCount()) {
                            if (i < hotSearchList.size) {
                                SearchReportHelper.reportHotWordBrowse(hotSearchList[i].title, i)
                            }
                        }
                    }
                }

            } else {
                gone()
            }
        }
    }

    private fun initEveryoneSearch() {
        hotSearchWord = arguments?.getString(SearchConstants.HOT_SEARCH_WORD)
        everyOneSearchAdapter = SearchHotKeywordAdapter().apply {
            addChildClickViewIds(R.id.tv_keyword)
            setOnItemChildClickListener { _, _, position ->
                val parent = parentFragment
                if (parent is SearchSubjectFragment) {
                    val keyWord = kotlin.runCatching {
                        everyOneSearchAdapter?.getItem(position)?.title
                    }.getOrNull()

                    keyWord ?: return@setOnItemChildClickListener
                    parent.clickKeyWordSearch(keyWord, SearchReportHelper.SEARCH_FROM_HOT) {
                        mSearchHistoryAdapter?.addData(HistoryVal(TYPE_HISTORY_VAL, it))
                    }

                    //上报
                    SearchReportHelper.reportHotWordClick(keyWord, position)
                }
            }
        }
        mViewBinding?.everyoneSearchRv?.apply {
            layoutManager = LinesFlexBoxLayoutManager(requireContext()).apply {
                flexDirection = FlexDirection.ROW
                flexWrap = FlexWrap.WRAP
                justifyContent = JustifyContent.FLEX_START
                setFlowLine(historyExpandLine)
            }
            adapter = everyOneSearchAdapter
        }
    }

    private fun initRankWithData(it: HotSubjectEntity?) {
        if (it?.hot.isNullOrEmpty()) {
            mViewBinding?.searchHotRankMagicIndicatorLl?.invisible()
        } else {
            mViewBinding?.searchHotRankMagicIndicatorLl?.visible()
            mHotSubjectEntity = it
            mViewBinding?.searchHotRankMagicIndicator?.navigator?.notifyDataSetChanged()
            mViewBinding?.searchHotRankViewPager?.adapter?.notifyDataSetChanged()
        }
    }

    /**
     * 排行榜viewPager
     */
    private fun initViewPager() {
        mViewBinding?.searchHotRankMagicIndicator?.navigator =
            CommonNavigator(requireContext()).apply {
                adapter = object : CommonNavigatorAdapter() {
                    override fun getCount(): Int {
                        return mHotSubjectEntity?.hot?.size ?: 0
                    }

                    override fun getTitleView(context: Context, index: Int): IPagerTitleView {
                        val title = mHotSubjectEntity?.hot?.get(index)?.name ?: ""
                        return getIndicatorText(context, index, title)
                    }

                    override fun getIndicator(context: Context): IPagerIndicator {
                        return getIndicatorView(context)
                    }

                }
            }

        mViewBinding?.searchHotRankViewPager?.adapter = object : FragmentStateAdapter(this) {
            override fun getItemCount(): Int {
                return mHotSubjectEntity?.hot?.size ?: 0
            }

            override fun createFragment(position: Int): Fragment {
                val item = mHotSubjectEntity?.hot?.get(position)
                item?.index = position
                return SearchRankPagerFragment.newInstance(item)
            }
        }
        mViewBinding?.searchHotRankViewPager?.setLayerType(View.LAYER_TYPE_HARDWARE, null)
        mViewBinding?.searchHotRankViewPager?.setPageTransformer(CustomMarginTransformer())
        mViewBinding?.searchHotRankViewPager?.offscreenPageLimit = 1

        mViewBinding?.searchHotRankViewPager?.registerOnPageChangeCallback(object :
            ViewPager2.OnPageChangeCallback() {
            override fun onPageScrollStateChanged(state: Int) {
                super.onPageScrollStateChanged(state)
                mViewBinding?.searchHotRankMagicIndicator?.onPageScrollStateChanged(state)
            }

            override fun onPageSelected(position: Int) {
                super.onPageSelected(position)
                val currentFragment = childFragmentManager.findFragmentByTag("f$position")
                if (currentFragment is SearchRankPagerFragment) {
                    currentFragment.onPageSelected()
                }
                mViewBinding?.searchHotRankMagicIndicator?.onPageSelected(position)
            }

            override fun onPageScrolled(position: Int, positionOffset: Float, positionOffsetPixels: Int) {
                super.onPageScrolled(position, positionOffset, positionOffsetPixels)
                mViewBinding?.searchHotRankMagicIndicator?.onPageScrolled(
                    position,
                    positionOffset,
                    positionOffsetPixels
                )
            }
        })
    }

    private fun getIndicatorText(context: Context, index: Int, textStr: String): IPagerTitleView {
        return CustomPagerTitleView(
            context,
            Gravity.CENTER,
            SizeUtils.dp2px(10f)
        ).apply {
            setSelectTextSize(16f)
            text = textStr
            setOnClickListener(object : MultipleClickListener() {
                override fun onSingleClick(v: View?) {
                    mViewBinding?.searchHotRankViewPager?.setCurrentItem(index, false)
                    mViewBinding?.searchHotRankMagicIndicator?.onPageScrolled(index, 0f, 0)
                }

                override fun onDoubleClick(v: View?) {
                }
            })
        }
    }

    private fun getIndicatorView(context: Context): IPagerIndicator {
        return GradientLinePagerIndicator(context).apply {
            setMode(LinePagerIndicator.MODE_WRAP_CONTENT)
            // 宽高
            setLineHeight(SizeUtils.dp2px(2f).toFloat())
            setLineWidth(SizeUtils.dp2px(30f).toFloat())
            setStartInterpolator(AccelerateInterpolator())
            setEndInterpolator(DecelerateInterpolator(2f))
            // 线条颜色
            val color = ContextCompat.getColor(context, com.tn.lib.widget.R.color.white)
            setColors(color, color, color)
        }
    }

    private fun clearTipsDialog() {
        try {
            TRDialog.Builder().setTitle(getString(R.string.search_clear_title))
                .setMessage(getString(R.string.search_clear_des))
                .setLeftText(getString(R.string.search_clear_cancel))
                .setRightText(getString(R.string.search_clear_clear))
                .setRightBtnResId(com.tn.lib.widget.R.drawable.libui_sub_btn2_normal)
                .setListener(this).build().showDialog(this, "clear_tips")
        } catch (e: Exception) {
            Logger.d("e ${e.localizedMessage}")
        }
    }

    override fun onLeftButtonClick(dialog: TRDialog) {
        //
    }

    override fun onRightButtonClick(dialog: TRDialog) {
        if (TextUtils.equals(dialog.tag, "clear_tips")) {
            SearchManager.INSTANCE.clearSearch()
            mHistoryList.clear()
            mViewBinding?.rlTips?.visibility = View.GONE
            mSearchHistoryAdapter?.setNewInstance(mutableListOf())
            val flexboxLayoutManager =
                mViewBinding?.rvHistory?.layoutManager as LinesFlexBoxLayoutManager
            flexboxLayoutManager.resetLineData()
            expand = false
            expandHistoryLine(isExpand = false)
            SearchReportHelper.reportHistoryDelete()
        }
    }

    class CustomMarginTransformer(
        private val revealPx: Int = 38.dp(), // 要露出的像素值
    ) : ViewPager2.PageTransformer {
        override fun transformPage(page: View, position: Float) {
            when {
                position < -1 -> page.translationX = 0f
                position <= 0 -> page.translationX = 0f
                position <= 1 -> page.translationX = -(revealPx * position)
                else -> page.translationX = -(revealPx * (position - 1))
            }
        }
    }
}