package com.transsion.search.fragment.result

import android.annotation.SuppressLint
import android.content.Context
import android.os.Bundle
import android.view.Gravity
import android.view.LayoutInflater
import android.view.View
import android.view.animation.AccelerateInterpolator
import android.view.animation.DecelerateInterpolator
import android.widget.ImageView
import android.widget.TextView
import androidx.appcompat.widget.AppCompatTextView
import androidx.core.content.ContextCompat
import androidx.fragment.app.Fragment
import androidx.lifecycle.ViewModelProvider
import androidx.viewpager2.adapter.FragmentStateAdapter
import androidx.viewpager2.widget.ViewPager2
import com.blankj.utilcode.util.SizeUtils
import com.tn.lib.logger.Logger
import com.tn.lib.util.networkinfo.NetworkUtil
import com.tn.lib.view.GradientLinePagerIndicator
import com.transsion.baselib.report.LogViewConfig
import com.transsion.baseui.util.MultipleClickListener
import com.transsion.baseui.widget.CustomPagerTitleView
import com.transsion.baseui.widget.ResourcesRequestView
import com.transsion.push.notification.permission.NoticePermissionFrom
import com.transsion.push.notification.permission.NotificationPermissionManager
import com.transsion.search.R
import com.transsion.search.bean.SearchResultEntity
import com.transsion.search.bean.TabItem
import com.transsion.search.databinding.FragmentSearchResultBinding
import com.transsion.search.fragment.BaseSearchMainFragment
import com.transsion.search.widget.SearchReportHelper
import net.lucode.hackware.magicindicator.MagicIndicator
import net.lucode.hackware.magicindicator.buildins.commonnavigator.CommonNavigator
import net.lucode.hackware.magicindicator.buildins.commonnavigator.abs.CommonNavigatorAdapter
import net.lucode.hackware.magicindicator.buildins.commonnavigator.abs.IPagerIndicator
import net.lucode.hackware.magicindicator.buildins.commonnavigator.abs.IPagerTitleView
import net.lucode.hackware.magicindicator.buildins.commonnavigator.indicators.LinePagerIndicator

/**
 * Created by gaoxin.liu on 2025/3/18.
 * Description:搜索结果页
 */
class SearchResultFragment : BaseSearchMainFragment<FragmentSearchResultBinding>(){
    companion object {
        const val PAGE_NAME = "searchresult"
    }

    private var keyword: String? = null
    private val viewModel by lazy {
        ViewModelProvider(requireActivity())[SearchResultViewModel::class.java]
    }

    //搜索结果
    private var searchResultData: SearchResultEntity? = null
    private var requestView: ResourcesRequestView? = null
    private var requestMovieCount: Int? = null
    private var searchFrom: String? = null

    override fun getViewBinding(inflater: LayoutInflater): FragmentSearchResultBinding {
        return FragmentSearchResultBinding.inflate(inflater)
    }

    @SuppressLint("NotifyDataSetChanged")
    override fun initView(view: View, savedInstanceState: Bundle?) {
        initViewPager()

        //初始化的搜索结果
        viewModel.searchResultLiveData.observe(viewLifecycleOwner) {
            hideLoading()
            if (it == null || it.tabs.isNullOrEmpty() || it.results.isNullOrEmpty()) {
                mViewBinding?.searchResultEmptyView?.removeAllViews()
                mViewBinding?.searchResultEmptyView?.addView(noResultView())
                mViewBinding?.searchResultMagicIndicatorDivider?.visibility = View.INVISIBLE
                mViewBinding?.searchResultEmptyView?.visibility = View.VISIBLE

                //上报无结果
                if (it?.pager != null && it.results.isNullOrEmpty()) {
                    SearchReportHelper.reportNoResult(keyword, searchFrom)
                }

            } else {
                searchResultData = it
                mViewBinding?.searchResultMagicIndicator?.navigator?.notifyDataSetChanged()
                mViewBinding?.searchResultViewPager?.adapter?.notifyDataSetChanged()
            }
        }

        //求片
        viewModel.requestMovieResult.observe(this) { result ->
            Logger.d("Post request succeed $result, ${result?.count}")
            if (result != null && requestMovieCount == null) {
                requestMovieCount = result.count ?: 0
                requestView?.setCount(result.count ?: 0, animated = false)
            }
        }

        mViewBinding?.let {
            bindViewPager(it.searchResultMagicIndicator, it.searchResultViewPager)
        }
    }

    private fun initViewPager() {
        mViewBinding?.searchResultMagicIndicator?.navigator = CommonNavigator(context).apply {
            adapter = object : CommonNavigatorAdapter() {
                override fun getCount(): Int {
                    return searchResultData?.tabs?.size ?: 0
                }

                override fun getTitleView(context: Context, index: Int): IPagerTitleView {
                    return getIndicatorText(context, index, searchResultData?.tabs?.get(index)?.name)
                }

                override fun getIndicator(context: Context): IPagerIndicator {
                    return getIndicatorView(context)
                }
            }
        }

        mViewBinding?.searchResultViewPager?.adapter = object : FragmentStateAdapter(this) {
            override fun getItemCount(): Int {
                return searchResultData?.tabs?.size ?: 0
            }

            override fun createFragment(position: Int): Fragment {
                return getFragmentByType(searchResultData?.tabs?.get(position))
            }
        }
    }

    private fun getFragmentByType(tabItem: TabItem?): Fragment {
        var extraResultData: SearchResultEntity? = null
        if (tabItem?.tabId == searchResultData?.tabId) {
            extraResultData = searchResultData
        }
        return SearchResultTabFragment.newInstance(keyword, tabItem, extraResultData, searchFrom)
    }

    //绑定viewpager和indicator
    private fun bindViewPager(magicIndicator: MagicIndicator, viewPager: ViewPager2) {
        viewPager.registerOnPageChangeCallback(object : ViewPager2.OnPageChangeCallback() {
            override fun onPageScrolled(position: Int, positionOffset: Float, pOffPixels: Int) {
                magicIndicator.onPageScrolled(position, positionOffset, pOffPixels)
            }

            override fun onPageSelected(position: Int) {
                magicIndicator.onPageSelected(position)
                //tab点击上报
                SearchReportHelper.reportTabClick(keyword, searchResultData?.tabs?.get(position)?.tabId)
            }

            override fun onPageScrollStateChanged(state: Int) {
                magicIndicator.onPageScrollStateChanged(state)
            }
        })
    }

    override fun startLoading() {
        mViewBinding?.searchResultProgressBar?.visibility = View.VISIBLE
        mViewBinding?.searchResultViewPager?.visibility = View.INVISIBLE
        mViewBinding?.searchResultMagicIndicator?.visibility = View.INVISIBLE
        mViewBinding?.searchResultMagicIndicatorDivider?.visibility = View.INVISIBLE

        mViewBinding?.searchResultEmptyView?.visibility = View.INVISIBLE
    }

    override fun hideLoading() {
        mViewBinding?.searchResultProgressBar?.visibility = View.INVISIBLE
        mViewBinding?.searchResultViewPager?.visibility = View.VISIBLE
        mViewBinding?.searchResultMagicIndicator?.visibility = View.VISIBLE
        mViewBinding?.searchResultMagicIndicatorDivider?.visibility = View.VISIBLE
    }

    override fun lazyLoadData() {
        //do nothing
    }

    private fun noResultView(): View {
        if (NetworkUtil.hasCapabilityAsync()) {
            val view = layoutInflater.inflate(R.layout.view_search_empty, mViewBinding?.searchResultEmptyView, false)
            val requestView = view.findViewById<ResourcesRequestView>(R.id.resources_request)
            this.requestView = requestView
            val originKeyword = keyword ?: ""
            val splitKeyword = if (originKeyword.length > 10) originKeyword.substring(0, 10) + "..." else originKeyword
            requestView.set(getString(R.string.search_value_no_result, keyword))
            requestView.setGray(true)
            viewModel.postRequestResource(splitKeyword, before = true)
            requestMovieCount = null

            requestView.mViewBinding?.ivCenterView?.setOnClickListener {
                //无资源，检测通知权限弹窗
                NotificationPermissionManager.checkRequestPermission(this, NoticePermissionFrom.DETAIL_NO_RES)
                viewModel.postRequestResource(splitKeyword)
                // 如果已经初始化了，则之后的数字变化就本地自增
                if (requestMovieCount != null) {
                    requestMovieCount = requestMovieCount!! + 1
                    requestView.setCount(requestMovieCount!!)
                }
            }
            return view

        } else {
            val view = layoutInflater.inflate(R.layout.view_search_no_network,  mViewBinding?.searchResultEmptyView, false)
            val emptyTipsView = view.findViewById<TextView>(R.id.tv_empty_tips)
            val imageView = view.findViewById<ImageView>(R.id.iv_tips)
            val retryBtn = view.findViewById<AppCompatTextView>(R.id.tv_retry)
            emptyTipsView.setText(com.transsion.baseui.R.string.base_net_err)
            imageView.setImageResource(com.tn.lib.widget.R.mipmap.ic_no_network)
            retryBtn.visibility = View.VISIBLE
            retryBtn.setOnClickListener {
                startSearch(keyword ?: "", searchFrom)
            }
            return view
        }
    }

    fun startSearch(keyword: String, searchFrom: String?) {
        this.keyword = keyword
        startLoading()
        searchResultData = null
        //重置页面
        initViewPager()

        this.searchFrom = searchFrom
        viewModel.setNextPage(1)
        viewModel.requestSearchResult(keyword)

        //上报埋点
        SearchReportHelper.reportSearchChange(keyword, searchFrom)
        logViewConfig?.paramsMap?.put("keyword", keyword)

    }

    fun navigateToTab(id: String){
        val index = searchResultData?.tabs?.indexOfFirst { it.tabId == id }
        mViewBinding?.searchResultViewPager?.setCurrentItem(index ?: 0, false)
    }

    override fun newLogViewConfig(): LogViewConfig {
        return LogViewConfig(PAGE_NAME).apply {
            needPv = true
            paramsMap["search_id"] = sSearchId
        }
    }

    private fun getIndicatorText(context: Context, index: Int, textStr: String?): IPagerTitleView {
        return CustomPagerTitleView(context, Gravity.CENTER, SizeUtils.dp2px(12f)).apply {
            setSelectTextSize(16f)
            text = textStr

            setOnClickListener(object : MultipleClickListener() {
                override fun onSingleClick(v: View?) {
                    mViewBinding?.searchResultViewPager?.setCurrentItem(index, false)
                }

                override fun onDoubleClick(v: View?) {
                }
            })
        }
    }

    private fun getIndicatorView(context: Context): IPagerIndicator {
        return GradientLinePagerIndicator(context).apply {
            setMode(LinePagerIndicator.MODE_WRAP_CONTENT)
            // 宽高
            setLineHeight(SizeUtils.dp2px(2f).toFloat())
            setLineWidth(SizeUtils.dp2px(24f).toFloat())
            setStartInterpolator(AccelerateInterpolator())
            setEndInterpolator(DecelerateInterpolator(2f))
            // 线条颜色
            val color = ContextCompat.getColor(context, com.tn.lib.widget.R.color.white)
            setColors(color, color, color)
        }
    }
}