package com.transsion.search.fragment.result

import android.content.Intent
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.widget.ImageView
import android.widget.TextView
import androidx.appcompat.widget.AppCompatTextView
import androidx.lifecycle.ViewModelProvider
import androidx.recyclerview.widget.GridLayoutManager
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.alibaba.android.arouter.launcher.ARouter
import com.blankj.utilcode.util.SizeUtils
import com.tn.lib.decoration.GridItemDecoration
import com.tn.lib.decoration.SimpleItemDecoration
import com.tn.lib.logger.Logger
import com.tn.lib.util.networkinfo.NetworkUtil
import com.tn.lib.view.layoutmanager.NpaGridLayoutManager
import com.transsion.baselib.exposure.IExposureCallback
import com.transsion.baselib.exposure.ListExposureHelper
import com.transsion.baselib.utils.openDeeplink
import com.transsion.baseui.widget.ResourcesRequestView
import com.transsion.moviedetailapi.MovieDetailPageIdentity
import com.transsion.push.notification.permission.NoticePermissionFrom
import com.transsion.push.notification.permission.NotificationPermissionManager
import com.transsion.room.api.RoomConstants
import com.transsion.room.api.RoomPageIdentity
import com.transsion.search.R
import com.transsion.search.bean.SearchResultEntity
import com.transsion.search.bean.TabItem
import com.transsion.search.databinding.FragmentSearchResultListBinding
import com.transsion.search.fragment.BaseSearchMainFragment
import com.transsion.search.fragment.result.adapter.SearchResultTabAdapter
import com.transsion.search.fragment.result.adapter.SearchResultTabHeaderAdapter
import com.transsion.search.goSubjectDetail
import com.transsion.search.isTablet
import com.transsion.search.widget.SearchReportHelper
import com.transsion.search.widget.SearchReportHelper.Companion.reportResultScrollTime
import androidx.core.net.toUri

/**
 * Created by gaoxin.liu on 2025/3/18.
 * Description: 结果list列表
 */
class SearchResultTabFragment : BaseSearchMainFragment<FragmentSearchResultListBinding>() {
    companion object {
        private const val KEY_TAB_ITEM = "tabItem"
        private const val KEY_TAB_RESULT = "tabResult"
        private const val KEY_TAB_KEYWORD = "tabKeyword"
        private const val KEY_TAB_SEARCH_FROM = "searchFrom"

        const val PER_PAGE_COUNT = 10

        fun newInstance(keyword: String?, tabItem: TabItem?,
                        resultData: SearchResultEntity?, searchFrom: String?): SearchResultTabFragment {
            val bundle = Bundle().apply {
                putString(KEY_TAB_KEYWORD, keyword)
                putParcelable(KEY_TAB_ITEM, tabItem)
                if (resultData != null) {
                    putParcelable(KEY_TAB_RESULT, resultData)
                }
                putString(KEY_TAB_SEARCH_FROM, searchFrom)
            }

            return SearchResultTabFragment().apply {
                arguments = bundle
            }
        }
    }

    private var mainAdapter: SearchResultTabAdapter? = null
    private val viewModel by lazy {
        ViewModelProvider(this)[SearchResultViewModel::class.java]
    }

    private var keyword: String? = null
    private var tabItem: TabItem? = null

    //埋点用搜索词来源
    private var searchFrom: String? = null
    private var exposureHelper: ListExposureHelper? = null
    //滑动间隔上报
    private var lastScrollTime: Long = 0

    //可能存在二级tab
    private var currentTabId: String? = null
    //只有第一个tab是传递过来的
    private var extraResultData: SearchResultEntity? = null
    private var requestView: ResourcesRequestView? = null
    private var requestMovieCount: Int? = null
    private var isLoadMoreEnd = false

    override fun getViewBinding(inflater: LayoutInflater): FragmentSearchResultListBinding {
        return FragmentSearchResultListBinding.inflate(inflater)
    }

    override fun initView(view: View, savedInstanceState: Bundle?) {
        initExtraData()

        mViewBinding?.searchResultListTabs?.apply {
            layoutManager = LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false)
            val tabAdapter = SearchResultTabHeaderAdapter(currentTabId)
            tabAdapter.setList(tabItem?.subs ?: arrayListOf())
            adapter = tabAdapter

            tabAdapter.setOnItemClickListener { _, _, position ->
                val tabItem = tabItem?.subs?.get(position)
                if (currentTabId != tabItem?.tabId) {
                    currentTabId = tabItem?.tabId
                    tabAdapter.setSelectedTab(tabItem?.tabId ?: "")

                    startLoading(true)
                    viewModel.setNextPage(1)
                    viewModel.requestSearchResult(keyword ?: "", currentTabId)

                    //当前曝光上报
                    exposureHelper?.clearCacheAndContinue()
                    //tab点击上报
                    SearchReportHelper.reportTabClick(keyword, currentTabId)
                }
            }
        }

        mViewBinding?.searchResultListRecycler?.apply {
            if (isTablet(context.resources)) {
                layoutManager = NpaGridLayoutManager(context, 2).apply {
                    spanSizeLookup = object : GridLayoutManager.SpanSizeLookup() {
                        override fun getSpanSize(position: Int): Int {
                            val opType = mainAdapter?.getItem(position)?.type
                            return if (opType == ResultType.MORE || opType == ResultType.TITLE) {
                                2
                            } else {
                                1
                            }
                        }
                    }
                }

                val middleMargin = SizeUtils.dp2px(12f)
                addItemDecoration(GridItemDecoration(0, middleMargin, 0, 0))

            } else {
                layoutManager = LinearLayoutManager(context, LinearLayoutManager.VERTICAL, false)
                //非all和movie的tab，添加额外的margin，剩下12dp写在item布局里
                if (currentTabId != "All" && currentTabId != "Movie" && itemDecorationCount == 0) {
                    addItemDecoration(SimpleItemDecoration(verticalMargin = 8f))
                }
            }

            mainAdapter = SearchResultTabAdapter(exposureHelper, keyword, searchFrom, currentTabId)
            adapter = mainAdapter

            //加载更多
            mainAdapter?.loadMoreModule?.setOnLoadMoreListener {
                viewModel.requestSearchResult(keyword?: "", currentTabId?: "", true)
            }

            mainAdapter?.setOnItemClickListener { _, _, position ->
                mainAdapter?.getItem(position)?.let {
                    onItemClick(it, position)
                }
            }
        }

        initRecyclerScroll()
        initObserver()
    }

    private fun onItemClick(it: ResultWrapData, position: Int) {
        when (it.type) {
            ResultType.SUBJECT -> {
                //跳转详情
                goSubjectDetail(it.subject, "search_result")
                SearchReportHelper.reportSubjectClick(
                    it.subject,
                    position,
                    keyword,
                    searchFrom,
                    currentTabId
                )
            }

            ResultType.STAFF -> {
                ARouter.getInstance().build(MovieDetailPageIdentity.MOVIE_STAFF)
                    .withSerializable("staff", it.staff).navigation()
                SearchReportHelper.reportStaffClick(it.staff, position, keyword, searchFrom, currentTabId)
            }

            ResultType.GROUP -> {
                //跳转圈子
                ARouter.getInstance().build(RoomPageIdentity.ROOM_DETAIL)
                    .withString(RoomConstants.FIELD_ID, it.groups?.groupId)
                    .navigation()
                SearchReportHelper.reportGroupClick(it.groups, position, keyword, searchFrom, currentTabId)
            }

            ResultType.VERTICAL_RANK -> {
                it.verticalRank?.deeplink?.openDeeplink()
                SearchReportHelper.reportRankClick(it.verticalRank, position, keyword, searchFrom, currentTabId)
            }

            ResultType.MORE -> {
                if (parentFragment is SearchResultFragment) {
                    (parentFragment as SearchResultFragment).navigateToTab(it.moreTabId ?: "")
                }
            }

            else -> {
                //do nothing
            }
        }
    }

    private fun initObserver() {
        viewModel.searchResultLiveData.observe(viewLifecycleOwner) {
            mainAdapter?.setList(it?.convertData?: arrayListOf())
            mainAdapter?.setCurrentTab(currentTabId?: "")
            if (it?.convertData.isNullOrEmpty()) {
                requestMovieCount = null
                mainAdapter?.setEmptyView(noResultView())
            }

            if (it?.pager?.hasMore == false || (it?.convertData?.size ?: 0) < PER_PAGE_COUNT) {
                mainAdapter?.loadMoreModule?.loadMoreEnd()
                isLoadMoreEnd = true
            } else {
                mainAdapter?.loadMoreModule?.loadMoreComplete()
            }

            hideLoading()
        }

        viewModel.searchMoreLiveData.observe(viewLifecycleOwner) {
            if (it?.tabId == currentTabId) {
                if (it?.results == null) {
                    mainAdapter?.loadMoreModule?.loadMoreFail()

                } else {
                    mainAdapter?.addData(it.convertData?: arrayListOf())

                    if (!it.pager.hasMore || it.convertData.isNullOrEmpty()) {
                        mainAdapter?.loadMoreModule?.loadMoreEnd()
                    } else {
                        mainAdapter?.loadMoreModule?.loadMoreComplete()
                    }
                }
            }
        }

        //求片
        viewModel.requestMovieResult.observe(this) { result ->
            Logger.d("Post request succeed $result, ${result?.count}")
            if (result != null && requestMovieCount == null) {
                requestMovieCount = result.count ?: 0
                requestView?.setCount(result.count ?: 0, animated = false)
            }
        }
    }

    //获取初始化的数据
    @Suppress("DEPRECATION")
    private fun initExtraData() {
        keyword = arguments?.getString(KEY_TAB_KEYWORD)
        tabItem = arguments?.getParcelable(KEY_TAB_ITEM)
        extraResultData = arguments?.getParcelable(KEY_TAB_RESULT)
        searchFrom = arguments?.getString(KEY_TAB_SEARCH_FROM)

        currentTabId = if (tabItem?.subs?.isNotEmpty() == true) {
            tabItem?.subs?.get(0)?.tabId
        } else {
            tabItem?.tabId
        }
    }

    //曝光处理
    private fun initRecyclerScroll() {
        exposureHelper = ListExposureHelper(0.6f, object : IExposureCallback {
            override fun exposure(position: Int, view: View?) {

            }

            override fun exposure(position: Int, duration: Long, view: View?) {
                if (position >= (mainAdapter?.data?.size ?: 0)) {
                    return
                }
                //曝光会延后触发，用缓存的tabId
                val tabId = mainAdapter?.getCurrentTabId()?: currentTabId

                val item = mainAdapter?.data?.get(position)
                when (item?.type) {
                    ResultType.SUBJECT -> {
                        SearchReportHelper.reportSubjectBrowse(item.subject, position,
                            duration, keyword, searchFrom, tabId)
                    }

                    ResultType.STAFF -> {
                        SearchReportHelper.reportStaffBrowse(item.staff, position,
                            duration, keyword, searchFrom, tabId)
                    }

                    ResultType.GROUP -> {
                        SearchReportHelper.reportGroupBrowse(item.groups, position,
                            duration, keyword, searchFrom, tabId)
                    }

                    ResultType.VERTICAL_RANK -> {
                        SearchReportHelper.reportRankBrowse(item.verticalRank, position,
                            duration, keyword, searchFrom, tabId)
                    }

                    else -> {

                    }
                }
            }
        }).apply {
            setCheckType(2)
            mViewBinding?.searchResultListRecycler?.addOnScrollListener(this)
        }

        //上报滑动事件
        mViewBinding?.searchResultListRecycler?.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrollStateChanged(recyclerView: RecyclerView, newState: Int) {
                if (newState == RecyclerView.SCROLL_STATE_DRAGGING) {
                    var fromIndex = -1
                    val layoutManager = mViewBinding?.searchResultListRecycler?.layoutManager
                    if (layoutManager is LinearLayoutManager) {
                        fromIndex = layoutManager.findLastVisibleItemPosition()
                    }

                    reportResultScrollTime(
                        currentTabId,
                        keyword,
                        System.currentTimeMillis() - lastScrollTime,
                        fromIndex
                    )
                } else if (newState == RecyclerView.SCROLL_STATE_IDLE) {
                    lastScrollTime = System.currentTimeMillis()
                }
            }
        })

        //初始化滑动开始时间
        mViewBinding?.searchResultListRecycler?.post {
            if (lastScrollTime == 0L) {
                lastScrollTime = System.currentTimeMillis()
            }
        }
    }

    override fun onResume() {
        super.onResume()
        if (isLoadMoreEnd) {
            mainAdapter?.loadMoreModule?.loadMoreEnd()
        }
    }

    override fun onPause() {
        super.onPause()
        exposureHelper?.clearCache()
    }

    override fun onHiddenChanged(hidden: Boolean) {
        super.onHiddenChanged(hidden)
        if (hidden) {
            exposureHelper?.clearCache()
        }
    }

    private fun startLoading(showTab: Boolean) {
        mViewBinding?.searchResultTabProgressBar?.visibility = View.VISIBLE
        if (showTab.not()) {
            mViewBinding?.searchResultListTabs?.visibility = View.GONE
        }
        mViewBinding?.searchResultListRecycler?.visibility = View.INVISIBLE
        mViewBinding?.searchResultListTabs?.isEnabled = false
    }

    override fun hideLoading() {
        mViewBinding?.searchResultTabProgressBar?.visibility = View.INVISIBLE
        mViewBinding?.searchResultListTabs?.visibility = View.VISIBLE
        mViewBinding?.searchResultListRecycler?.visibility = View.VISIBLE
        mViewBinding?.searchResultListTabs?.isEnabled = true
    }

    override fun lazyLoadData() {
        //tabId和默认tabId一致，显示结果
        if (extraResultData != null) {
            mainAdapter?.setList(extraResultData?.convertData?: arrayListOf())
            if (extraResultData?.pager?.hasMore == false) {
                mainAdapter?.loadMoreModule?.loadMoreEnd(true)
            } else {
                viewModel.setNextPage(2)
            }

        } else {
            startLoading(showTab = false)
            viewModel.requestSearchResult(keyword ?: "", currentTabId?: "")
        }
    }

    private fun noResultView(): View {
        if (NetworkUtil.hasCapabilityAsync()) {
            val view = layoutInflater.inflate(R.layout.view_search_empty, mViewBinding?.searchResultListRecycler, false)
            val requestView = view.findViewById<ResourcesRequestView>(R.id.resources_request)
            this.requestView = requestView
            val originKeyword = keyword ?: ""
            val splitKeyword = if (originKeyword.length > 10) originKeyword.substring(0, 10) + "..." else originKeyword
            requestView.set(getString(R.string.search_value_no_result, keyword))
            requestView.setGray(true)
            viewModel.postRequestResource(splitKeyword, before = true)
            requestMovieCount = null

            requestView.mViewBinding?.ivCenterView?.setOnClickListener {
                //无资源，检测通知权限弹窗
                NotificationPermissionManager.checkRequestPermission(this, NoticePermissionFrom.DETAIL_NO_RES)
                viewModel.postRequestResource(splitKeyword)
                // 如果已经初始化了，则之后的数字变化就本地自增
                if (requestMovieCount != null) {
                    requestMovieCount = requestMovieCount!! + 1
                    requestView.setCount(requestMovieCount!!)
                }
            }
            return view

        } else {
            val view = layoutInflater.inflate(R.layout.view_search_no_network,  mViewBinding?.searchResultListRecycler, false)
            val emptyTipsView = view.findViewById<TextView>(R.id.tv_empty_tips)
            val imageView = view.findViewById<ImageView>(R.id.iv_tips)
            val retryBtn = view.findViewById<AppCompatTextView>(R.id.tv_retry)
            emptyTipsView.setText(com.transsion.baseui.R.string.base_net_err)
            imageView.setImageResource(com.tn.lib.widget.R.mipmap.ic_no_network)
            retryBtn.visibility = View.VISIBLE
            retryBtn.setOnClickListener {
                lazyLoadData()
            }
            return view
        }
    }
}