package com.transsion.search.fragment.values

import android.content.res.ColorStateList
import android.graphics.Rect
import android.graphics.Typeface
import android.net.Uri
import android.os.Bundle
import android.view.Gravity
import android.view.LayoutInflater
import android.view.View
import android.widget.ImageView
import android.widget.TextView
import androidx.appcompat.widget.AppCompatTextView
import androidx.core.content.ContextCompat
import androidx.lifecycle.MutableLiveData
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import androidx.viewpager2.widget.ViewPager2
import com.alibaba.android.arouter.launcher.ARouter
import com.blankj.utilcode.util.SizeUtils
import com.chad.library.adapter.base.BaseQuickAdapter
import com.google.android.material.appbar.AppBarLayout
import com.google.android.material.tabs.TabLayout
import com.google.android.material.tabs.TabLayoutMediator
import com.google.gson.JsonObject
import com.tn.lib.logger.Logger
import com.tn.lib.net.base.BaseObserver
import com.tn.lib.net.env.Host
import com.tn.lib.net.manager.NetServiceGenerator
import com.tn.lib.util.networkinfo.NetworkUtil
import com.transsion.baselib.helper.DataHelper
import com.transsion.baselib.helper.DeeplinkHelper
import com.transsion.baselib.report.LogViewConfig
import com.transsion.baseui.fragment.PageStatusFragment
import com.transsion.baseui.widget.ResourcesRequestView
import com.transsion.moviedetailapi.MovieDetailPageIdentity
import com.transsion.publish.model.PostEntity
import com.transsion.push.notification.permission.NoticePermissionFrom
import com.transsion.push.notification.permission.NotificationPermissionManager
import com.transsion.search.R
import com.transsion.search.fragment.values.adapter.SearchValuesRelatedCollectionAdapter
import com.transsion.search.fragment.values.adapter.SearchValuesTypeFragmentAdapter
import com.transsion.search.bean.Count
import com.transsion.search.bean.SearchValuesRelatedCollectionEntity
import com.transsion.search.bean.SearchWorkEntity
import com.transsion.search.bean.VerticalRank
import com.transsion.search.constant.SearchConstants
import com.transsion.search.databinding.FragmentSearchValuesLayoutBinding
import com.transsion.search.fragment.SearchSubjectFragment
import io.reactivex.rxjava3.schedulers.Schedulers
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.RequestBody.Companion.toRequestBody
import java.util.HashMap

/**
 * @author xinbing.zhang
 * @date :2023/8/17 10:45
 * @description: https://transsioner.feishu.cn/docx/Dhl3d8YBgofkRcxzCnYcDjp3ndg 搜索增加作品类型选项
 *
 */
class SearchValuesFragment : PageStatusFragment<FragmentSearchValuesLayoutBinding>() {

    /*** 榜单数据*/
    private var mVerticalRanks: List<VerticalRank>? = null

    /*** 当前正在搜索的关键字*/
    private var mKeyWord: String = ""
    private var mCopyKeyWord: String? = ""

    private var mType: Int = 1
    private var mEntity: SearchWorkEntity? = null

    /**
     * ViewPager Adapter
     */
    private var fragmentStateAdapter: SearchValuesTypeFragmentAdapter? = null

    /*** 搜索结果 -- 集合列表 适配器*/
    private var searchValuesRelatedCollectionAdapter: SearchValuesRelatedCollectionAdapter? = null

    /*** 组装数据 --> 更新数据*/
    private val collectionList = mutableListOf<SearchValuesRelatedCollectionEntity>()


    // =============================================================================================


    companion object {

        const val TAG = "search"

        private const val M_COPY_KEY_WORD = "mCopyKeyWord"

        const val SEARCH_VALUES_RANK_TYPE_TITLE = 0
        const val SEARCH_VALUES_RANK_TYPE_VERTICAL_RANK = 1
        const val SEARCH_VALUES_RANK_TYPE_STAFF = 2
        const val SEARCH_VALUES_ITEM_SUBJECT = 3
        const val SEARCH_VALUES_ITEM_ACCURATE_SUBJECT = 4



        fun newInstance(): SearchValuesFragment {
            return SearchValuesFragment()
        }

        //串联搜索点击，item曝光，item点击 埋点的id
        var trid = ""

        /**
         * 搜索结果点击事件
         */
        fun reportClick(
            groupId: String?,
            subjectId: String?,
            staffId: String?,
            ops: String?,
            position: Int,
            type: Int,
            trid: String?,
            season:String? = null
        ) {
            val map = HashMap<String, String>()
            map["subject_id"] = subjectId ?: ""
            map["group_id"] = groupId ?: ""
            map["sequence"] = position.toString()
            map["ops"] = ops ?: ""
            map["staff_id"] = staffId ?: ""
            map["type"] = type.toString()
            if (trid?.isNotEmpty() == true) {
                map["trid"] = trid
            }
            if (season?.isNotEmpty() == true){
                map["season"] = season.toString()
            }
            DataHelper.reportFeedClickEvent("searchresult", map)
        }

        /**
         * 搜索结果点击事件
         */
        fun reportShow(
            group_id: String?,
            subject_id: String?,
            staffId: String?,
            ops: String?,
            position: Int,
            type: Int,
        ) {
            val map = HashMap<String, String>()
            map["subject_id"] = subject_id ?: ""
            map["group_id"] = group_id ?: ""
            map["sequence"] = position.toString()
            map["ops"] = ops ?: ""
            map["staff_id"] = staffId ?: ""
            map["type"] = type.toString()
            if (trid.isNotEmpty()) {
                map["trid"] = trid
            }
            DataHelper.reportBrowseEvent("searchresult", map)
        }

    }


    // ============================ 下面是生命周期方法回调 =============================================


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        savedInstanceState?.let {
            mCopyKeyWord = it.getString(M_COPY_KEY_WORD)
//            AdLogger.logW(
//                "SearchValuesFragment --> savedInstanceState --> mCopyKeyWord = $mCopyKeyWord -- this = $this",
//                TAG
//            )
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        if (mKeyWord.length < 30) {
            outState.putString(M_COPY_KEY_WORD, mKeyWord)
        }
    }

    override fun isShowPageStateLayoutTitle(): Boolean = false

    override fun isAudioShowNoNetworkLayout(): Boolean = false

    override fun isMonitorNetworkState(): Boolean = true

    override fun getViewBinding(inflater: LayoutInflater): FragmentSearchValuesLayoutBinding =
        FragmentSearchValuesLayoutBinding.inflate(inflater)

    override fun getPageStateLayoutTitle(): String = ""

    override fun initViewData() {
        //
    }

    override fun initViewModel() {
        //
    }

    override fun initListener() {
        postRequestResult.observe(this) { result ->
            Logger.d("Post request succeed $result, ${result?.count}")
            if (result != null && count == null) {
                Logger.d("count changed ${result.count}")
                count = result.count ?: 0
                requestView?.setCount(result.count ?: 0, animated = false)
            }
        }
    }

    override fun retryLoadData() {
        if (null == mVerticalRanks) {
            (parentFragment as? SearchSubjectFragment)?.searchJob("")
        }
    }

    override fun getEmptyDescText(): String {
        return context?.getString(com.transsion.search.R.string.user_works_empty) + " \"$mKeyWord\""
    }

    override fun getEmptyView(isShowTitleLayout: Boolean): View {
        return noResultView()
    }


    // ================================ 下面是对外暴露方法 =============================================


    /**
     * 设置空数据界面
     */
    fun setEmpty(keyword: String) {
        // 1. 当前信息到本地
        mKeyWord = keyword
        showEmptyView(false)
//        AdLogger.log(
//            "SearchValuesFragment --> setEmpty() --> 点击搜索之后 没有获取到数据 展示空数据界面",
//            TAG
//        )
    }

    /**
     * 用户搜索的词条
     *
     * 这个方法是输入框回调的地方调用的
     */
    fun setSearchValue(keyword: String, entity: SearchWorkEntity?, type: Int) {
//        AdLogger.log(
//            "SearchValuesFragment --> setSearchValue() --> keyword = $keyword -- entity = $entity",
//            TAG
//        )

        // 框架层 展示内容
        showContentView()

        // 1. 当前信息到本地
        mKeyWord = keyword
        if (mKeyWord.isNotEmpty()) {
            SearchConstants.searchMmKV.putString(M_COPY_KEY_WORD, mKeyWord)
        }
        mType = type
        mEntity = entity
        mVerticalRanks = entity?.verticalRanks

        // 2. 榜单数据 Related Collection
        upDateRelatedCollection()
        // 3. TabLayout
        entity?.counts?.let {
            if (it.isNotEmpty()) {
                // 创建 TabLayout + ViewPager2
                showValues(it)
                setTabLayoutVisibility(true)
            } else {
                setTabLayoutVisibility(false)
            }
        }
    }

    /**
     * 没有数据的时候隐藏TabLayout + ViewPage
     */
    private fun setTabLayoutVisibility(isShow: Boolean) {
        mViewBinding?.tabs?.visibility = if (isShow) View.VISIBLE else View.GONE
        mViewBinding?.viewPager?.visibility = if (isShow) View.VISIBLE else View.GONE
        mViewBinding?.viewLine?.visibility =
            if (isShow && collectionList.isNotEmpty()) View.VISIBLE else View.GONE
        setCanDrag(isShow)

        // 每次都是展开的
        mViewBinding?.appBarLayout?.setExpanded(true, false)
    }

    /**
     * 没有作品数据的时候禁止滑动
     */
    private fun setCanDrag(canDrag: Boolean) {
        kotlin.runCatching {
            val mAppBarChildAt = mViewBinding?.appBarLayout?.getChildAt(0)
            val mAppBarParams = mAppBarChildAt?.layoutParams as? AppBarLayout.LayoutParams
            if (canDrag) {
                mAppBarParams?.scrollFlags =
                    AppBarLayout.LayoutParams.SCROLL_FLAG_SCROLL or AppBarLayout.LayoutParams.SCROLL_FLAG_EXIT_UNTIL_COLLAPSED
                mAppBarChildAt?.layoutParams = mAppBarParams
            } else {
                mAppBarParams?.scrollFlags = 0
                mAppBarChildAt?.layoutParams = mAppBarParams
            }
        }
    }

    /**
     * 获取搜索的关键字
     */
    fun getKewWord(): String {
        return SearchConstants.searchMmKV.getString(M_COPY_KEY_WORD, "") ?: ""
    }

    /**
     * 页面销毁的时候
     */
    fun destroy() {
//        AdLogger.log("SearchValuesFragment --> destroy() --> 搜索结果页面隐藏的时候销毁数据", TAG)

        // 重置数据 --> 重新联网的时候通过这个判断是否加载数据
        mVerticalRanks = null
        // 隐藏布局 --> 再次进来的时候会先展示上一次的数据 --> 在刷新
        //mViewBinding?.groupCollection?.visibility = View.GONE
        // 目前简单处理一下，清空之前的数据
        mViewBinding?.apply {
            // ViewPage2清空数据
            if (null != fragmentStateAdapter) {
                fragmentStateAdapter?.clear()
            }
            // TabLayout 清空数据
            tabs.removeAllTabs()
        }
    }


    // ========================== 下面是Related数据展示 ===============================================


    /**
     * 更新Related Collection
     */
    private fun upDateRelatedCollection() {

        // 通过RecyclerView展示
        initRelatedList()
        // 组装数据
        assembleData()

        // 如果有数据，那就展示列表
        if (collectionList.isNotEmpty()) {
            mViewBinding?.rvRank?.visibility = View.VISIBLE
            mViewBinding?.viewLine?.visibility = View.VISIBLE
            // 设置数据
            searchValuesRelatedCollectionAdapter?.setList(collectionList)
        } else {
            mViewBinding?.rvRank?.visibility = View.GONE
            mViewBinding?.viewLine?.visibility = View.GONE
        }

    }

    /**
     * 组装数据
     */
    private fun assembleData() {
        // 重置数据
        collectionList.clear()

        // 位置：位于目前的合辑区域（如果同时出现合集和演员，合集在上演员在下）
        // 合集数据组合
        mEntity?.verticalRanks?.let {
            if (it.isNotEmpty()) {
                // 当前类型的标题
                collectionList.add(
                    SearchValuesRelatedCollectionEntity(
                        SEARCH_VALUES_RANK_TYPE_TITLE,
                        getString(R.string.search_related_collection),
                        null,
                        null
                    )
                )
                it.forEach { each ->
                    collectionList.add(
                        SearchValuesRelatedCollectionEntity(
                            SEARCH_VALUES_RANK_TYPE_VERTICAL_RANK, "", each, null
                        )
                    )
                }
            }
        }

        // 演员数据组合
        mEntity?.staffs?.let {
            if (it.isNotEmpty()) {
                collectionList.add(
                    SearchValuesRelatedCollectionEntity(
                        SEARCH_VALUES_RANK_TYPE_TITLE,
                        getString(R.string.search_related_people),
                        null,
                        null
                    )
                )
                it.forEach { each ->
                    collectionList.add(
                        SearchValuesRelatedCollectionEntity(
                            SEARCH_VALUES_RANK_TYPE_STAFF, "", null, each
                        )
                    )
                }
            }
        }
    }

    /**
     * 初始化
     */
    private fun initRelatedList() {
        if (null == searchValuesRelatedCollectionAdapter) {
            mViewBinding?.rvRank?.apply {
                searchValuesRelatedCollectionAdapter =
                    SearchValuesRelatedCollectionAdapter().apply {
                        setOnItemClickListener { adapter, _, position ->
                            onRelatedItemClick(adapter, position)
                        }
                    }
                adapter = searchValuesRelatedCollectionAdapter
                layoutManager = LinearLayoutManager(activity)
                addItemDecoration(object : RecyclerView.ItemDecoration() {
                    override fun getItemOffsets(
                        outRect: Rect, view: View, parent: RecyclerView, state: RecyclerView.State
                    ) {
                        super.getItemOffsets(outRect, view, parent, state)
                        val position = parent.getChildAdapterPosition(view)
                        val adapter = parent.adapter
                        if (null != adapter && position == 0) {
                            outRect.top = SizeUtils.dp2px(16f)
                        }
                    }
                })
            }
        }

        // 将搜索关键字传入
        searchValuesRelatedCollectionAdapter?.setKeyWord(mKeyWord)
    }

    /**
     * 点击事件处理
     */
    private fun onRelatedItemClick(adapter: BaseQuickAdapter<*, *>, position: Int) {
        (adapter.data[position] as? SearchValuesRelatedCollectionEntity)?.let {

            // 无论点击了什么都关闭输出弹窗
            (parentFragment as? SearchSubjectFragment)?.hideSoftInput()

            when (it.type) {
                SEARCH_VALUES_RANK_TYPE_TITLE -> {
                    // 点击了 标题
                }

                SEARCH_VALUES_RANK_TYPE_VERTICAL_RANK -> {
                    reportClick(
                        "",
                        "",
                        it.staff?.staffId,
                        "",
                        position,
                        SEARCH_VALUES_RANK_TYPE_VERTICAL_RANK,
                        trid
                    )

                    DeeplinkHelper.toRouteUri(
                        Uri.parse(
                            it.verticalRank?.deeplink ?: ""
                        )
                    )?.let { uri ->
                        ARouter.getInstance().build(uri).navigation()
                    }
                }

                SEARCH_VALUES_RANK_TYPE_STAFF -> {
                    reportClick(
                        "", "", it.staff?.staffId, "", position, SEARCH_VALUES_RANK_TYPE_STAFF, trid
                    )
                    ARouter.getInstance().build(MovieDetailPageIdentity.MOVIE_STAFF)
                        .withSerializable("staff", it.staff).navigation()
                }

                else -> {
                    //
                }
            }
        }
    }


    // ===================== 下面是ViewPage展示数据处理 ================================================


    /**
     * 初始化 Tab + ViewPager
     */
    private fun showValues(count: List<Count>) {
        initViewPager2(count)
        initTabLayout(count)
    }

    /**
     * 初始化 ViewPager2
     */
    private fun initViewPager2(count: List<Count>) {
        mViewBinding?.apply {

            // 创建新的Adapter 进行覆盖
            fragmentStateAdapter = SearchValuesTypeFragmentAdapter(
                this@SearchValuesFragment, count, mKeyWord, mEntity, mType
            )

            // 目前全量缓存
            viewPager.offscreenPageLimit = count.size
            // ViewPager 初始化
            viewPager.adapter = fragmentStateAdapter

            // 滑动监听
            viewPager.registerOnPageChangeCallback(object : ViewPager2.OnPageChangeCallback() {
                override fun onPageSelected(position: Int) {
                    super.onPageSelected(position)
                    //
                }
            })
        }
    }

    /**
     * 初始化 TabLayout
     */
    private fun initTabLayout(count: List<Count>) {
        mViewBinding?.apply {
            // TabLayout 初始化
            val mediator = tabLayoutMediator(count)
            // TabLayout 选中监听
            tabs.addOnTabSelectedListener(object : TabLayout.OnTabSelectedListener {
                override fun onTabSelected(tab: TabLayout.Tab?) {
                    val textView = tab?.customView as TextView?
                    textView?.typeface = Typeface.defaultFromStyle(Typeface.BOLD)
                }

                override fun onTabUnselected(tab: TabLayout.Tab?) {
                    val textView = tab?.customView as TextView?
                    textView?.typeface = Typeface.defaultFromStyle(Typeface.NORMAL)
                }

                override fun onTabReselected(tab: TabLayout.Tab?) {
                    //
                }
            })
            mediator.attach()
        }
    }

    /**
     * 自定义 TabLayout --> Tab
     */
    private fun FragmentSearchValuesLayoutBinding.tabLayoutMediator(
        count: List<Count>
    ): TabLayoutMediator {
        val mediator = TabLayoutMediator(
            tabs, viewPager
        ) { tab, position -> //这里可以自定义TabView
            val count1 = count[position]
            val txt = if ((count1.num ?: 0) >= 100) {
                count1.name + " 99+"
            } else {
                count1.name + " " + count1.num
            }
            val tabView = TextView(requireContext())
            val states = arrayOfNulls<IntArray>(2)
            states[0] = intArrayOf(android.R.attr.state_selected)
            states[1] = intArrayOf()
            val selColor = ContextCompat.getColor(requireContext(), com.tn.lib.widget.R.color.main)
            val unSelColor =
                ContextCompat.getColor(requireContext(), com.tn.lib.widget.R.color.text_02)
            val colors = intArrayOf(selColor, unSelColor)
            val colorStateList = ColorStateList(states, colors)
            tabView.text = txt
            tabView.gravity = Gravity.CENTER
            tabView.textSize = 14f
            tabView.setTextColor(colorStateList)
            tab.customView = tabView
            tab.view.setOnClickListener {
                tab.select()
            }
        }
        return mediator
    }


    // ============================ 下面是 resources request 处理 =================================================

    private var count: Int? = null

    private var requestView: ResourcesRequestView? = null

    private val service: com.transsion.search.net.SearchApi by lazy {
        NetServiceGenerator.instance.getService(com.transsion.search.net.SearchApi::class.java)
    }

    val postRequestResult = MutableLiveData<PostEntity?>()

    private fun postRequestResource(keyword: String, before: Boolean = false) {
        val jsonObject = JsonObject()
        jsonObject.addProperty("q", keyword)
        jsonObject.addProperty("before", before)
        // 如果是获取初始值则重置 count
        count = if (before) null else count
        val params = jsonObject.toString().toRequestBody("application/json".toMediaTypeOrNull())

        service.postRequestResource(Host.getHost(), params).subscribeOn(Schedulers.io())
            .subscribe(object : BaseObserver<PostEntity?>() {
                override fun onFailure(code: String?, message: String?) {
                    Logger.e(TAG, "SearchValuesFragment --> postRequestResource --> failed to post request resources $message")
                    postRequestResult.postValue(null)
                }

                override fun onSuccess(data: PostEntity?) {
                    data?.let {
                        postRequestResult.postValue(data)
                    }
                }
            })
    }

    private fun noResultView(): View {
        if (NetworkUtil.hasCapabilityAsync()) {
            val view = layoutInflater.inflate(R.layout.view_search_empty, null)
            val requestView = view.findViewById<ResourcesRequestView>(R.id.resources_request)
            this.requestView = requestView
            val keyword = if (mKeyWord.length > 10) mKeyWord.substring(0, 10) + "..." else mKeyWord
            requestView.set(getString(R.string.search_value_no_result, keyword))
            requestView.setGray(true)
            postRequestResource(mKeyWord, before = true)
            requestView.mViewBinding?.ivCenterView?.setOnClickListener {
                //无资源，检测通知权限弹窗
                NotificationPermissionManager.checkRequestPermission(this, NoticePermissionFrom.DETAIL_NO_RES)
                postRequestResource(mKeyWord)
                // 如果已经初始化了，则之后的数字变化就本地自增
                if (count != null) {
                    count = count!! + 1
                    requestView.setCount(count!!)
                }
            }
            return view
        } else {
            val view = layoutInflater.inflate(R.layout.view_search_no_network, null)
            val emptyTipsView = view.findViewById<TextView>(R.id.tv_empty_tips)
            val imageView = view.findViewById<ImageView>(R.id.iv_tips)
            val retryBtn = view.findViewById<AppCompatTextView>(R.id.tv_retry)
            emptyTipsView.setText(com.transsion.baseui.R.string.base_net_err)
            imageView.setImageResource(com.tn.lib.widget.R.mipmap.ic_no_network)
            retryBtn.visibility = View.VISIBLE
            retryBtn.setOnClickListener {
                retryLoadData()
            }
            return view
        }
    }

    override fun newLogViewConfig(): LogViewConfig {
        return LogViewConfig("searchresult")
    }
}