package com.transsion.search.viewmodel

import android.app.Application
import android.text.TextUtils
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import com.blankj.utilcode.util.Utils
import com.tn.lib.net.base.BaseObserver
import com.tn.lib.net.bean.BaseDto
import com.tn.lib.util.networkinfo.NetworkUtil
import com.tn.lib.widget.toast.core.ToastUtils
import com.transsion.baselib.report.Report
import com.transsion.search.activity.SearchManagerActivity.Companion.log
import com.transsion.search.bean.HotSubjectEntity
import com.transsion.search.bean.JoinGroupEntity
import com.transsion.search.bean.SearchGroupEntity
import com.transsion.search.bean.SearchSuggestEntity
import com.transsion.search.bean.SearchWorkEntity
import com.transsion.search.fragment.suggest.SearchSuggestFragment
import com.transsion.search.widget.BoundaryLinkedHashMap
import com.transsion.search.net.RequestJoinGroupEntity
import com.transsion.search.net.RequestSearchEntity
import com.transsion.search.net.SearchModel
import io.reactivex.rxjava3.disposables.Disposable
import io.reactivex.rxjava3.schedulers.Schedulers
import kotlinx.coroutines.launch


class SearchViewModel(application: Application) : AndroidViewModel(application) {

    private val mGroupsLiveData = MutableLiveData<SearchGroupEntity?>()
    private val mWorksLiveData = MutableLiveData<SearchWorkEntity?>()
    private val mWorksLiveDataV2 = MutableLiveData<SearchWorkEntity?>()
    private val mJoinGroupLiveData = MutableLiveData<String?>()
    private val mGroupsRecommendLiveData = MutableLiveData<SearchGroupEntity?>()
    val hotSubjectLiveData = MutableLiveData<HotSubjectEntity?>()

    private val searchApi: SearchModel by lazy { SearchModel() }

    fun getSearchGroupLiveData(): MutableLiveData<SearchGroupEntity?> {
        return mGroupsLiveData
    }

    fun getSearchWorkLiveData(): MutableLiveData<SearchWorkEntity?> {
        return mWorksLiveData
    }

    fun getSearchWorkLiveDataV2(): MutableLiveData<SearchWorkEntity?> {
        return mWorksLiveDataV2
    }

    fun getGroupRecommendLiveData(): MutableLiveData<SearchGroupEntity?> {
        return mGroupsRecommendLiveData
    }

    fun getJoinGroupLiveData(): MutableLiveData<String?> {
        return mJoinGroupLiveData
    }


    private var mFollowDisposable: Disposable? = null


    fun searchWork(page: Int, perPage: Int, keyword: String) {
        val request = RequestSearchEntity(page, perPage, keyword)
        viewModelScope.launch {
            kotlin.runCatching {
                searchApi.searchSubject(request)
            }.let {
                it.onSuccess {baseDto ->
                    mWorksLiveData.postValue(baseDto.data)
                }

                it.onFailure {
                    com.tn.lib.logger.Logger.w("liup",it.message.toString())
                    if (NetworkUtil.hasCapabilityAsync()){
                        ToastUtils.show(Utils.getApp().getString(com.tn.lib.widget.R.string.no_error_content))
                    } else {
                        ToastUtils.show(Utils.getApp().getString(com.tn.lib.widget.R.string.no_network_toast))
                    }

                    mWorksLiveData.postValue(null)
                }
            }
        }
    }

    /**
     * 同一个Activity里面 Fragment使用的 ViewModel 会复用
     */
    fun searchWorkV2(page: Int, perPage: Int, keyword: String, subjectType: Int) {
        val request = RequestSearchEntity(page, perPage, keyword, subjectType)

        viewModelScope.launch {
            kotlin.runCatching {
                searchApi.searchSubject(request)
            }.let {
                it.onSuccess { baseDto ->
                    mWorksLiveDataV2.postValue(baseDto.data)
                }

                it.onFailure {exception: Throwable ->
                    if (NetworkUtil.hasCapabilityAsync()){
                        ToastUtils.show(Utils.getApp().getString(com.tn.lib.widget.R.string.no_error_content))
                    } else {
                        ToastUtils.show(Utils.getApp().getString(com.tn.lib.widget.R.string.no_network_toast))
                    }

                    mWorksLiveDataV2.postValue(null)
                }
            }
        }
    }

    fun searchGroup(page: Int, perPage: Int, keyword: String) {
        val request = RequestSearchEntity(page, perPage, keyword)
        searchApi.searchGroup(request).subscribeOn(Schedulers.io())
            .subscribe(object : BaseObserver<SearchGroupEntity>() {
                override fun onFailure(code: String?, message: String?) {
                    message?.let {
                        ToastUtils.show(it)
                    }
                    mGroupsLiveData.postValue(null)
                }

                override fun onSuccess2(baseDto: BaseDto<SearchGroupEntity>) {
                    mGroupsLiveData.postValue(baseDto.data)
                }
            })
    }

    fun joinGroup(groupId: String) {
        val request = RequestJoinGroupEntity(groupId)
        searchApi.joinGroup(request).subscribeOn(Schedulers.io())
            .subscribe(object : BaseObserver<JoinGroupEntity>() {
                override fun onFailure(code: String?, message: String?) {
                    message?.let {
                        ToastUtils.show(it)
                    }
                    mJoinGroupLiveData.postValue("")
                }

                override fun onSuccess(data: JoinGroupEntity?) {
                    if (TextUtils.isEmpty(data?.m)) {
                        mJoinGroupLiveData.postValue("succeed")
                        return
                    }
                    mJoinGroupLiveData.postValue(data?.m)
                }

                override fun onSuccess2(baseDto: BaseDto<JoinGroupEntity>) {
                    //
                }
            })
    }

    fun getHotSubject() {
        searchApi.getHotSubject().subscribeOn(Schedulers.io())
            .subscribe(object : BaseObserver<HotSubjectEntity>() {
                override fun onFailure(code: String?, message: String?) {
                    hotSubjectLiveData.postValue(null)
                }

                override fun onSuccess(data: HotSubjectEntity?) {
                    hotSubjectLiveData.postValue(data)
                }
            })
    }

    /**
     * 获取搜索推荐 相关 =======================================================
     */
    val suggestMap = BoundaryLinkedHashMap()

    val suggestFailureMap = BoundaryLinkedHashMap()

    val suggestLiveData = MutableLiveData<String>()

    fun getSearchSuggest(keyword: String) {
        searchApi.getSearchSuggest(keyword).subscribeOn(Schedulers.io())
            .subscribe(object : BaseObserver<SearchSuggestEntity>() {
                override fun onFailure(code: String?, message: String?) {
//                    AdLogger.logE("SearchViewModel --> getSearchSuggest --> failed to get search recommendations",SearchSuggestFragment.TAG)
                    suggestLiveData.postValue(keyword)
                }

                override fun onSuccess(data: SearchSuggestEntity?) {
                    data?.let {
                        suggestMap[it.keyword] = it
                        suggestLiveData.postValue(keyword)

                        // 当用户输入超过2字符的query无联想词结果时候也给我报一下吧
                        if (it.items?.isEmpty() == true && (it.keyword?.length
                                ?: 0) >= 2 && !suggestFailureMap.containsKey(
                                it.keyword
                            )
                        ) {
                            suggestFailureMap[it.keyword] = null
                            log("no associative word results when the user enters query longer than 2 characters")
                            // 在这里上报埋点
                            val map = HashMap<String, String>()
                            map["key_word"] = it.keyword ?: ""
                            map["ops"] = it.ops ?: ""
                            Report.reportShow(
                                SearchSuggestFragment.PAGE_NAME_SEARCH_SUGGEST_FRAGMENT,
                                "browse",
                                map
                            )
                        }
                    }
                }
            })
    }


    override fun onCleared() {
        super.onCleared()
        if (mFollowDisposable?.isDisposed == false) {
            mFollowDisposable?.dispose()
        }
    }

}