package com.transsion.share.share

import android.annotation.SuppressLint
import android.app.Activity
import android.app.Dialog
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.res.Configuration
import android.os.Build
import android.os.Bundle
import android.os.SystemClock
import android.text.TextUtils
import android.view.Gravity
import android.view.View
import android.view.WindowManager
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.ViewModelProvider
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.blankj.utilcode.util.PermissionUtils
import com.blankj.utilcode.util.ScreenUtils
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.tn.lib.logger.MBLogger
import com.tn.lib.net.env.Host
import com.tn.lib.util.networkinfo.NetworkUtil
import com.transsion.base.report.athena.LogViewConfig
import com.transsion.base.report.athena.Report
import com.transsion.share.R
import com.transsion.share.ShareInit
import com.transsion.share.api.ActionViewModel
import com.transsion.share.bean.PostType
import com.transsion.share.bean.ShareBean
import com.transsion.share.bean.ShareData
import com.transsion.share.bean.ShareSource
import com.transsion.share.bean.ShareType
import com.transsion.share.share.ShareUtils.TELEGRAM_PACKAGE_NAME
import com.transsion.share.share.ShareUtils.WHATSAPP_PACKAGE_NAME

/**
 * 分享
 */
class ShareDialogFragment : DialogFragment(R.layout.player_share_dialog) {

    private var actionViewModel: ActionViewModel? = null
    private var shareList: RecyclerView? = null
    private var shareDataStartTime = 0L
    private var isDelete: Boolean = false
    private var isDownload: Boolean = false
    private var isCheckOut: Boolean = false
    private var postType: PostType? = null
    private var authorId: String? = null
    private var reportType: String? = null
    private var ops: String? = null
    private var downloadName: String? = null

    private var shareItemCallback: ShareItemCallback? = null

    private val logViewConfig: LogViewConfig? by lazy { LogViewConfig(pageName = "share_panel") }

    companion object {
        const val TAG = "ShareDialogFragment"
        const val SUBJECT_ID = "subjectId"
        const val POST_TYPE = "postType"
        const val VIDEO_POSITION = "video_position"
        const val SUBJECT_DELETE = "subject_delete"
        const val SUBJECT_DOWNLOAD = "subject_download"
        const val SUBJECT_CHECKOUT = "subject_checkout"
        const val SOURCE = "source"
        const val AUTHOR_ID = "author_id"


        const val SHARE_TITLE = "share_title"
        const val SHARE_SCORE = "share_score"
        const val REPORT_TYPE = "report_type"
        const val OPS = "ops"
        const val DOWNLOAD_NAME = "download_name"


        const val PERMISSION_CODES_DOWNLOAD = 11

        /**
         * PostType : 分享类型
         * id: 作品id/圈子id/帖子id
         * authorId: 作者uid
         * isDownload 作品下载场景
         * isDelete  作品删除场景
         * isCheckOut 圈子退出场景
         * shareTitle 分享标题 兜底无网络时分享 帖子标题、圈子名称、作品名称
         * shareScore 分享作品评分 兜底无网时分享
         * setShareItemCallback 回调点击事件
         */
        fun newInstance(
            postType: PostType,
            id: String?,
            authorId: String?,
            reportType: String,
            shareTitle: String? = "",
            shareScore: String? = "",
            isDownload: Boolean = false,
            isDelete: Boolean = false,
            isCheckOut: Boolean = false,
            source: String,
            ops: String? = "",
            downloadName: String? = ""
        ): ShareDialogFragment {
            val fragment = ShareDialogFragment()
            val bundle = Bundle()
            bundle.putSerializable(POST_TYPE, postType)
            bundle.putString(SUBJECT_ID, id)
            bundle.putBoolean(SUBJECT_DELETE, isDelete)
            bundle.putBoolean(SUBJECT_DOWNLOAD, isDownload)
            bundle.putBoolean(SUBJECT_CHECKOUT, isCheckOut)
            bundle.putString(SHARE_TITLE, shareTitle)
            bundle.putString(SHARE_SCORE, shareScore)
            bundle.putString(SOURCE, source)
            bundle.putString(AUTHOR_ID, authorId)
            bundle.putString(REPORT_TYPE, reportType)
            bundle.putString(OPS, ops)
            bundle.putString(DOWNLOAD_NAME, downloadName)
            fragment.arguments = bundle
            return fragment
        }
    }

    private var mShareData: ShareData? = null
    private var mSubjectId: String = ""
    private var mShareTitle: String = ""
    private var mShareScore: String = ""
    private var mSource: String = ""

    fun setShareItemCallback(callback: ShareItemCallback) {
        this.shareItemCallback = callback
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setStyle(STYLE_NORMAL, R.style.shareDialogTheme)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        isDelete = arguments?.getBoolean(SUBJECT_DELETE) == true
        isDownload = arguments?.getBoolean(SUBJECT_DOWNLOAD) == true
        isCheckOut = arguments?.getBoolean(SUBJECT_CHECKOUT) == true
        mSubjectId = arguments?.getString(SUBJECT_ID) ?: ""
        postType = arguments?.getSerializable(POST_TYPE) as PostType?
        mShareTitle = arguments?.getString(SHARE_TITLE).toString()
        mShareScore = arguments?.getString(SHARE_SCORE).toString()
        mSource = arguments?.getString(SOURCE).toString()
        authorId = arguments?.getString(AUTHOR_ID)
        reportType = arguments?.getString(REPORT_TYPE)
        ops = arguments?.getString(OPS)
        downloadName = arguments?.getString(DOWNLOAD_NAME)
        initView(view)
        initViewModel()
        logViewConfig?.isLoadSuccess = true
        logViewConfig?.needPv = false
    }

    private fun initView(view: View) {
        view.apply {
            findViewById<View>(R.id.iv_close)?.setOnClickListener {
                dismiss()
            }

            shareList = findViewById(R.id.share_list)
            val shareAdapter = ShareAdapter(buildShareList())
            shareList?.let {
                it.layoutManager = LinearLayoutManager(context, RecyclerView.HORIZONTAL, false)
                it.adapter = shareAdapter
            }

            shareAdapter.setOnItemClickListener(object : OnItemClickListener {
                @SuppressLint("MissingPermission")
                override fun onItemClickListener(shareData: ShareData) {
                    mShareData = shareData
                    if (shareData.shareType == ShareType.DELETE) {
                        if (!NetworkUtil.hasCapabilityAsync()) {
                            ShareInit.toast(getString(R.string.player_no_network_tip2))
                            return
                        }
                        shareItemCallback?.delete(mSubjectId, postType)
                        click("delete")
                        dismiss()
                    } else if (shareData.shareType == ShareType.DOWNLOAD) {
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                            getVideoShareData(shareData)
                        } else {
                            if (checkPermission(
                                    ShareTools.STORAGE_PERMISSIONS
                                )
                            ) {
                                getVideoShareData(shareData)
                            } else {
                                requestPermissions(
                                    ShareTools.STORAGE_PERMISSIONS,
                                    PERMISSION_CODES_DOWNLOAD
                                )
                            }
                        }
                    } else if (shareData.shareType == ShareType.CHECKOUT) {
                        shareItemCallback?.checkOut(mSubjectId)
                        click("checkout")
                        dismiss()
                    } else if (shareData.shareType == ShareType.REPORT) {
                        click("report")
                        ShareInit.reportShow(requireContext(), reportType, mSubjectId, shareItemCallback)
                        dismiss()
                    } else {
                        getVideoShareData(shareData)
                    }
                }
            })
        }
    }


    private fun buildShareList(): List<ShareData> {
        val shareList = mutableListOf<ShareData>()
        // 复制链接
        shareList.add(
            ShareData(
                ShareType.WHATSAPP,
                R.mipmap.share_ic_whatsapp,
                getString(R.string.player_whatsapp)
            )
        )
        shareList.add(
            ShareData(
                ShareType.TELEGRAM,
                R.mipmap.share_ic_telegram,
                getString(R.string.player_telegram)
            )
        )
        shareList.add(
            ShareData(
                ShareType.COPY_LINK,
                R.mipmap.share_ic_copy_link,
                getString(R.string.share_copy_link)
            )
        )
        //下载后移
        if (isDownload) {
            val name = if (downloadName.isNullOrEmpty()) {
                getString(R.string.player_download)
            } else {
                downloadName ?: getString(R.string.player_download)
            }
            shareList.add(
                ShareData(
                    ShareType.DOWNLOAD,
                    R.mipmap.share_ic_download,
                    name
                )
            )
        }
        if (isDelete) {
            shareList.add(
                ShareData(
                    ShareType.DELETE,
                    R.mipmap.share_ic_delete,
                    getString(R.string.player_delete)
                )
            )
        }
        if (isCheckOut) {
            shareList.add(
                ShareData(
                    ShareType.CHECKOUT,
                    R.mipmap.share_ic_checkout,
                    getString(R.string.player_check_out)
                )
            )
        }
        if (ShareInit.getUserId() != authorId && !reportType.isNullOrEmpty()) { // 不是自己可以展示举报按钮
            shareList.add(
                ShareData(
                    ShareType.REPORT,
                    R.mipmap.share_ic_report,
                    getString(R.string.player_report)
                )
            )
        }
        return shareList
    }

    private fun initViewModel() {
        actionViewModel = ViewModelProvider(this)[ActionViewModel::class.java].apply {
            getShareLiveData().observe(viewLifecycleOwner) {
                if (it == null) {
                    handleShareData(mShareData?.shareType?.let { it1 -> disconnectedMode(it1) })
                    return@observe
                }
                handleShareData(it)
            }
        }
    }

    override fun onCreateDialog(savedInstanceState: Bundle?): Dialog {
        val ctx = requireContext()
        val dialog: Dialog = BottomSheetDialog(ctx, theme)
        dialog.window?.setWindowAnimations(R.style.share_dialog_animations)
        //dialog.setContentView(view)
        dialog.window?.let {
            it.setGravity(Gravity.BOTTOM)
            it.setDimAmount(0.5f)
            val lp = it.attributes ?: WindowManager.LayoutParams()
            lp.width = ScreenUtils.getScreenWidth()
            lp.height = WindowManager.LayoutParams.WRAP_CONTENT
            it.setBackgroundDrawable(null)
            it.attributes = lp
            //it.clearFlags(WindowManager.LayoutParams.FLAG_DIM_BEHIND)
        }
        return dialog
    }

    override fun onDestroy() {
        super.onDestroy()
    }

    @SuppressLint("MissingPermission")
    private fun getVideoShareData(shareData: ShareData?) {
        if (shareData == null) return
        if (!NetworkUtil.hasCapabilityAsync() && (shareData.shareType == ShareType.DELETE || shareData.shareType == ShareType.DOWNLOAD)) {
            ShareInit.toast(getString(R.string.player_no_network_tip2))
            return
        }
        if (!NetworkUtil.hasCapabilityAsync()) {
            handleShareData(disconnectedMode(shareData.shareType))
            return
        }
        shareDataStartTime = System.currentTimeMillis();
        mSubjectId = arguments?.getString(SUBJECT_ID) ?: ""
        var uid = ShareInit.getUserId()
        if (TextUtils.isEmpty(uid)) {
            uid = "0"
        }
        var scene = ""
        if (shareData.shareType == ShareType.TELEGRAM) {
            scene = "Telegram"
        } else if (shareData.shareType == ShareType.WHATSAPP) {
            scene = "WhatsApp"
        }
        actionViewModel?.share(
            shareData.shareType,
            buildDeepLinkPath(),
            mSubjectId,
            uid ?: "0",
            scene
        )
    }

    private fun handleShareData(shareBean: ShareBean?) {
        when (shareBean?.shareType) {
            ShareType.DELETE -> {
                shareItemCallback?.delete(mSubjectId)
                click("delete ")
                dismiss()
            }

            ShareType.DOWNLOAD -> {
                shareItemCallback?.download(
                    shareBean.downUrl,
                    shareBean.title,
                    "",
                    shareBean.coverUrl
                )
                click("download")
                dismiss()
            }

            ShareType.COPY_LINK -> {
                shareBean.shortUrl?.let { shortUrl ->
                    val clipData = ClipData.newPlainText("Simple test", shortUrl) //把clip对象放在剪贴板中
                    val clipboard =
                        context?.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
                    clipboard.setPrimaryClip(clipData)
                    dismiss()
                }
                shareItemCallback?.shareClick(mSubjectId)
                click("copylink")
            }

            ShareType.WHATSAPP -> {
                shareBean.shortUrl?.let { shortUrl ->
                    activity?.let {
                        shareShortUrl(
                            WHATSAPP_PACKAGE_NAME,
                            it,
                            shortUrl,
                            shareBean.desc,
                            shareBean.title
                        )
                    }
                }
                shareItemCallback?.shareClick(mSubjectId)
                click("whatsup")
            }

            ShareType.TELEGRAM -> {
                shareBean.shortUrl?.let { shortUrl ->
                    activity?.let {
                        shareShortUrl(
                            TELEGRAM_PACKAGE_NAME,
                            it,
                            shortUrl,
                            shareBean.desc,
                            shareBean.title
                        )
                    }
                }
                shareItemCallback?.shareClick(mSubjectId)
                click("Telegram")
            }

            ShareType.CHECKOUT -> {
                shareItemCallback?.checkOut(mSubjectId)
                click("checkout")
            }

            else -> {}
        }
    }

    private fun checkPermission(permissions: Array<String>): Boolean {
        return PermissionUtils.isGranted(*permissions)
    }

    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        if (requestCode == PERMISSION_CODES_DOWNLOAD) {
            if (ShareTools.verifyGrantResults(grantResults)) {
                if (mShareData != null) {
                    getVideoShareData(mShareData)
                }
            } else {
                showPermissionDenyDialog()
            }
        }
    }

    private fun showPermissionDenyDialog() {
        val settings = "\"" + getString(R.string.system_settings) + "\""
        // app name
        val appName = "\"" + ShareInit.getAppName() + "\""
//        TNToast.show(getString(R.string.permission_deny_down_tip, appName, settings))
        val msg = getString(R.string.permission_deny_down_tip, appName, settings)
        ShareTools.showConfirmationDialog(requireContext(), "", msg) {
            ShareTools.switchToSysSettingDetails(activity ?: return@showConfirmationDialog)
        }
    }

    private fun shareShortUrl(
        pkg: String,
        activity: Activity,
        shortUrl: String,
        desc: String?,
        title: String?
    ) {
        var shareContent = if (!TextUtils.isEmpty(desc)) {
            desc + "\n" + shortUrl
        } else {
            shortUrl
        }
        if (!TextUtils.isEmpty(title)) {
            shareContent = title + "\n" + shareContent
        }
        if (TextUtils.equals(pkg, WHATSAPP_PACKAGE_NAME)) {
            ShareUtils.shareToWhatsapp(activity, shareContent, callback = { result ->
                if (result) {
                    dismissAllowingStateLoss()
                } else {
                    ShareInit.toast(getString(R.string.tip_app_not_install))
                }
            })
        } else if (TextUtils.equals(pkg, TELEGRAM_PACKAGE_NAME)) {
            ShareUtils.shareToTelegram(activity, shareContent, callback = { result ->
                if (result) {
                    dismissAllowingStateLoss()
                } else {
                    ShareInit.toast(getString(R.string.tip_app_not_install))
                }
            })
        }
    }

    private fun disconnectedMode(shareType: ShareType): ShareBean {
        var shareBean = ShareBean()
        shareBean.shareType = shareType
        shareBean.shortUrl = buildLink()
        shareBean.desc = buildDisconnectedDesc()
        return shareBean
    }

    private fun buildDisconnectedDesc(): String {
        when (postType) {
            PostType.POST_TYPE -> {
                return context?.getString(R.string.share_long_link_post_s)!!
            }

            PostType.SUBJECT_TYPE -> {
                return context?.getString(
                    R.string.share_long_link_sub_no_duration,
                    mShareTitle,
                    mShareScore
                )!!
            }

            PostType.GROUP_TYPE -> {
                return context?.getString(R.string.share_long_link_group_s)!!
            }

            else -> {}
        }
        return ""
    }

    private fun buildDesc(shareType: ShareType): String {
        when (shareType) {
            ShareType.WHATSAPP -> {
                when (postType) {
                    PostType.POST_TYPE -> {
                        return context?.getString(R.string.share_long_link_post, mShareTitle)!!
                    }

                    PostType.SUBJECT_TYPE -> {
                        return context?.getString(
                            R.string.share_long_link_subject,
                            mShareTitle,
                            mShareScore
                        )!!
                    }

                    PostType.GROUP_TYPE -> {
                        return context?.getString(R.string.share_long_link_group, mShareTitle)!!
                    }

                    else -> {}
                }
            }

            ShareType.TELEGRAM -> {
                when (postType) {
                    PostType.POST_TYPE -> {
                        return context?.getString(R.string.share_long_link_post_tg)!!
                    }

                    PostType.SUBJECT_TYPE -> {
                        return context?.getString(
                            R.string.share_long_link_subject_tg,
                            mShareTitle,
                            mShareScore
                        )!!
                    }

                    PostType.GROUP_TYPE -> {
                        return context?.getString(R.string.share_long_link_group_tg)!!
                    }

                    else -> {}
                }
            }

            else -> {}
        }
        return ""
    }

    private fun buildLink(): String {
        var uid = ShareInit.getUserId()
        if (TextUtils.isEmpty(uid)) {
            uid = "0"
        }
        val url =
            Host.getShareHost() + "?type=" + buildDeepLinkPath() + "&package_name=${context?.packageName}" + "&id=" + mSubjectId + "&uid=" + uid
        MBLogger.d(TAG, "share long link:$url")
        return url
    }

    private fun buildDeepLinkPath(): String {
        when (postType) {
            PostType.POST_TYPE -> {
                return "/post/detail"
            }

            PostType.SUBJECT_TYPE -> {
                return "/movie/detail"
            }

            PostType.GROUP_TYPE -> {
                return "/room/detail"
            }

            PostType.STAFF_TYPE -> {
                return "/movie/staff"
            }

            PostType.SHORT_TV_TYPE -> {
                return "/shorts/detail"
            }

            PostType.LIVE_TYPE -> {
                return "/live/detail"
            }

            PostType.UGC_VIDEO -> {
                return "/ugc_video/detail"
            }

            PostType.UGC_SHORT_TV -> {
                return "/ugc_shorts/detail"
            }

            else -> {}
        }
        return ""
    }

    private fun click(action: String, failMsg: String? = "") {
        var map = HashMap<String, String>()
        map.put("module_name", action)
        failMsg?.let { map.put("fail_msg", it) }
        map.put(getSourceKey(), mSubjectId)
        map.put("ops", ops ?: "")
        Report.reportClick("share_panel", map)
    }

    private fun getSourceKey(): String {
        when (mSource) {
            ShareSource.POST_DETAIL -> {
                return "post_id"
            }

            ShareSource.SUBJECT_DETAIL -> {
                return "subject_id"
            }

            ShareSource.GROUP_DETAIL -> {
                return "group_id"
            }

            ShareSource.USER_CENTER -> {
                return "host_uid"
            }

            ShareSource.UGC_VIDEO_DETAIL -> {
                return "ugc_video_id"
            }

            else -> {
                return "post_id"
            }
        }
    }

    override fun onResume() {
        super.onResume()
        logViewConfig?.let {
            it.resumeTime = SystemClock.elapsedRealtime()
            if (it.needPv) {
                Report.reportPV(it)
            }
        }
    }

    override fun onPause() {
        super.onPause()
        logViewConfig?.let {
            if (it.needTerminal) {
                Report.reportPT(it)
            }
        }
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        dismiss()
    }

    fun showDialog(context: Context?, tag: String?) {
        if (context is FragmentActivity
            && !context.isFinishing
            && !context.isDestroyed
        ) {
            try {
                showDialog(context.supportFragmentManager, tag)
            } catch (e: IllegalStateException) {
                e.printStackTrace()
            } catch (e: java.lang.Exception) {
                e.printStackTrace()
            }
        }
    }

    fun showDialog(fragment: Fragment, tag: String?) {
        try {
            show(fragment.childFragmentManager, tag)
        } catch (e: IllegalStateException) {
            e.printStackTrace()
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
        }
    }

    private fun showDialog(fragmentManager: FragmentManager, tag: String?) {
        try {
            fragmentManager.executePendingTransactions()
            if (this.isAdded || fragmentManager.findFragmentByTag(tag) != null) {
                return
            }
            show(fragmentManager, tag)
        } catch (e: IllegalStateException) {
            e.printStackTrace()
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
        }
    }
}