package com.transsion.subtitle_download.task

import android.text.TextUtils
import com.blankj.utilcode.util.Utils
import com.tn.lib.net.bean.BaseDto
import com.tn.lib.net.env.Host
import com.tn.lib.net.manager.NetServiceGenerator
import com.tn.lib.util.networkinfo.NetworkUtil
import com.transsion.subtitle_download.SubtitleDownloadManager
import com.transsion.subtitle_download.api.SubtitleDownloadApi
import com.transsion.subtitle_download.bean.SubResStreamType
import com.transsion.subtitle_download.bean.SubtitleAppType
import com.transsion.subtitle_download.bean.SubtitleItem
import com.transsion.subtitle_download.bean.SubtitleListBean
import com.transsion.subtitle_download.bean.SubtitleType
import com.transsion.subtitle_download.bean.TransformBeanType
import com.transsion.subtitle_download.db.SubtitleDownloadDatabase
import com.transsion.subtitle_download.db.SubtitleDownloadTable
import com.transsion.subtitle_download.utils.Logger
import com.transsion.subtitle_download.utils.ObserveNetworkState
import com.transsion.subtitle_download.utils.SubtitleMMKV
import com.transsion.subtitle_download.utils.SubtitleUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * @author: zhangxinbing
 * @date : 2024/5/17 14:07
 * @description: 通过资源获取站内字幕列表
 */
internal object GetInStationSubtitleListTask {

    private fun getClassTag(): String = javaClass.simpleName

    private val api by lazy {
        NetServiceGenerator.instance.getService(SubtitleDownloadApi::class.java)
    }


    // =============================================================================================


    /**
     * 下载内置资源字幕
     */
    fun download(
        subjectId: String?,
        resourceId: String?,
        shortTvMiniId: String?,
        episode: Int = 0,//epse组合，用于下载字幕获取
        ep: Int = 0,//集
        se: Int = 0,//季
        videoResourceId: String?,
        subjectName: String?,
        @SubResStreamType resStreamType: Int,
        resultCallback: ((hasSubtitles: Boolean) -> Unit)? = null
    ) {
        CoroutineScope(Dispatchers.IO).launch {
            runCatching {
                // 请求字幕列表 -- 每次都请求接口是因为需要判断字幕是否有更新
                val subtitles =
                    when (resStreamType) {
                        SubResStreamType.RES_TYPE_DASH -> {
                            api.getStreamSubtitles(Host.getHost(), subjectId, videoResourceId)
                        }

                        SubResStreamType.RES_TYPE_SHORT_TV -> {
                            when (SubtitleDownloadManager.getAppType()) {
                                SubtitleAppType.NOVEL -> api.getNovelShortTVSubtitles(Host.getHost(), shortTvMiniId)
                                SubtitleAppType.MB -> api.getMbShortTVSubtitles(Host.getHost(), shortTvMiniId)
                            }
                        }

                        else -> {
                            api.getSubtitles(Host.getHost(), subjectId, videoResourceId, episode)
                        }
                    }
                Logger.logD("${getClassTag()} --> download() --> 请求接口获取字幕列表并进行处理 --> subjectId = $subjectId -- resourceId = $resourceId -- ep = $ep,resStreamType:$resStreamType, shortTvMiniId:$shortTvMiniId  --> subtitles = $subtitles")

                // 处理获取到的字幕列表
                disposeResponse(
                    subtitles,
                    resultCallback,
                    resourceId,
                    subjectId,
                    subjectName,
                    resStreamType,
                    ep,
                    se
                )
            }.getOrElse {
                // 通过resourceID请求字幕列表接口失败上报
                Logger.logE("${getClassTag()} --> download() --> 请求接口获取字幕列表并进行处理 --> subjectId = $subjectId -- resourceId = $resourceId -- ep = $ep resStreamType:$resStreamType, shortTvMiniId:$shortTvMiniId - --> 请求接口发生异常 --> Throwable.it = $it")
                withContext(Dispatchers.Main) {
                    resultCallback?.invoke(false)
                }
            }
        }
    }


    /**
     * 下载指定字幕(包括搜索字幕)
     * 如果是搜索的话，必须要指定字幕类型
     */
    fun download(dbBean: SubtitleDownloadTable) {
        CoroutineScope(Dispatchers.IO).launch {

            // 先从数据库获取有没有这个字幕的信息
            var subtitleTableBean =
                SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                    .getSubtitleTableBean(dbBean.id, dbBean.resourceStreamType)

            // 如果不存在需要先存入数据库，再进行下载
            if (null == subtitleTableBean) {
                SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                    .insert(dbBean)
                subtitleTableBean = dbBean
                //Logger.logD("${getClassTag()} --> download()222 --> 字幕下载 --> ${subtitleTableBean.url} --> 插入数据库")
            } else {
                subtitleTableBean.isSetImmediately = dbBean.isSetImmediately
            }

            // 创建一个下载任务
            SubtitleDownloadTaskManager.addTaskAndExecute(subtitleTableBean, isPrivilege = true)
        }
    }


    // =============================================================================================


    /**
     * 内置字幕接口请求成功之后解析数据
     */
    private suspend fun disposeResponse(
        subtitles: BaseDto<SubtitleListBean>,
        resultCallback: ((hasSubtitles: Boolean) -> Unit)?,
        resourceId: String?,
        subjectId: String?,
        subjectName: String?,
        @SubResStreamType resStreamType: Int,
        ep: Int = 0,//集
        se: Int = 0,//季
    ) {

        // 需要将列表进行排序，将匹配当前系统语言的放到列表的第一个位置
        val extSubtitleList = subtitles.data?.extSubtitle
        sort(extSubtitleList)

        extSubtitleList?.let { subtitleItemList ->
            // 当前资源没有对应的字幕资源
            if (subtitleItemList.isEmpty()) {
                withContext(Dispatchers.Main) {
                    resultCallback?.invoke(false)
                }
                Logger.logW("${getClassTag()} --> disposeResponse() --> subtitleItemList.isEmpty() == true -- 当前资源没有对应的字幕资源 --> resourceId = $resourceId -- subjectId = $subjectId")
                return
            }
            withContext(Dispatchers.Main) {
                resultCallback?.invoke(true)
            }
            // 获取到了字幕列表，对比本地数据库，增量更新
            val subtitleDbList =
                SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                    .getSubtitleList(resourceId ?: "")

            // 当前次触发是否有新任务创建
            var hasTaskCreate = false

            // 数据库还没有对应的数据
            if (subtitleDbList?.isEmpty() == true) {
                Logger.logD("${getClassTag()} --> disposeResponse() --> 数据库还没有对应的数据 --> 全部保存 --> 下载对应系统语言的字幕 -- subtitleItemList.size = ${subtitleItemList.size} -- getSystemLanguage = ${SubtitleUtils.getSystemLanguage()}")

                var isDownloadEn = true

                subtitleItemList.forEach { subtitleItem ->

                    // 如果当前列表里面有匹配系统的语言字幕，前置已经做了排序了
                    if (TextUtils.equals(
                            subtitleItem.lanAbbr?.uppercase(),
                            SubtitleUtils.getSystemLanguage().uppercase()
                        )
                    ) {
                        isDownloadEn = false
                    }

                    val saveAdnDownload = saveAdnDownload(
                        subjectId,
                        resourceId,
                        subjectName,
                        resStreamType,
                        ep,
                        se,
                        subtitleItem,
                        isDownloadEn = isDownloadEn
                    )
                    if (saveAdnDownload) {
                        hasTaskCreate = true
                    }
                }
            } else {
                // 数据库有数据了，增量更新
                Logger.logD("${getClassTag()} --> disposeResponse() --> 获取到了字幕列表，对比本地数据库，增量更新 --> 数据库有数据了，增量更新")
                // 删除无效，字幕更新id变更，需要先删除再下载（但是这里先删除再下载，可能会有时机问题）
                subtitleDbList?.let { list ->
                    for (dbItem in list) {
                        if (dbItem.resourceStreamType != resStreamType) {
                            //不是相同的媒体类型，不做处理-流媒体下载区分开
                            continue
                        }
                        if (dbItem.isDownloaded && dbItem.isInner.not()) {
                            //非内置的不能删除
                            continue
                        }
                        if (isExistedDb(dbItem.id, subtitleItemList).not() && NetworkUtil.hasCapabilityAsync()) {
                            // 删除资源
                            SubtitleUtils.deleteFile(dbItem.path)
                            // 同时从数据库删除
                            SubtitleDownloadDatabase.getInstance(Utils.getApp())
                                .subtitleDownloadDao().delete(dbItem)
                            Logger.logD("${getClassTag()} --> disposeResponse() --> delete(dbItem) --> 删除无效字幕数据&资源文件成功 --> it = $dbItem")
                        }
                    }
                }
                // 增量更新
                var isDownloadEn = true

                subtitleItemList.forEach { subtitleItem ->

                    // 如果当前列表里面有匹配系统的语言字幕，前置已经做了排序了
                    if (TextUtils.equals(
                            subtitleItem.lanAbbr?.uppercase(),
                            SubtitleUtils.getSystemLanguage().uppercase()
                        )
                    ) {
                        isDownloadEn = false
                    }

                    if (isExisted(subtitleItem.id, resStreamType, subtitleDbList).not()) {
                        val saveAdnDownload = saveAdnDownload(
                            subjectId,
                            resourceId,
                            subjectName,
                            resStreamType,
                            ep,
                            se,
                            subtitleItem,
                            isDownloadEn = isDownloadEn
                        )
                        if (saveAdnDownload) {
                            hasTaskCreate = true
                        }
                    }
                }
            }

            if (hasTaskCreate.not()) {
                Logger.logD("${getClassTag()} --> disposeResponse() --> 本次没有生成新的字幕下载任务")
                ObserveNetworkState.retryDownload("${getClassTag()} --> disposeResponse()")
            }
        } ?: run {
            withContext(Dispatchers.Main) {
                resultCallback?.invoke(false)
            }
            Logger.logE("${getClassTag()} --> disposeResponse() --> subtitles.data?.extSubtitle == null -- 当前资源没有对应的字幕资源")
        }
    }


    /**
     * 需要将列表进行排序，将匹配当前系统语言的放到列表的第一个位置
     */
    private fun sort(extSubtitleList: MutableList<SubtitleItem>?) {
        kotlin.runCatching {
            var systemLIndex = -1
            extSubtitleList?.forEachIndexed { index, subtitleItem ->
                if (TextUtils.equals(
                        subtitleItem.lanAbbr?.uppercase(), SubtitleUtils.getSystemLanguage().uppercase()
                    )
                ) {
                    systemLIndex = index
                }
            }
            if (systemLIndex > 0) {
                extSubtitleList?.removeAt(systemLIndex)?.let {
                    extSubtitleList.add(0, it)
                }
            }
        }
    }


    /**
     * 保存字幕到数据库并且下载资源
     */
    private suspend fun saveAdnDownload(
        subjectId: String?,
        resourceId: String?,
        subjectName: String?,
        @SubResStreamType resStreamType: Int,
        ep: Int = 0,//集
        se: Int = 0,//季
        it: SubtitleItem,
        isDownloadEn: Boolean = true // 如果匹配到了系统字幕，那就不下载EN字幕了
    ): Boolean {

        // 如果数据库已经存在相关的字幕信息，那就不需要插入一天数据到数据库
        // 如果不存在，那就将网络请求的对象转换成DBBean插入数据库
        var dbBean = SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
            .getSubtitleTableBean(it.id ?: "", resStreamType)

        if (null == dbBean) {
            dbBean = TransformBeanType.transformSubtitle2DbBean(
                subjectId, resourceId, subjectName, resStreamType, ep, se, it
            )
            // 标记一下当前是什么类型的，因为请求的时候使用的不同的请求接口
            dbBean.type = SubtitleType.TYPE_INNER
            // 保存到数据库存
            SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                .insert(dbBean)
            SubtitleDownloadManager.notifySaveDownload(dbBean)
        }

        //上次选中的语言
        val lastSelectLan = SubtitleMMKV.mmkv.getString(SubtitleMMKV.K_SELECT_LAN, null)
        if (!lastSelectLan.isNullOrEmpty() && lastSelectLan.uppercase() == dbBean.lanAbbr?.uppercase()) {
            SubtitleDownloadTaskManager.addTaskAndExecute(dbBean)
            return true
        }

        // 找到跟随系统的那条字幕进行下载
        return if (TextUtils.equals(
                dbBean.lanAbbr?.uppercase(), SubtitleUtils.getSystemLanguage().uppercase()
            )
        ) {
            SubtitleDownloadTaskManager.addTaskAndExecute(dbBean)
            true
        } else if (TextUtils.equals(
                dbBean.lanAbbr?.uppercase(), SubtitleUtils.getLanguageEn().uppercase()
            ) && isDownloadEn
        ) {
            SubtitleDownloadTaskManager.addTaskAndExecute(dbBean)
            true
        } else {
            // Logger.logW("${getClassTag()} --> download() --> saveAdnDownload() --> 找到跟随系统的那条字幕进行下载 --> 与系统语言不匹配也不是EN")
            false
        }
    }

    /**
     * 通过字幕ID判断是否存在
     */
    private fun isExistedDb(stId: String?, subtitleItemList: MutableList<SubtitleItem>): Boolean {
        var isExisted = false
        subtitleItemList.forEach {
            if (TextUtils.equals(stId, it.id)) {
                isExisted = true
            }
        }
        return isExisted
    }

    private fun isExisted(
        stId: String?,
        resStreamType: Int,
        subtitleDownloadTable: MutableList<SubtitleDownloadTable>?
    ): Boolean {
        var isExisted = false
        subtitleDownloadTable?.forEach {
            if (TextUtils.equals(stId, it.id) && resStreamType == it.resourceStreamType) {
                isExisted = true
            }
        }
        return isExisted
    }
}