package com.transsion.subtitle_download.utils

import android.os.Build
import com.blankj.utilcode.util.FileIOUtils
import com.blankj.utilcode.util.Utils
import com.tn.lib.util.md5.MD5
import com.transsion.subtitle_download.SubtitleDownloadManager
import com.transsion.subtitle_download.db.SubtitleDownloadTable
import ir.ac.iust.htmlchardet.HTMLCharsetDetector
import java.io.File
import java.util.Locale
import java.util.Random


/**
 * @author: zhangxinbing
 * @date : 2024/5/13 20:01
 * @description:
 */
internal object SubtitleUtils {

    private fun getClassTag(): String = javaClass.simpleName

    /**
     * 获取系统的序言 -- 例如，"en"代表英语，"es"代表西班牙语等
     */
    fun getSystemLanguage(): String {
        return kotlin.runCatching {
            //优先使用app保存更改的语言，然后才使用手机的语言
//            val savedLanguage = RoomAppMMKV.mmkv.getString(RoomAppMMKV.KEY_SELECT_LANGUAGE_SHORT_NAME, null)
            val savedLanguage = SubtitleDownloadManager.getAppLanguageName()

            val locale = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                Utils.getApp().resources.configuration.locales.get(0)
            } else {
                Utils.getApp().resources.configuration.locale
            }
            if (locale.language != savedLanguage && savedLanguage?.isNotEmpty() == true) {
                Locale(savedLanguage, locale.country).language
            } else {
                locale.language
            }
        }.getOrElse {
            ""
        }
    }

    /**
     * 获取英语语言
     */
    fun getLanguageEn(): String = "en"

    /**
     * 如果目标文件夹不存在，则创建目标文件夹
     */
    fun createDirectoryIfNotExists(directoryPath: String) {
        val directory = File(directoryPath)
        if (!directory.exists()) {
            directory.mkdirs()
        }
    }

    /**
     * 删除文件
     */
    fun deleteFile(path: String?) {
        kotlin.runCatching {
            val file = File(path ?: "")
            val msg = if (file.exists()) {
                // 尝试删除文件
                val delete = file.delete()
                if (delete) {
                    "文件删除成功"
                } else {
                    "文件删除失败"
                }
            } else {
                "文件不存在"
            }
            Logger.logD("${getClassTag()} --> deleteFile() --> $msg --> path = $path")
        }.getOrElse {
            Logger.logE("${getClassTag()} --> deleteFile() --> 删除文件发生异常 --> it = $it")
        }
    }

    /**
     * 自定义搜索字幕的ID
     */
    fun getSearchSubtitleId(url: String): String {
        return MD5.MD5Encode(url)
    }

    /**
     * 下载完成，解析字幕文件的编码格式，后续解析字幕用，阿拉伯、中文等爬来的旧文件会出现乱码
     * 阿拉伯 windows-1256
     * 中文 GBK GB18030
     */
    fun parseFileCharsetName(subtitleBean: SubtitleDownloadTable): Boolean {
        val file = File(subtitleBean.path ?: "")
        if (file.exists().not()) {
            return false
        }
        return try {
            val bytes = FileIOUtils.readFile2BytesByStream(subtitleBean.path!!)
            val fileCharsetName = HTMLCharsetDetector.detect(bytes, true)
            Logger.logD("${getClassTag()} --> parseFileCharsetName() --> fileCharsetName = $fileCharsetName")
            if (fileCharsetName.isNotEmpty()) {
                subtitleBean.fileCharsetName = fileCharsetName
            }
            true
        } catch (e: Throwable) {
            e.printStackTrace()
            false
        }
    }

    /**
     * 判断文件是否存在
     */
    fun isFileExist(filePath: String?): Boolean {
        return filePath?.let {
            val file = File(it)
            file.exists()
        } ?: run {
            false
        }
    }

    /**
     * 获取任务ID
     */
    fun getTaskId(): String {
        val str = System.currentTimeMillis().toString() + generateRandomString(10)
        return MD5.MD5Encode(str)
    }

    fun generateRandomString(length: Int): String {
        val random = Random()
        val charSet = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
        return buildString {
            repeat(length) {
                val randomIndex = random.nextInt(charSet.length)
                append(charSet[randomIndex])
            }
        }
    }
}