package com.transsion.subtitle_download

import android.text.TextUtils
import com.transsion.subtitle_download.db.SubtitleDownloadDatabase
import com.transsion.subtitle_download.db.SubtitleDownloadTable
import com.transsion.subtitle_download.utils.Logger
import com.blankj.utilcode.util.Utils
import com.transsion.subtitle_download.bean.SubResStreamType
import com.transsion.subtitle_download.bean.SubtitleAppType
import com.transsion.subtitle_download.bean.SubtitleStatus
import com.transsion.subtitle_download.task.GetInStationSubtitleListTask
import com.transsion.subtitle_download.utils.ObserveNetworkState
import com.transsion.subtitle_download.task.SubtitleDownloadTaskReport
import com.transsion.subtitle_download.utils.SubtitleUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.util.concurrent.CopyOnWriteArrayList

/**
 * @author: zhangxinbing
 * @date : 2024/5/13 15:19
 * @description: 字幕下载管理类
 */
object SubtitleDownloadManager {

    private fun getClassTag(): String = javaClass.simpleName
    private var apptype = SubtitleAppType.MB


    /**
     * 需要注意 -- 当前回调都是在子线程中触发的
     */
    private val mListeners = CopyOnWriteArrayList<SubtitleDownloadListener>()


    // ============================== 下面是Public API ==============================================


    /**
     * Application 初始化的时候必须要调用
     */
    fun init(appType: SubtitleAppType) {
        this.apptype = appType
        Logger.logE("${getClassTag()} --> init() --> 字幕下载工具初始化")
        ObserveNetworkState.setNetworkStateListener()
    }

    /**
     * 添加全局的监听器
     */
    fun addDownloadListener(listener: SubtitleDownloadListener) {
        mListeners.add(listener)
    }

    /**
     * 移除全局的监听器
     */
    fun removeDownloadListener(listener: SubtitleDownloadListener) {
        mListeners.remove(listener)
    }

    /**
     * 更新字幕下载进度
     */
    internal fun notifyDownloadProgress(progress: Int, stDownloadTable: SubtitleDownloadTable) {
        mListeners.forEach {
            it.onDownloading(progress, stDownloadTable)
        }
    }

    /**
     * 正在解压
     */
    internal fun notifyUnGzZip(stDownloadTable: SubtitleDownloadTable) {
        mListeners.forEach {
            it.onUnGzZip(stDownloadTable)
        }
    }

    /**
     * 字幕下载成功
     */
    internal fun notifyDownloadComplete(stDownloadTable: SubtitleDownloadTable) {
        // 字幕下载成功
        SubtitleDownloadTaskReport.reportDownloadSuccess(stDownloadTable)

        mListeners.forEach {
            it.onComplete(stDownloadTable)
        }
    }

    internal fun notifySaveDownload(bean: SubtitleDownloadTable) {
        mListeners.forEach {
            it.onSaveDownload(bean)
        }
    }

    /**
     * 字幕下载失败通知
     */
    internal fun notifyDownloadFail(e: Exception, dbBean: SubtitleDownloadTable) {
        mListeners.forEach {
            it.onFail(e, dbBean)
        }
    }

    /**
     * 删除数据
     */
    suspend fun deleteSubtitle(resourceId: String) {
        withContext(Dispatchers.IO) {
            Logger.logE("${getClassTag()} --> deleteSubtitle() --> deleteByResourceId(resourceId) --> resourceId = $resourceId")
            SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                .deleteByResourceId(resourceId)
        }
    }

    /**
     * 获取数据
     */
    suspend fun getSubtitleList(resourceId: String): MutableList<SubtitleDownloadTable>? {
        return withContext(Dispatchers.IO) {

            // 这里需要校验一下源文件是否还存在，不存在就需要修改当前状态为未下载
            val subtitleList =
                SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                    .getSubtitleList(resourceId)

            // 校验文件是否存在，不存在的话需要修改状态为未下载
            subtitleList?.forEach {
                if (SubtitleUtils.isFileExist(it.path).not()) {
                    it.status = SubtitleStatus.STATUS_NONE
                }
            }

            // 返回处理好的数据
            subtitleList
        }
    }


    // ============================ 下面是下载任务 ====================================================


    /**
     * 下载内置资源字幕（原有API，保持兼容）
     *  resourceId:外面拼接，作为一个资源字幕的标识
     *  videoResourceId：接口下发的资源id，用于请求接口
     */
    fun download(
        subjectId: String?,
        resourceId: String?,
        videoResourceId: String?,
        shortTvMiniId: String?,
        subjectName: String?,
        @SubResStreamType resStreamType: Int,
        episode: Int = 0,//epse组合，用于下载字幕获取
        ep: Int = 0,//集
        se: Int = 0,//季
        resultCallback: ((hasSubtitles: Boolean) -> Unit)? = null
    ) {
        GetInStationSubtitleListTask.download(
            subjectId,
            resourceId,
            shortTvMiniId,
            episode,
            ep,
            se,
            videoResourceId,
            subjectName,
            resStreamType,
            resultCallback
        )
    }

    /**
     * 下载UGC视频字幕（新增API）
     * @param subjectId 作品ID
     * @param resourceId 字幕资源ID（外面拼接，作为一个资源字幕的标识）
     * @param videoResourceId 接口下发的资源id，用于请求接口（必填，不为空）
     * @param ugcVideoId UGC视频ID（必填）
     * @param collectionId UGC合集ID（可选）
     * @param idType ID类型：ResourceId/StreamId/VideoId（必填）
     * @param resultCallback 结果回调
     */
    fun downloadUGCVideo(
        subjectId: String?,
        resourceId: String?,
        videoResourceId: String,//必填，不为空
        ugcVideoId: String?,//必填
        collectionId: String? = null,//UGC合集ID
        idType: String,//ID类型：ResourceId/StreamId/VideoId（必填）
        resultCallback: ((hasSubtitles: Boolean) -> Unit)? = null
    ) {
        GetInStationSubtitleListTask.downloadUGCVideo(
            subjectId,
            resourceId,
            videoResourceId,
            ugcVideoId,
            collectionId,
            idType,
            resultCallback
        )
    }

    /**
     * 下载指定字幕(包括搜索字幕)
     * 如果是搜索的话，必须要指定字幕类型
     */
    fun download(dbBean: SubtitleDownloadTable) {
        GetInStationSubtitleListTask.download(dbBean)
    }


    // =========================== 下面是为了版本兼容提供的API =========================================


    /**
     * 兼容老版本已经下载的字幕，迁移的对象 path、resourceId、id 字段是必须要设置的
     */
    suspend fun migration(dbBean: SubtitleDownloadTable) {

        if (TextUtils.isEmpty(dbBean.path) || TextUtils.isEmpty(dbBean.resourceId) || TextUtils.isEmpty(
                dbBean.id
            )
        ) {
            Logger.logE("${getClassTag()} --> migration() --> 缺少必要字段 --> path = ${dbBean.path} -- resourceId = ${dbBean.resourceId} -- id = ${dbBean.id}")
            return
        }

        withContext(Dispatchers.IO) {
            // 先从数据库获取有没有这个字幕的信息
            val subtitleTableBean =
                SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                    .getSubtitleTableBean(dbBean.id, dbBean.resourceStreamType)

            // 如果不存在需要先存入数据库，再进行下载
            if (null == subtitleTableBean) {
                SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                    .insert(dbBean)
                Logger.logD("${getClassTag()} --> migration() --> 字幕迁移 --> 插入数据库 -- success")
            } else {
                Logger.logD("${getClassTag()} --> migration() --> 字幕迁移 --> 已经存在 --> 不需要插入数据库")
            }
        }
    }

    private var appLanguageNameRequest: (() -> String?)? = null
    fun initAppLanguageName(request: (() -> String?)? = null) {
        appLanguageNameRequest = request
    }

    fun getAppLanguageName(): String? {
        return appLanguageNameRequest?.invoke()
    }

    fun getAppType() = apptype

}