package com.transsion.subtitle_download.db

import android.content.Context
import androidx.room.Database
import androidx.room.Room
import androidx.room.RoomDatabase
import androidx.room.migration.Migration
import androidx.sqlite.db.SupportSQLiteDatabase
import com.transsion.subtitle_download.utils.Logger

/**
 * @author: zhangxinbing
 * @date : 2024/5/13 17:17
 * @description: 字幕数据库入口
 */
@Database(
    entities = [SubtitleDownloadTable::class], version = 3, exportSchema = true
)
abstract class SubtitleDownloadDatabase : RoomDatabase() {

    abstract fun subtitleDownloadDao(): SubtitleDownloadDao


    // =============================================================================================


    companion object {

        const val SUBTITLE_DATABASE_NAME = "oneroom-subtitle-db"

        @Volatile
        private var instance: SubtitleDownloadDatabase? = null

        /**
         * 数据库只能当前Module里面使用
         */
        internal fun getInstance(context: Context): SubtitleDownloadDatabase {
            return instance ?: synchronized(this) {
                instance ?: buildDatabase(context).also { instance = it }
            }
        }

        // Create and pre-populate the database. See this article for more details:
        // https://medium.com/google-developers/7-pro-tips-for-room-fbadea4bfbd1#4785
        private fun buildDatabase(context: Context): SubtitleDownloadDatabase {
            return Room.databaseBuilder(
                context.applicationContext,
                SubtitleDownloadDatabase::class.java,
                SUBTITLE_DATABASE_NAME
            )
                //是否允许在主线程进行查询
                .allowMainThreadQueries().addMigrations(MIGRATION_1_2, MIGRATION_2_3)
                .addCallback(object : Callback() {
                    override fun onCreate(db: SupportSQLiteDatabase) {
                        super.onCreate(db)
                        Logger.logE("SubtitleDownloadDataBase --> getInstance() --> buildDatabase() --> onCreate() --> db_name is = " + db.path)
                    }

                    override fun onOpen(db: SupportSQLiteDatabase) {
                        super.onOpen(db)
                        Logger.logE("SubtitleDownloadDataBase --> getInstance() --> buildDatabase() --> onOpen()")
                    }
                }).build()
        }


        /**
         *         MIGRATION
         * 如果有多个MIGRATION则往后添加：
        .addMigrations(MIGRATION_1_2, MIGRATION_2_3, MIGRATION_3_4, MIGRATION_4_5, MIGRATION_5_6, MIGRATION_6_7)
         */
        private val MIGRATION_1_2 = object : Migration(1, 2) {
            override fun migrate(database: SupportSQLiteDatabase) {
                database.execSQL("ALTER TABLE ST_DOWNLOAD_TABLE ADD COLUMN failCount INTEGER NOT NULL DEFAULT 0")
            }
        }

        /**
         * 数据库版本 2 -> 3
         * 添加 UGC 视频相关字段：ugcVideoId 和 ugcVideoCollectionId
         */
        private val MIGRATION_2_3 = object : Migration(2, 3) {
            override fun migrate(database: SupportSQLiteDatabase) {
                // 添加 ugcVideoId 字段
                database.execSQL("ALTER TABLE ST_DOWNLOAD_TABLE ADD COLUMN ugcVideoId TEXT")
                // 添加 ugcVideoCollectionId 字段
                database.execSQL("ALTER TABLE ST_DOWNLOAD_TABLE ADD COLUMN ugcVideoCollectionId TEXT")
            }
        }
    }

}