package com.transsion.subtitle_download.db

import android.os.Parcelable
import androidx.annotation.Keep
import androidx.room.Entity
import androidx.room.Ignore
import androidx.room.PrimaryKey
import com.transsion.subtitle_download.bean.SubResStreamType
import com.transsion.subtitle_download.bean.SubtitleStatus
import com.transsion.subtitle_download.bean.SubtitleStatus.Companion.STATUS_NONE
import com.transsion.subtitle_download.bean.SubtitleType
import kotlinx.parcelize.IgnoredOnParcel
import kotlinx.parcelize.Parcelize
import java.io.Serializable

/**
 * @author: zhangxinbing
 * @date : 2024/5/13 16:51
 * @description: 字幕下载 -- 字幕表
 */
@Parcelize
@Keep
@Entity(tableName = "st_download_table", primaryKeys = ["id", "resourceStreamType"])
open class SubtitleDownloadTable(
    var id: String, // 主键ID -- 网络请求获取到的字幕ID
    @get:SubResStreamType @SubResStreamType var resourceStreamType: Int = SubResStreamType.RES_TYPE_DOWNLOAD, // 资源流类型-dash流媒体还是下载

    //外部组合subjectId+se+ep+资源语言+资源类型（资源语言即现在的英语以及后续的法语等，资源类型:下载、流媒体两者区分开）
    val resourceId: String, // 字幕与可播放资源唯一关联ID

    // 下面是字幕接口数据
    // var stId: String? = null,//字幕资源id，该资源唯一标识
    var lan: String? = null,//字幕语言缩写，跟手机里获取的语音缩写一一对应
    var lanName: String? = null,//字幕语言翻译后的名称
    var url: String? = null,//下载链接
    var size: Long? = null,//资源大小
    var delay: Long? = null,//字幕校准时长
    var downloads: Int? = null,//下载数
    @get:SubtitleType @SubtitleType var type: Int = SubtitleType.TYPE_INNER,//来源？ 对应SubtitleBean type

    // 下面是资源信息
    var postId: String? = null, // 帖子id
    var subtitleName: String? = null, // 字幕名称，下载搜索/本地导入 用这个
    var subjectId: String? = null,//作品id
    var ep: Int = 0, // 多资源的集数
    var se: Int = 0, // 多资源的季数
    var resolution: Int = 0, // 视频的分辨率，多分辨率会有该字段
    var subjectName: String? = null, // 作品名称

    // 字幕状态
    @get:SubtitleStatus @SubtitleStatus var status: Int = STATUS_NONE,

    //下载字幕的文件格式
    var fileCharsetName: String? = Charsets.UTF_8.name(),

    // 本地存储路径
    var path: String? = "", var zipPath: String = "",

    // 任务ID
    var taskId: String? = "",

    // 记录失败次数
    var failCount: Int = 0,

    var ugcVideoId: String? = null,
    var ugcVideoCollectionId: String? = null,

) : Parcelable, Serializable {


    @IgnoredOnParcel
    @Ignore
    var isSetImmediately = false//当前下载是否立即设置
    @IgnoredOnParcel
    @Ignore
    var isAutoDownload = true//是否是自动下载的

    val isInner: Boolean get() = type == SubtitleType.TYPE_INNER

    val isDownloaded: Boolean get() = status == SubtitleStatus.STATUS_COMPLETED

    val isSearch: Boolean
        get() = type == SubtitleType.TYPE_SEARCH_INNER || type == SubtitleType.TYPE_SEARCH_OPEN_SUBTITLES
                || type == SubtitleType.TYPE_SEARCH_OPEN_SUBTITLES_NEW

    val isOpenSbNewApi: Boolean get() = type == SubtitleType.TYPE_SEARCH_OPEN_SUBTITLES_NEW

    /** 作品名-用于文件展示的名字，移除特殊字符，空格替换成下划线*/
    val subjectFileName: String
        get() = ((subjectName?.replace(Regex("[\\\\!/:#*?\"<>|&,]"), "") ?: resourceId).replace(
            " ", "_"
        ))

    /** 外部展示的名字*/
    val name: String
        get() = if (type == SubtitleType.TYPE_INNER) lanName ?: "" else subtitleName ?: ""

    val lanAbbr: String?
        get() = if (lan == "in_id") "in" else lan

    /**
     * 获取当前字幕的信息
     */
    fun getSubtitleInfo(): String {
        val fileType = "${url?.substringBefore("?")?.substringAfterLast(".")}"
        return if (ep > 0) {
            "${subjectFileName}_S${se}_E${ep}_${lanName}.${fileType}"
        } else {
            "${subjectFileName}_${lanName}.${fileType}"
        }
    }
}