package com.transsion.subtitle_download.task

import android.os.Environment
import com.blankj.utilcode.util.Utils
import com.tn.lib.net.manager.NetServiceGenerator
import com.transsion.subtitle_download.SubtitleDownloadManager
import com.transsion.subtitle_download.bean.SubtitleStatus
import com.transsion.subtitle_download.bean.SubtitleType
import com.transsion.subtitle_download.db.SubtitleDownloadDatabase
import com.transsion.subtitle_download.db.SubtitleDownloadTable
import com.transsion.subtitle_download.utils.Logger
import com.transsion.subtitle_download.utils.SubtitleUtils
import okhttp3.Call
import okhttp3.Callback
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.util.concurrent.TimeUnit

/**
 * @author: zhangxinbing
 * @date : 2024/5/17 11:25
 * @description: 下载任务 -- 通用方法提供
 */
abstract class BaseSubtitleDownloadTask {

    companion object {

        /**
         * 字幕下载目录
         */
        val SUBTITLE_DOWNLOAD_FILE_PATH = Utils.getApp()
            .getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS)?.absolutePath + File.separatorChar + "subtitle"

        /**
         * 零时压缩包目录
         */
        val SUBTITLE_DOWNLOAD_ZIP_FILE_PATH = Utils.getApp()
            .getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS)?.absolutePath + File.separatorChar + "subtitle_zip"

        /**
         * 这个可以使用网络库提供的客户端
         */
        private val client by lazy {
            OkHttpClient.Builder().connectTimeout(30, TimeUnit.SECONDS)
                .writeTimeout(30, TimeUnit.SECONDS).readTimeout(30, TimeUnit.SECONDS)
                //.retryOnConnectionFailure(true)
                .cache(null) // 关闭请求缓存后，每次发送请求都会直接从服务器获取最新的数据，而不会使用之前缓存的数据。
                .build()
        }

    }


    // =============================================================================================


    fun getClassTag(): String = javaClass.simpleName


    // =============================================================================================


    /**
     * 字幕下载 -- 所有的字幕下载都走这里 -- 这里是通用的 -- 通过OkHttp请求接口
     */
    fun download(dbBean: SubtitleDownloadTable, callback: () -> Unit) {
        Logger.logD("${getClassTag()} --> download() --> 字幕下载 --> 开始执行下载任务了 --> getSubtitleInfo = ${dbBean.getSubtitleInfo()}")
        Logger.logD("开始执行下载任务了 --> url = ${dbBean.url}")

        // 下载任务埋点
        // SubtitleDownloadTaskReport.reportDownloadTrigger(dbBean)

        // 创建一个GET方式的请求结构
        val request: Request = Request.Builder().url(dbBean.url ?: "").build()
        client.newCall(request).enqueue(object : Callback {
            override fun onFailure(call: Call, e: IOException) {
                // 失败--中断流程
                disposeOnFailure(
                    dbBean, RuntimeException(e.message)
                )
                // 流程中断 -- 任务下载失败，继续下一个
                callback.invoke()
            }

            override fun onResponse(call: Call, response: Response) {
                if (response.isSuccessful) {
                    // 解析响应
                    disposeOnResponse(response, dbBean)
                    // 流程中断 -- 任务下载完成，继续下一个
                    callback.invoke()
                } else if (response.code != 404 && response.code in 400..599) {
                    //404不存在，就不用重试了，其他4xx5xx则请求接口获取新下载链接，可能过期了
                    disposeOnChangeUrl(response, dbBean, callback)
                } else {
                    // 失败--中断流程
                    disposeOnFailure(
                        dbBean, RuntimeException(response.message)
                    )
                    // 流程中断 -- 任务下载失败，继续下一个
                    callback.invoke()
                }
            }
        })
    }

    /**
     * 处理4xx，5xx错误更改下载链接再下载
     */
    private fun disposeOnChangeUrl(
        response: Response, dbBean: SubtitleDownloadTable, callback: () -> Unit
    ) {
        //todo 需要增加后端接口请求新链接，这里还是按失败处理，后续再调整逻辑

        // 失败--中断流程
        disposeOnFailure(
            dbBean, RuntimeException(response.message)
        )
        // 流程中断 -- 任务下载失败，继续下一个
        callback.invoke()
    }


    /**
     * 接口请求成功响应处理 主要是区分站外字幕下载的是压缩包，需要先解压缩才能使用
     */
    abstract fun disposeOnResponse(response: Response, dbBean: SubtitleDownloadTable)


    /**
     * 处理下载失败响应 统一处理
     * 1. 日志输出失败信息
     * 2. 保存数据库
     * 3. 回调下载结果
     */
    fun disposeOnFailure(dbBean: SubtitleDownloadTable, e: Exception) {
        // 加载失败了需要记录失败的次数,从新存入数据库
        dbBean.failCount += 1

        Logger.logE(
            "${getClassTag()} --> disposeOnFailure() --> e = ${e.message} -- getSubtitleInfo = ${dbBean.getSubtitleInfo()} --> dbBean.failCount = ${dbBean.failCount}"
        )

        // 请求失败状态保存数据库
        dbBean.status = SubtitleStatus.STATUS_ERROR
        // 如果是搜索字幕，下载失败就从数据库移除
        // 新增判断条件 --> 如果失败次数大于5，那就将当前记录删除，失败这么多次了也不会下载成功了
        if (dbBean.type == SubtitleType.TYPE_SEARCH_INNER || dbBean.type == SubtitleType.TYPE_SEARCH_OPEN_SUBTITLES
            || dbBean.type == SubtitleType.TYPE_SEARCH_OPEN_SUBTITLES_NEW || (dbBean.failCount > 5)
        ) {
            Logger.logE("${getClassTag()} --> disposeOnFailure() --> delete(dbBean) --> 如果是搜索字幕，下载失败就从数据库移除 --> getSubtitleInfo = ${dbBean.getSubtitleInfo()} --> failCount = ${dbBean.failCount}")
            SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                .delete(dbBean)
        } else {
            SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                .update(dbBean)
        }

        // 回调结果
        SubtitleDownloadManager.notifyDownloadFail(e, dbBean)
    }


    /**
     * 保存数据到文件 --
     * 1. 仅仅处理下载任务
     * 2. 回调下载进度
     * 3. 回调下载结果
     */
    open fun disposeOutputStream(
        response: Response,
        destination: String,
        length: Long,
        dbBean: SubtitleDownloadTable,
        progressCallback: (progress: Int) -> Unit,
        completeCallback: () -> Unit
    ): Boolean {

        // 如果文件长度是小于等于0 就是失败的
        if (length <= 0 && dbBean.isOpenSbNewApi.not()) {
            disposeOnFailure(
                dbBean,
                RuntimeException("${getClassTag()} --> disposeOutputStream --> length <= 0 --> length = $length")
            )
            return false
        }


        // 创建目录
        SubtitleUtils.createDirectoryIfNotExists(SUBTITLE_DOWNLOAD_FILE_PATH)
        SubtitleUtils.createDirectoryIfNotExists(SUBTITLE_DOWNLOAD_ZIP_FILE_PATH)

        return try {
            response.body?.byteStream().use { byteStream ->
                FileOutputStream(destination).use { fos ->
                    val buf = ByteArray(100 * 1024)
                    var sum = 0
                    var len = 0
                    dbBean.status = SubtitleStatus.STATUS_LOADING
                    while ((byteStream?.read(buf)?.also { len = it }) != -1) {
                        fos.write(buf, 0, len)
                        sum += len
                        val progress = (sum * 1.0f / length * 100).toInt()
                        // 回调进度
                        progressCallback.invoke(progress)
                        if (progress >= 100) {
                            // 回调下载完成
                            completeCallback.invoke()
                        }
                    }
                }
            }
            true
        } catch (e: Exception) {
            // 失败--中断流程
            disposeOnFailure(
                dbBean,
                RuntimeException("${getClassTag()} --> disposeOutputStream --> 字幕下载出问题了,e = ${e.message}")
            )
            false
        }
    }

}