package com.transsion.subtitle_download.task

import com.blankj.utilcode.util.Utils
import com.transsion.subtitle_download.SubtitleDownloadManager
import com.transsion.subtitle_download.bean.SubtitleStatus
import com.transsion.subtitle_download.db.SubtitleDownloadDatabase
import com.transsion.subtitle_download.db.SubtitleDownloadTable
import com.transsion.subtitle_download.utils.Logger
import com.transsion.subtitle_download.utils.SubtitleUtils
import okhttp3.Response
import java.io.File
import java.io.FileOutputStream
import java.io.InputStream
import java.util.regex.Pattern

/**
 * @author: zhangxinbing
 * @date : 2024/5/17 11:49
 * @description:opensubtitle 新api搜索
 */
internal class SearchOenSubtitleNewApiDownloadTask : BaseSubtitleDownloadTask() {

    /**
     * 站内资源 处理下载成功响应
     */
    override fun disposeOnResponse(response: Response, dbBean: SubtitleDownloadTable) {
        // 请求成功 解析数据 -- 当前不在主线程
        //val mediaType = response.body?.contentType().toString()
        val length = response.body?.contentLength() ?: 0
        if (null == response.body) {
            Logger.logE("${getClassTag()} --> disposeOnResponse() --> null == response.body --> headers = ${response.headers}")
        }

        //https://www.opensubtitles.com/download/xxx/subfile/squid.game.s01e02.WEB.h264-RBB.en.srt
        val fileType = "${dbBean.url?.substringBefore("?")?.substringAfterLast(".")}"
        val filename = getFileNameFromResponse(response) ?: ("${dbBean.name}.${fileType}")
        val filePath = if (dbBean.ep > 0) {
            SUBTITLE_DOWNLOAD_FILE_PATH + File.separatorChar + "${dbBean.subjectFileName}_S${dbBean.se}_E${dbBean.ep}"
        } else {
            SUBTITLE_DOWNLOAD_FILE_PATH + File.separatorChar + dbBean.subjectFileName
        }
        SubtitleUtils.createDirectoryIfNotExists(filePath)
        val destination = filePath + File.separatorChar + filename
        Logger.logD("${getClassTag()} --> disposeOnResponse() --> 接口请求成功 开始保存数据到本 --> ${dbBean.name}.${dbBean.resourceId},文件大小 = $length --> destination = $destination")


        // 下面从返回的输入流中读取字节数据并保存为本地文件
        disposeOutputStream(response, destination, length, dbBean, progressCallback = {
            // 下载进度回调
            SubtitleDownloadManager.notifyDownloadProgress(it, dbBean)
        }, completeCallback = {
            // 下载成功之后保存数据库
            dbBean.path = destination
            //下载完成解析文件格式
            SubtitleUtils.parseFileCharsetName(dbBean)
            dbBean.status = SubtitleStatus.STATUS_COMPLETED
            SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                .update(dbBean)
            Logger.logD("${getClassTag()} --> disposeOnResponse() --> 文件保存在 = $destination -- resourceId = ${dbBean.resourceId} --> getSubtitleInfo = ${dbBean.getSubtitleInfo()}")
            // 下载完成
            SubtitleDownloadManager.notifyDownloadComplete(dbBean)
        })
    }

    override fun disposeOutputStream(
        response: Response,
        destination: String,
        length: Long,
        dbBean: SubtitleDownloadTable,
        progressCallback: (progress: Int) -> Unit,
        completeCallback: () -> Unit
    ): Boolean {
        // 创建目录
        SubtitleUtils.createDirectoryIfNotExists(SUBTITLE_DOWNLOAD_FILE_PATH)
        SubtitleUtils.createDirectoryIfNotExists(SUBTITLE_DOWNLOAD_ZIP_FILE_PATH)

        return try {
            response.body?.byteStream()?.use { inputStream ->
                FileOutputStream(destination).use { fos ->
                    val buffer = ByteArray(8 * 1024)
                    var bytesRead: Int
                    while (inputStream.read(buffer).also { bytesRead = it } != -1) {
                        fos.write(buffer, 0, bytesRead)
                    }
                    completeCallback()
                    true
                }
            } ?: false
        } catch (e: Throwable) {
            false
        }
    }

    private fun getFileNameFromResponse(response: Response): String? {
        val contentDisposition = response.header("Content-Disposition")
        if (contentDisposition != null) {
            val pattern = Pattern.compile("filename=\"?([^\";]*)\"?")
            val matcher = pattern.matcher(contentDisposition)
            if (matcher.find()) {
                return matcher.group(1)
            }
        }
        return null // 如果没有匹配的文件名，返回null
    }
}