package com.transsion.subtitle_download.task

import android.net.Uri
import com.blankj.utilcode.util.Utils
import com.transsion.subtitle_download.SubtitleDownloadManager
import com.transsion.subtitle_download.bean.SubtitleStatus
import com.transsion.subtitle_download.db.SubtitleDownloadDatabase
import com.transsion.subtitle_download.db.SubtitleDownloadTable
import com.transsion.subtitle_download.utils.GzipUti
import com.transsion.subtitle_download.utils.Logger
import com.transsion.subtitle_download.utils.SubtitleUtils
import okhttp3.Response
import java.io.File

/**
 * @author: zhangxinbing
 * @date : 2024/5/17 11:14
 * @description: 站外搜索下载
 */
internal class SearchOpenSubtitleDownloadTask : BaseSubtitleDownloadTask() {

    /**
     * 处理站外搜索响应
     */
    override fun disposeOnResponse(response: Response, dbBean: SubtitleDownloadTable) {
        if (response.code in 200..299) {
            val contentDis = response.headers["Content-Disposition"]
            val list = contentDis?.split("filename=")
            Logger.logD("${getClassTag()} --> disposeOnResponse() --> 请求成功 --> response.code = ${response.code} -- list = $list")
            //文件名不支持特殊字符
            val zipPath = if (list?.isNotEmpty() == true && list.size > 1) {
                val decodeFilename = Uri.decode(list[1]).toString()
                val regex = Regex("[\\\\/:*?\"<>|]")
                val filename = decodeFilename.replace(regex, "")
                SUBTITLE_DOWNLOAD_ZIP_FILE_PATH + File.separatorChar + filename
            } else {
                SUBTITLE_DOWNLOAD_ZIP_FILE_PATH + File.separatorChar + Uri.decode(
                    dbBean.name
                ).hashCode()
            }
            val length = response.body?.contentLength() ?: 0

            // 下面从返回的输入流中读取字节数据并保存为本地文件 -- 这个下载的是压缩包
            val disposeOutputStream =
                disposeOutputStream(response, zipPath, length, dbBean, progressCallback = {
                    // 下载进度回调
                    SubtitleDownloadManager.notifyDownloadProgress(it, dbBean)
                }, completeCallback = {
                    // 下载成功之后保存数据库
                    dbBean.zipPath = zipPath
                    SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                        .update(dbBean)
                    Logger.logD("${getClassTag()} --> disposeOnResponse() --> 压缩文件保存在 = $zipPath -- getSubtitleInfo = ${dbBean.getSubtitleInfo()}")
                })
            if (disposeOutputStream.not()) {
                // 流程中断 -- 下载压缩包失败
                return
            }

            // 解压
            val unGzZip = unGzZip(dbBean)
            if (unGzZip.not()) {
                // 流程中断 -- 解压缩失败
                return
            }

            // 解析字幕文件的编码格式
            val parseFileCharsetName = parseFileCharsetName(dbBean)
            if (parseFileCharsetName.not()) {
                // 流程中断 -- 解压缩失败
                return
            }

            Logger.logE("--------------------------------------------------------------------")
        } else {
            // 失败--中断流程
            disposeOnFailure(
                dbBean,
                RuntimeException("${getClassTag()} --> disposeOnResponse() --> 搜索字幕接口请求失败 --> response.code = ${response.code}")
            )
        }
    }


    /**
     * 压缩文件下载成功之后需要解压缩 解压缩成功之后需要保存到数据库的
     */
    private fun unGzZip(dbBean: SubtitleDownloadTable): Boolean {

        // 开始解压缩
        SubtitleDownloadManager.notifyUnGzZip(dbBean)

        var isUnZipSuccess: Boolean
        dbBean.zipPath.let {
            Logger.logD("${getClassTag()} --> unGzZip() --> 字幕压缩包下载成功 --> 开始解压缩 .... zipPath = $it")
            // 解压文件保存的地址
            val parentPath = if (dbBean.ep > 0) {
                SUBTITLE_DOWNLOAD_FILE_PATH + File.separatorChar + "${dbBean.subjectFileName}_S${dbBean.se}_E${dbBean.ep}"
            } else {
                SUBTITLE_DOWNLOAD_FILE_PATH + File.separatorChar + dbBean.subjectFileName
            }
            SubtitleUtils.createDirectoryIfNotExists(parentPath)
            val newPath = parentPath + File.separatorChar + dbBean.name

            // 解压
            val result = GzipUti.unGzip(dbBean.zipPath, newPath)
            if (result) {
                Logger.logD("${getClassTag()} --> unGzZip() --> 字幕压缩包下载成功 --> 开始解压缩 --> 解压缩成功")
                dbBean.path = newPath
                isUnZipSuccess = true
            } else {
                // 失败--中断流程
                disposeOnFailure(
                    dbBean,
                    RuntimeException("${getClassTag()} --> unGzZip() --> 字幕压缩文件解压失败")
                )
                isUnZipSuccess = false
            }

            //无论成功失败，都删除压缩包 -- 这里修改成删除压缩缓存文件夹里面的所有文件
            runCatching {
                val file = File(it)
                file.delete()
                Logger.logD("${getClassTag()} --> unGzZip() --> result = $result -- 无论成功失败，都删除压缩包 -- 这里修改成删除压缩缓存文件夹里面的所有文件")
            }
        }
        return isUnZipSuccess
    }

    /**
     * 搜索字幕 -- 解析字幕文件的编码格式
     */
    private fun parseFileCharsetName(dbBean: SubtitleDownloadTable): Boolean {
        val parseState = SubtitleUtils.parseFileCharsetName(dbBean)
        return if (parseState) {
            Logger.logD("${getClassTag()} --> parseFileCharsetName() --> 字幕压缩包下载成功 --> 开始解析字幕文件的编码格式 --> 成功 --> 保存状态到数据库")
            // 到这里 -- 压缩包下载成功 -- 解压缩成功 -- 解析字幕文件的编码格式成功 --> 保存状态到数据库
            // 下载失败保存数据库
            dbBean.status = SubtitleStatus.STATUS_COMPLETED
            SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                .update(dbBean)
            SubtitleDownloadManager.notifyDownloadComplete(dbBean)
            true
        } else {
            kotlin.runCatching {
                //无论成功失败，都删除压缩包 -- 这里修改成删除压缩缓存文件夹里面的所有文件
                val file = File(dbBean.path ?: "")
                file.delete()
            }
            // 失败--中断流程
            disposeOnFailure(
                dbBean,
                RuntimeException("${getClassTag()} --> parseFileCharsetName() --> 解析字幕文件的编码格式失败 --> 删除字幕文件")
            )
            false
        }
    }


}