package com.transsion.subtitle_download.task

import android.text.TextUtils
import android.util.Log
import androidx.annotation.WorkerThread
import com.blankj.utilcode.util.Utils
import com.transsion.subtitle_download.SubtitleDownloadManager
import com.transsion.subtitle_download.bean.SubtitleStatus
import com.transsion.subtitle_download.db.SubtitleDownloadTable
import com.transsion.subtitle_download.bean.SubtitleType
import com.transsion.subtitle_download.db.SubtitleDownloadDatabase
import com.transsion.subtitle_download.utils.Logger
import com.transsion.subtitle_download.utils.SubtitleUtils
import java.util.concurrent.LinkedBlockingDeque


/**
 * @author: zhangxinbing
 * @date : 2024/5/13 19:54
 * @description: 下载字幕的任务队列 当前类所有的方法都执行在子线程
 */
internal object SubtitleDownloadTaskManager {

    private fun getClassTag(): String = javaClass.simpleName

    /**
     * Vector是线程安全的，可以在多线程环境下安全地访问和操作。这里处理单线程任务。
     */
    //private val taskList = Vector<SubtitleDownloadTable>()
    private val taskList = LinkedBlockingDeque<SubtitleDownloadTable>()

    /**
     * 正在下载任务
     */
    @Volatile
    private var isDownloading = false


    // ================================ 下面是字幕下载任务唯一入口 =====================================


    /**
     * 添加一个字幕下载任务 ，添加进来的一定是数据库获取的对象
     * 当前方法不能在主线程中执行
     */
    @WorkerThread
    fun addTaskAndExecute(dbBean: SubtitleDownloadTable, isPrivilege: Boolean = false) {

        // 如果字幕资源已经下载完成了，那就直接跳过 否则需要重新下载
        if (SubtitleUtils.isFileExist(dbBean.path)) {
            Logger.logW("${getClassTag()} --> addTaskAndExecute() --> 开始下载字幕 --> 如果字幕资源已经下载完成了，那就直接跳过 --> next() -- getSubtitleInfo = ${dbBean.getSubtitleInfo()}")
            // 流程中断 -- 任务已完成，继续下一个
            //已有资源的直接下载完成
            dbBean.status = SubtitleStatus.STATUS_COMPLETED
            SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                .update(dbBean)
            SubtitleDownloadManager.notifyDownloadComplete(dbBean)
            return
        }

        // 任务重复判断
        var hasTask = false
        taskList.forEach {
            if (TextUtils.equals(it.id, dbBean.id)) {
                hasTask = true
            }
        }
        if (hasTask) {
            return
        }

        // 添加到任务就上报下载任务埋点
        SubtitleDownloadTaskReport.reportDownloadTrigger(dbBean)

        // 优先级处理
        if (isPrivilege) {
            //taskList.add(0, dbBean)
            taskList.addFirst(dbBean)
        } else {
            taskList.add(dbBean)
        }
        Logger.logD("${getClassTag()} --> addTaskAndExecute() --> 剩余任务数量 -- taskList.size = ${taskList.size} --> isDownloading = $isDownloading --> dbBean.failCount = ${dbBean.failCount}")
        // 触发下载任务
        downloadNext()
    }


    // ======================== 下面是具体任务下载解析过程 =================================================


    /**
     * 下载结束之后调用
     */
    private fun downloadFinal() {
        //Logger.logD("${getClassTag()} --> downloadFinal() --> 无论成功失败异常都会回调 --> taskList = ${taskList.size}")
        // TODO ////// 一个点关闭
        isDownloading = false
        downloadNext()
    }

    /**
     * 继续下载下一个任务
     */
    private fun downloadNext() {
        if (isDownloading || taskList.isEmpty()) {
            if (taskList.isEmpty()) {
                Logger.logW("${getClassTag()} --> downloadNext() --> taskList = ${taskList.size} -- 当前没有可执行任务了")
            }
            return
        }

        // TODO ////// 一个点开启
        isDownloading = true
        runCatching {
            taskList.removeFirst()?.let {
                // 区分一下是搜索字幕下载还是内置字幕下载
                when (it.type) {
                    SubtitleType.TYPE_SEARCH_OPEN_SUBTITLES -> {
                        SearchOpenSubtitleDownloadTask().download(it) {
                            downloadFinal()
                        }
                    }

                    SubtitleType.TYPE_SEARCH_OPEN_SUBTITLES_NEW -> {//新版api，可用站内task下载
                        SearchOenSubtitleNewApiDownloadTask().download(it) {
                            downloadFinal()
                        }
                    }

                    else -> {
                        InStationSubtitleDownloadTask().download(it) {
                            downloadFinal()
                        }
                    }
                }
            }
        }.getOrElse {
            Logger.logE(
                "${getClassTag()} --> downloadNext() --> 执行下载任务发生异常 -- it = ${
                    Log.getStackTraceString(
                        it
                    )
                }"
            )
            downloadFinal()
        }
    }

}