package com.transsion.subtitle_download.utils

import android.net.Network
import android.net.NetworkCapabilities
import com.blankj.utilcode.util.Utils
import com.tn.lib.util.networkinfo.NetworkUtil
import com.tn.lib.util.networkinfo.OnNetworkStatusChangedListener
import com.transsion.subtitle_download.db.SubtitleDownloadDatabase
import com.transsion.subtitle_download.task.SubtitleDownloadTaskManager
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

/**
 * @author: zhangxinbing
 * @date : 2024/5/17 10:10
 * @description: 监听网络变化 -- 重新下载失败的字幕任务
 */
internal object ObserveNetworkState : OnNetworkStatusChangedListener {

    private fun getClassTag(): String = javaClass.simpleName


    // =============================================================================================


    override fun onDisconnected() {
        // 网络断开的时候不做处理
    }

    override fun onConnected(network: Network, networkCapabilities: NetworkCapabilities) {
        // 网络重新链接的时候检查一下字幕下载
        Logger.logD("${getClassTag()} --> onConnected() --> 网络重新链接的时候检查一下字幕下载 --> 延迟5秒，优化网络抖动 .....")
        retryDownload("网络重新连接")
    }


    // =============================================================================================


    /**
     * 监听网络变化来重试字幕下载任务
     */
    fun setNetworkStateListener() {
        NetworkUtil.registerNetworkStatusChangedListener(this)

        // 冷启动的时候也需要检查一下是否有下载失败的字幕任务
        retryDownload("冷启动")
    }

    /**
     * retry
     */
    fun retryDownload(msg: String? = null) {
        CoroutineScope(Dispatchers.IO).launch {

            // 延迟5秒，优化网络抖动
            delay(5000)
            if (NetworkUtil.hasCapabilityAsync().not()) {
                return@launch
            }

            // 从数据库里面获取所有下载失败的字幕
            SubtitleDownloadDatabase.getInstance(Utils.getApp()).subtitleDownloadDao()
                .getDownloadFailureList()?.let {
                    Logger.logD("${getClassTag()} --> retryDownload() --> $msg --> 数据库获取到的失败的任务数量 it.size = ${it.size}")
                    it.forEach { table ->
                        SubtitleDownloadTaskManager.addTaskAndExecute(table)
                    }
                }
        }
    }

}