package com.transsion.sunflower

import android.annotation.SuppressLint
import android.annotation.TargetApi
import android.app.AlarmManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.os.PowerManager
import android.text.format.DateUtils
import android.util.Log
import androidx.core.app.NotificationCompat
import androidx.work.Constraints
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.PeriodicWorkRequest
import androidx.work.WorkManager
import com.tencent.mmkv.MMKV
import java.util.Calendar
import java.util.concurrent.TimeUnit


@TargetApi(33)
class FSNManager private constructor(private val sunflowerConfig: Sunflower) {
    private val ctx: Context = sunflowerConfig.ctx

    // 标记是否已经唤起
    private var hasPostInLock: Boolean = false
    val mmkv: MMKV by lazy { MMKV.mmkvWithID("sunflower_app") }
    private val notificationHelper by lazy {
        FSNotificationHelper(ctx, CHANNEL_ID, CHANNEL_NAME)
    }

    //秒
    private val delayTime: Long
        get() =
            if (sunflowerConfig.isDebug) {
                sunflowerConfig.config?.delayTime ?: 20
            } else {
                sunflowerConfig.config?.delayTime ?: (15 * 60)
            } * 1000

    fun updateConfig(config: FSNConfig?) {
        sunflowerConfig.config = config
    }

    /**
     * 广播监听acton
     * BroadcastReceiver - onChange
     * action:
     * android.intent.action.SCREEN_OFF
     * android.intent.action.SCREEN_ON
     */
    fun onActionChange(action: String?) {
        when (action) {
            Intent.ACTION_SCREEN_ON -> {
                clearNotification()
            }

            Intent.ACTION_SCREEN_OFF,
            Intent.ACTION_POWER_DISCONNECTED,
            Intent.ACTION_POWER_CONNECTED -> {
                postNotification(false, action)
            }

            Intent.ACTION_USER_PRESENT -> {
                // 用户解锁屏幕，重置唤起
                hasPostInLock = false
            }

            else -> {}
        }
    }

    private fun effectBlock(config: FSNConfig): Boolean {
        val switcher = config.enable
        val silentUserSwitcher = if (config.only4SilentUser) {
            sunflowerConfig.silentUsers?.invoke() == true
        } else {
            true
        }
        val meetMinInterval =
            (System.currentTimeMillis() / 1000 - getLastLaunchTime()) > config.effectInterval
        Log.d("qqqq", "effectBlock: ${!switcher}, ${!meetMinInterval}, ${!silentUserSwitcher}")
        // config开关：switcher == false 被拦截
        // 触发时间间隔: 小于时间间隔config.effectInterval，被拦截
        // 沉默用户配置，非沉默用户 被拦截
        return !switcher || !meetMinInterval || !silentUserSwitcher
    }

    /* 获取展示时间列表 */
    private fun getShowListTime(config: FSNConfig): MutableList<Long> {
        val currentTime = System.currentTimeMillis()
        // 展示天最小限制1天
        val dayLimit = config.dayLimit.coerceAtLeast(1)
        // 1. 读取并解析存储的时间戳列表
        val storedTimestampsStr = mmkv.getString(DAU_UP_UP_UP_SHOW_TIME_DAY, "") ?: ""
        val timestamps = if (storedTimestampsStr.isEmpty()) {
            mutableListOf()
        } else {
            storedTimestampsStr.split(',').mapNotNull { it.toLongOrNull() }.toMutableList()
        }
        // 2. 清理过期的记录：移除所有dayLimit天前的时间戳
        val sevenDaysAgo = currentTime - DateUtils.DAY_IN_MILLIS * dayLimit
        timestamps.removeAll { it < sevenDaysAgo }
        return timestamps
    }

    /* 展示频率限制拦截 */
    private fun showLimitBlock(config: FSNConfig): Boolean {
        val timestamps = getShowListTime(config)
        // 次数限制1次
        val timesLimit = config.timesLimit.coerceAtLeast(1)
        // 检查当前计数, {dayLimit}天内弹过{timesLimit}次
        // 超过显示频率
        return timestamps.size >= timesLimit
    }

    private fun getLastLaunchTime(): Long {
        return mmkv.getLong(DAU_UP_UP_UP_LAST_EFFECT_TIME, 0)
    }

    private fun updateNewestLaunchTime(config: FSNConfig) {
        mmkv.putLong(DAU_UP_UP_UP_LAST_EFFECT_TIME, System.currentTimeMillis() / 1000)
        // 记录展示时间列表
        val timestamps = getShowListTime(config)
        timestamps.add(System.currentTimeMillis())
        val timestampsStr = timestamps.joinToString(",")
        mmkv.putString(DAU_UP_UP_UP_SHOW_TIME_DAY, timestampsStr)
    }

    fun clearNotification() {
        FSNotificationHelper.cancelNotification(ctx, DAU_UP_UP_UP_NOTIFICATION_ID)
    }

    fun postNotification(immediately: Boolean, eventSource: String) {
        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.TIRAMISU) {
            Log.d("qqqq", "postNotification: block by phone system version ")
            return
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N && Build.VERSION.SDK_INT <= Build.VERSION_CODES.TIRAMISU && !immediately) {
            Log.d("qqqq", "onReceive: fsi delay:$delayTime start")
            kotlin.runCatching {
                (ctx.applicationContext.getSystemService(Context.ALARM_SERVICE) as AlarmManager).set(
                    AlarmManager.RTC_WAKEUP,
                    System.currentTimeMillis() + delayTime,
                    "tag", { sendFSI(eventSource) },
                    Handler(Looper.getMainLooper())
                )
            }
        } else {
            Log.d("qqqq", "onReceive: fsi start")
            sendFSI(eventSource)
        }
    }

    fun postFSIWork() {
        Log.d("qqqq", "postFSIWork: ")
        val constraint = Constraints.Builder()
            .build()
        WorkManager.getInstance(ctx.applicationContext).enqueueUniquePeriodicWork(
            FSIWorker.WORKER_NAME,
            ExistingPeriodicWorkPolicy.UPDATE,
            PeriodicWorkRequest.Builder(
                FSIWorker::class.java,
                if (sunflowerConfig.isDebug) 1 else 60,
                TimeUnit.MINUTES
            )
                .setConstraints(constraint)
                .build()
        )
    }

    fun sendFSI(eventSource: String) {
        // 已经被唤起
        if (hasPostInLock) {
            sunflowerConfig.postFSNFailed?.invoke("block by had post")
            return
        }
        val config = sunflowerConfig.config
        // simba配置为空时，不展示
        if (config == null) {
            sunflowerConfig.postFSNFailed?.invoke(" block by simba config null")
            return
        }
        // app在前台时，不展示
        if (sunflowerConfig.isBackground?.invoke() == false) {
            sunflowerConfig.postFSNFailed?.invoke(" block by app foreground")
            return
        }
        if (effectBlock(config)) {
            //远程关闭或者
            sunflowerConfig.postFSNFailed?.invoke(" block by remote switcher")
            return
        }
        if (showLimitBlock(config)) {
            // 超过展示频率
            sunflowerConfig.postFSNFailed?.invoke(" block by remote times limit")
            return
        }

        // 需要在配置时间段内
        if (checkBlockTimeWithConfig(config)) {
            sunflowerConfig.postFSNFailed?.invoke(" block by time limit")
            return
        }
        // 亮屏不展示
        if (blockScreenState()) {
            sunflowerConfig.postFSNFailed?.invoke(" block by remote screen state")
            return
        }
        val intent = sunflowerConfig.targetIntent
        if (intent == null) {
            sunflowerConfig.postFSNFailed?.invoke(" block by intent == null, deeplink:${config.deepLink}")
            return
        }
        val fsi = PendingIntent.getActivity(
            ctx,
            1,
            intent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        notificationHelper.apply {
            //发送之前更新时间
            updateNewestLaunchTime(config)
            sunflowerConfig.postFSNShow?.invoke(eventSource)
            postFullScreenNotification(
                ctx,
                NotificationCompat.Builder(ctx, CHANNEL_ID)
                    .setSmallIcon(sunflowerConfig.logoResId)
                    .setVisibility(NotificationCompat.VISIBILITY_SECRET)
                    .setAutoCancel(true),
                fsi,
                DAU_UP_UP_UP_NOTIFICATION_ID
            )
            hasPostInLock = true
            Log.d("qqqq", "sendFSI-$eventSource done")
        }
    }


    private fun blockScreenState(): Boolean {
        return (ctx.getSystemService(Context.POWER_SERVICE) as PowerManager).isScreenOn
    }

    /**
     * 在设置时间段内，禁用
     */
    private fun checkBlockTimeWithConfig(config: FSNConfig): Boolean {
        try {
            val curHour = Calendar.getInstance().get(Calendar.HOUR_OF_DAY)
            val banTimeScope = config.banTimeScope
            Log.d(
                "qqqq",
                "checkBlockTimeWithConfig: hour:$curHour, banTimeScope:${banTimeScope.toList()}"
            )
            return curHour in banTimeScope
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return false
    }


    companion object {
        private const val DAU_UP_UP_UP_LAST_EFFECT_TIME = "dauupupup_last_effect_time"
        private const val DAU_UP_UP_UP_SHOW_TIME_DAY = "dauupupup_show_time_day"
        private const val CHANNEL_ID = "dauupupup"
        private const val CHANNEL_NAME = "dauupupup_name"
        private const val DAU_UP_UP_UP_NOTIFICATION_ID: Int = 10001

        @SuppressLint("StaticFieldLeak")
        @Volatile
        private var INSTANCE: FSNManager? = null

        internal fun getInstance(
            config: Sunflower
        ): FSNManager {
            return INSTANCE ?: synchronized(this) {
                val instance = FSNManager(config)
                INSTANCE = instance
                instance
            }
        }

        internal fun getInstance(): FSNManager? {
            return INSTANCE
        }
    }

}