package com.tn.lib.thread

import android.app.Activity
import android.os.Build
import android.os.Looper
import android.view.ViewTreeObserver
import com.blankj.utilcode.util.ThreadUtils
import java.lang.ref.SoftReference

/**
 * Transsion MI
 * Created By Liupeng
 * On: 2023/1/5 19:45
 *
 * Desc
 */
object IdleTaskManager {
    private val tasks: MutableList<Runnable> = ArrayList()

    /**
     * add list of tasks
     *
     * @param list tasks
     * @return this
     */
    fun initTasks(list: List<Runnable>?): IdleTaskManager {
        if (list != null && list.isNotEmpty()) {
            tasks.addAll(list)
        }
        return this
    }

    /**
     * run post delay ui
     *
     * @param activity context
     * @return UIPoolManager
     */
    fun runTasks(activity: Activity?): IdleTaskManager {
        val softReference = SoftReference(activity)
        var context: Activity? = null
        if (softReference.get().also { context = it } == null) return this
        context
            ?.window
            ?.decorView
            ?.viewTreeObserver
            ?.addOnPreDrawListener(object : ViewTreeObserver.OnPreDrawListener {
                override fun onPreDraw(): Boolean {
                    if (tasks.isEmpty()) {
                        if (softReference.get() != null) {
                            softReference
                                .get()!!
                                .window
                                .decorView
                                .viewTreeObserver
                                .removeOnPreDrawListener(this)
                        }
                    }
                    val iUiTask = object : IUiTask {
                        override fun queueIdle(): Boolean {
                            if (tasks.isNotEmpty()) {
                                val task = tasks[0]
                                task.run()
                                tasks.remove(task)
                            }
                            return tasks.isNotEmpty()
                        }
                    }
                    Looper.myQueue().addIdleHandler(iUiTask)
                    return true
                }
            })
        return this
    }

    /**
     * remove all tasks when destroy
     */
    fun removeTasks() {
        tasks.clear()
    }

    /**
     * add single task for ui
     *
     * @param runnable runnable
     */
    fun run(runnable: Runnable) {
        val uiTask: IUiTask = UiTask(runnable)
        IUiTask.WILL_REMOVE_TASKS.add(uiTask)
        Looper.myQueue().addIdleHandler(uiTask)
    }

    /**
     * add single task for ui thread on idle
     *
     * @param runnable runnable
     */
    fun runOnUi(runnable: Runnable) {
        val uiTask: IUiTask = UiTask(runnable)
        IUiTask.WILL_REMOVE_TASKS.add(uiTask)
        if (Looper.getMainLooper() == Looper.myLooper()) {
            Looper.myQueue().addIdleHandler(uiTask)
        } else {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                Looper.getMainLooper().queue.addIdleHandler(uiTask)
            } else {
                ThreadUtils.runOnUiThread { Looper.myQueue().addIdleHandler(uiTask) }
            }
        }
    }

    /**
     * remove single task for ui
     */
    fun onDestroy() {
        for (willRemoveTask in IUiTask.WILL_REMOVE_TASKS) {
            Looper.myQueue().removeIdleHandler(willRemoveTask)
        }
        IUiTask.WILL_REMOVE_TASKS.clear()
    }

}