package com.transsion.version.update

import android.app.Activity
import android.os.Build
import androidx.fragment.app.FragmentActivity
import com.blankj.utilcode.util.Utils
import com.tn.lib.logger.MBLogger
import com.tn.lib.net.manager.NetServiceGenerator
import com.transsion.version.update.dialog.UpdateConfig
import com.transsion.version.update.dialog.UpdateConfig.MAX_SHOW
import com.transsion.version.update.dialog.UpdateDialog
import com.transsion.version.update.dialog.UpdateNativeDialog
import com.transsion.version.update.ps.PsUpdateUtils
import com.transsion.version.update.store.StoreUpdateStrategy
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 *   Transsion MI
 *   Created By Liupeng
 *   On 2025/7/3 11:56
 *
 *   Desc
 */
class UpdateVersionRequest {
    private val TAG: String = "update"
    var scope = CoroutineScope(Dispatchers.IO)

    val checkUpGrateApi by lazy {
        NetServiceGenerator.instance.getService(ICheckUpgrateApi::class.java)
    }

    var mRemoteVersionInfo: RemoteVersionInfo? = null

    /**
     * 开始检测
     * @param context Context
     * @param fromPage String
     */
    fun startCheckUpdate(
        activity: Activity,
        fromPage: String,
        callback: ((noNeedUpdate: Boolean?) -> Unit)? = null,
        appPath: String,
        appName: String,
        manual: Boolean,
    ) {
        //传音设备 优先走ps升级
        scope.launch(Dispatchers.IO) {
            try {
                val versionInfoBaseDto = checkUpGrateApi.checkUpdate(appPath = appPath, "")
                if ((versionInfoBaseDto?.code == "0") && (versionInfoBaseDto.data != null)) {
                    val remoteVersionInfo = versionInfoBaseDto.data
                    //确定是有更新
                    mRemoteVersionInfo = remoteVersionInfo
                    reportCheckUpdate(fromPage, remoteVersionInfo)
                    if (remoteVersionInfo?.hasUpdate == true) {
                        remoteVersionInfo.noticeConfig?.updateVersionCode?.let {
                            UpdateMMKV.mmkv.putInt(UpdateConfig.TARGET_VERSION_CODE, it)
                        }
                        val forceUpdate = remoteVersionInfo.forceUpdate
                        if (!forceUpdate && isNewUserFirstLaunch()) {
                            MBLogger.d(TAG, "新用户首次启动不展示 非强制 更新弹窗")
                            return@launch
                        }
                        MBLogger.d(TAG, "start 展示 更新弹窗")
                        //强制升级 手动检查 频次限制
                        if (forceUpdate || manual || checkShowUpdateDialog(remoteVersionInfo.noticeConfig)) {
                            scope.launch(Dispatchers.Main) {
                                callback?.invoke(false)
                                showUpdateDialog(
                                    activity,
                                    fromPage,
                                    remoteVersionInfo,
                                    callback,
                                    manual = manual
                                )
                            }
                        } else {
                            callback?.invoke(false)
                        }
                        return@launch
                    }
                }
            } catch (e: Exception) {
                e.printStackTrace()
                //请求失败无需更新
                callback?.invoke(true)
            }
            //其他情况回调无需更新
            callback?.invoke(true)
        }
    }

    private fun reportCheckUpdate(fromPage: String, remoteVersionInfo: RemoteVersionInfo?) {
        val map = mutableMapOf<String, String>()
        map["action"] = "check"
        map["check_result"] = remoteVersionInfo?.hasUpdate.toString()
        map["force_update"] = remoteVersionInfo?.forceUpdate.toString()
        map["target_version_name"] = remoteVersionInfo?.noticeConfig?.updateVersionName.toString()
        map["page_from"] = fromPage
        UpdateReport.reportEventStage(map)
    }


    fun startUpdateProcess(activity: Activity, fromPage: String, manual: Boolean) {
        mRemoteVersionInfo?.let { versionInfo ->
            showUpdateDialog(activity, fromPage, versionInfo, null, manual)
        }

    }

    private fun showUpdateDialog(
        activity: Activity,
        fromPage: String,
        remoteVersionInfo: RemoteVersionInfo,
        callback: ((Boolean?) -> Unit)?,
        manual: Boolean
    ) {
        //自有品牌掉ps弹窗，弹不起来弹自己的弹窗
        if (UpdateManager.getInstance().deviceTranssion()) {
            activity.let {
                val targetVersionName = remoteVersionInfo.noticeConfig?.updateVersionName ?: ""
                PsUpdateUtils.showUpdateDialog(
                    fromPage, manual, it,
                    targetVersionName
                ) { errorMsg ->
                    showCustomDialog(
                        activity,
                        remoteVersionInfo,
                        fromPage,
                        callback,
                        manual,
                        errorMsg
                    )
                }
            }
        } else {
            showCustomDialog(activity, remoteVersionInfo, fromPage, callback, manual)
        }
    }


    private fun showCustomDialog(
        activity: Activity,
        remoteVersionInfo: RemoteVersionInfo,
        fromPage: String,
        callback: ((Boolean?) -> Unit)?,
        manual: Boolean,
        errorMsg: String = ""
    ) {
        if (activity is FragmentActivity) {
            //如果是强制升级 弹窗不消失 只需将值送入dialog即可
            UpdateDialog.newInstance(remoteVersionInfo, manual) { needUpgrade ->
                UpdateReport.clickDialogButton(
                    fromPage,
                    needUpgrade,
                    UpdateConstants.DIALOG_TYPE_MB
                )
                if (needUpgrade) {
                    val storeStrategy = StoreUpdateStrategy()
                    storeStrategy.openAppStore(fromPage, Utils.getApp(), remoteVersionInfo)
                }
            }.apply {
                val inPictureMode =
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                        activity.isInPictureInPictureMode
                    } else {
                        false
                    }
                if (inPictureMode) {
                    return@apply
                }
                showDialog(activity, UpdateDialog::class.java.simpleName)
            }
        } else {
            UpdateNativeDialog.create(activity, remoteVersionInfo, manual) { needUpgrade ->
                UpdateReport.clickDialogButton(
                    fromPage,
                    needUpgrade,
                    UpdateConstants.DIALOG_TYPE_MB
                )
                if (needUpgrade) {
                    val storeStrategy = StoreUpdateStrategy()
                    storeStrategy.openAppStore(fromPage, Utils.getApp(), remoteVersionInfo)
                }

            }.apply {
                val inPictureMode =
                    activity.isInPictureInPictureMode
                if (inPictureMode) {
                    return@apply
                }
                showDialog(activity)
            }
        }
        UpdateReport.showDialog(fromPage, manual, UpdateConstants.DIALOG_TYPE_MB, errorMsg)


    }



    private fun checkShowUpdateDialog(config: NoticeConfig?): Boolean {
        var res = false
        //获取已展示的版本 如果与下发版本不一致 则清空已展示次数，相同判断 是否应展示
        val notifyTimes: Int = config?.notifyTimes ?: MAX_SHOW
        val notifyInterval: Long = config?.notifyInterval?.toLong() ?: 24
        val updateVersionName: String = config?.updateVersionName ?: ""


        val showedTargetAppVersionName =
            UpdateMMKV.mmkv.getString(UpdateConfig.SHOWED_TARGET_VERSION_NAME, "")
        var showedTimes = UpdateMMKV.mmkv.getInt(UpdateConfig.UPDATE_DIALOG_SHOWED_TIMES, 0)
        MBLogger.d(
            TAG,
            "checkShowUpdateDialog 1 $showedTargetAppVersionName   2 $updateVersionName  3 $showedTimes 4 $notifyTimes 5 $notifyInterval"
        )
        if (showedTargetAppVersionName != updateVersionName) {
            showedTimes = 0
            UpdateMMKV.mmkv.putInt(UpdateConfig.UPDATE_DIALOG_SHOWED_TIMES, 0)
        }
        val currentTimeMillis = System.currentTimeMillis()
        val lastTimeStamp = UpdateMMKV.mmkv.getLong(UpdateConfig.UPDATE_DIALOG_SHOW_TIMESTAMP, 0)
        val intervalCondition =
            notifyInterval * 60 * 60 * 1000 < (currentTimeMillis - lastTimeStamp)
        if (showedTimes < notifyTimes && intervalCondition) {
            res = true
        }
        MBLogger.d(
            TAG,
            "checkShowUpdateDialog $res $showedTimes $notifyTimes 间隔时长 $intervalCondition"
        )
        return res
    }


    // 新增：判断新用户首次启动
    private fun isNewUserFirstLaunch(): Boolean {
        // 首次启动标记（通过MMKV持久化，默认true）
        val isFirstLaunch = UpdateMMKV.mmkv.getBoolean("NEW_USER_FIRST_LAUNCH", true)
        // 首次检查后标记为非首次
        if (isFirstLaunch) {
            UpdateMMKV.mmkv.putBoolean("NEW_USER_FIRST_LAUNCH", false)
        }
        return isFirstLaunch
    }





}