package com.tn.lib.util.expose

import android.graphics.Rect
import android.util.Log
import android.util.SparseArray
import android.view.View
import androidx.recyclerview.widget.GridLayoutManager
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import androidx.recyclerview.widget.StaggeredGridLayoutManager

/**
 * @author xinbing.zhang
 * @date :2023/10/24 11:12
 * @description: 列表曝光工具类
 */
class RecyclerViewExposeUtil : RecyclerView.OnScrollListener() {

    interface OnItemExposeListener {

        /**
         * item 可见性回调
         * 回调此方法时 视觉上一定是可见的（无论可见多少）
         * @param visible true，逻辑上可见，即宽/高 >50%
         * @param position item在列表中的位置
         */
        fun onItemViewVisible(visible: Boolean, position: Int)

        /**
         * 获取当前列表总数量 --> 判断是否是 loadingView
         */
        fun getTotalSize(): Int
    }

    companion object {

        /**
         * 页面生命周期曝光一次
         */
        const val EXPOSURE_TYPE_ONE_TIME_ONLY = 1

        /**
         * 不可见到可见曝光一次
         */
        const val EXPOSURE_TYPE_INVISIBLE_TO_VISIBLE = 2
    }


    // =============================================================================================


    /**
     * 曝光结果回调
     */
    private var mItemOnExposeListener: OnItemExposeListener? = null

    /**
     * 可见百分比 默认 50% -- 取值范围 0～100
     * 低于10%在准确精度方面有点问题
     */
    private var mVisiblePercentage = 10

    /**
     * 曝光类型
     *     1. 页面生命会周期仅曝光一次
     *     2. 不可见到可见都曝光
     */
    private var mExposureType: Int = EXPOSURE_TYPE_INVISIBLE_TO_VISIBLE

    private var mRecyclerView: RecyclerView? = null

    /**
     * 保存已经上报过的Item
     */
    private val sparseArray = SparseArray<Long?>()

    private fun getClassTag() = javaClass.simpleName


    // ========================= 下面是列表滚动回调 ====================================================


    override fun onScrollStateChanged(recyclerView: RecyclerView, newState: Int) {
        // 关注：SCROLL_STATE_IDLE:停止滚动； SCROLL_STATE_DRAGGING: 用户慢慢拖动
        // 关注：SCROLL_STATE_SETTLING：惯性滚动
        //if (newState == RecyclerView.SCROLL_STATE_IDLE) {
        // handleCurrentVisibleItems()
        //}
    }

    override fun onScrolled(recyclerView: RecyclerView, dx: Int, dy: Int) {
        super.onScrolled(recyclerView, dx, dy)
        //包括刚进入列表时统计当前屏幕可见views
        handleCurrentVisibleItems()
    }


    // =============================================================================================


    /**
     * 设置可见百分比 -- 必须大于0
     */
    fun setVisiblePercentage(visiblePercentage: Int) {
        mVisiblePercentage = if (visiblePercentage < 5) {
            5
        } else {
            visiblePercentage
        }
    }

    /**
     * 设置曝光类型
     */
    fun setExposureType(exposureType: Int) {
        mExposureType = exposureType
    }

    /**
     * 重置数据
     */
    fun reset() {
        sparseArray.clear()
    }

    /**
     * 资源回收
     */
    fun destroy() {
        reset()
        mRecyclerView?.removeOnScrollListener(this)
        mItemOnExposeListener = null
        mRecyclerView = null
    }

    /**
     * 手动调用上报 -- 重新回调当前页面的时候上报
     */
    fun manualCallReporting() {
        handleCurrentVisibleItems()
    }

    /**
     * 设置RecyclerView的item可见状态的监听
     *
     * @param recyclerView     recyclerView
     * @param onExposeListener 列表中的item可见性的回调
     */
    fun setRecyclerItemExposeListener(
        recyclerView: RecyclerView?, onExposeListener: OnItemExposeListener?
    ) {
        mItemOnExposeListener = onExposeListener
        mRecyclerView = recyclerView
        if (mRecyclerView == null || mRecyclerView?.visibility != View.VISIBLE) {
            return
        }

        //检测recyclerView的滚动事件
        mRecyclerView?.addOnScrollListener(this)
    }


    // =============================================================================================


    /**
     * 处理 当前屏幕上mRecyclerView可见的item view
     *
     * View.getGlobalVisibleRect(new Rect())，true表示view视觉可见，无论可见多少。
     */
    private fun handleCurrentVisibleItems() {
        if (mRecyclerView == null || mRecyclerView?.visibility != View.VISIBLE || mRecyclerView?.isShown == false || mRecyclerView?.getGlobalVisibleRect(
                Rect()
            ) == false
        ) {
            return
        }

        //保险起见，为了不让统计影响正常业务，这里做下try-catch
        kotlin.runCatching {
            var range = IntArray(2)
            val manager = mRecyclerView?.layoutManager
            when (manager) {

                is GridLayoutManager -> {
                    range = findRangeGrid(manager)
                }

                is LinearLayoutManager -> {
                    range = findRangeLinear(manager)
                }

                is StaggeredGridLayoutManager -> {
                    range = findRangeStaggeredGrid(manager)
                }
            }

            //log("屏幕内可见条目的起始位置：" + range[0] + "---" + range[1])
            // 注意，这里 会处理此刻 滑动过程中 所有可见的view
            for (i in range[0]..range[1]) {
                // 这里是为了处理 BaseQuickAdapter加载更多自定义View也曝光的问题
                if (i == (mItemOnExposeListener?.getTotalSize())) {
                    // 测试过程日志 PsLogger.logE("超过了数据源 ---  i = $i -- mAdapter?.data?.size = ${mAdapter?.data?.size}")
                    continue
                }
                setCallbackForLogicVisibleView(manager?.findViewByPosition(i), i)
            }
        }.getOrElse {
            Log.e(
                getClassTag(),
                "RecyclerViewExposeUtil --> handleCurrentVisibleItems() --> " + Log.getStackTraceString(
                    it
                )
            )
        }
    }

    /**
     * 为 逻辑上可见的view设置 可见性回调
     * 说明：逻辑上可见--可见且可见高度（宽度）>view高度（宽度）的50%
     *
     * @param view        可见item的view
     * @param position    可见item的position
     */
    private fun setCallbackForLogicVisibleView(view: View?, position: Int) {
        if (view == null || view.visibility != View.VISIBLE || !view.isShown || !view.getGlobalVisibleRect(
                Rect()
            )
        ) {
            return
        }
        val rect = Rect()
        val cover = view.getGlobalVisibleRect(rect)

        // 时机曝光面积
        val exposure = rect.height() * rect.width().toDouble()
        // View总面积
        val total = view.measuredHeight * view.measuredWidth.toDouble()
        val isItemViewVisibleInLogic = exposure > ((total * mVisiblePercentage) / 100)
        Log.i("exposure_log", "isItemViewVisibleInLogic = $isItemViewVisibleInLogic")

        // 符合曝光条件才回调
        if (cover && isItemViewVisibleInLogic) {
            // 已经上报过了，那就直接返回
            if (sparseArray[position] != null) {
                return
            }
            sparseArray.put(position, System.currentTimeMillis())
            mItemOnExposeListener?.onItemViewVisible(true, position)
        } else {
            mItemOnExposeListener?.onItemViewVisible(false, position)
            // 从不可见到可见都需要曝光
            if (mExposureType == EXPOSURE_TYPE_INVISIBLE_TO_VISIBLE) {
                sparseArray.remove(position)
            }
        }
    }

    /**
     * 获取线性布局中 第一个和最后一个Item Position
     */
    private fun findRangeLinear(manager: LinearLayoutManager): IntArray {
        val range = IntArray(2)
        range[0] = manager.findFirstVisibleItemPosition()
        range[1] = manager.findLastVisibleItemPosition()
        return range
    }

    /**
     * 获取网格布局中 第一个和最后一个Item Position
     */
    private fun findRangeGrid(manager: GridLayoutManager): IntArray {
        val range = IntArray(2)
        range[0] = manager.findFirstVisibleItemPosition()
        range[1] = manager.findLastVisibleItemPosition()
        return range
    }

    /**
     * 获取瀑布流布局中 第一个和最后一个Item Position
     */
    private fun findRangeStaggeredGrid(manager: StaggeredGridLayoutManager): IntArray {
        val startPos = IntArray(manager.spanCount)
        val endPos = IntArray(manager.spanCount)
        manager.findFirstVisibleItemPositions(startPos)
        manager.findLastVisibleItemPositions(endPos)
        return findRange(startPos, endPos)
    }

    private fun findRange(startPos: IntArray, endPos: IntArray): IntArray {
        var start = startPos[0]
        var end = endPos[0]
        for (i in 1 until startPos.size) {
            if (start > startPos[i]) {
                start = startPos[i]
            }
        }
        for (i in 1 until endPos.size) {
            if (end < endPos[i]) {
                end = endPos[i]
            }
        }
        return intArrayOf(start, end)
    }
}


