package com.tn.lib.util.networkinfo

import android.Manifest
import android.annotation.SuppressLint
import android.content.Context
import android.content.pm.PackageManager
import android.net.ConnectivityManager
import android.net.LinkProperties
import android.net.Network
import android.net.NetworkCapabilities
import android.net.NetworkRequest
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.telephony.TelephonyManager
import android.util.Log
import androidx.core.content.ContextCompat
import com.blankj.utilcode.util.Utils
import java.util.concurrent.CopyOnWriteArrayList

/**
 * @author xinbing.zhang
 * @date :2023/1/30 14:49
 * @description:
 * ConnectivityManager.NetworkCallback --> Added in API level 21
 * https://developer.android.com/reference/android/net/NetworkCapabilities
 * 这个是异步获取的
 * 无网络注册的时候 --> 不会回调
 * 有网络注册的时候 --> 会回调
 */
class NetWorkCallback : ConnectivityManager.NetworkCallback() {

    companion object {

        private const val TAG = "net_log"

        val mHandler by lazy { Handler(Looper.getMainLooper()) }

        val INSTANCE by lazy(LazyThreadSafetyMode.SYNCHRONIZED) {
            NetWorkCallback()
        }
    }


    // =============================================================================================


    /**
     * 网络状态回调
     */
    private val mListeners = CopyOnWriteArrayList<OnNetworkStatusChangedListener>()

    /**
     * 当前网络是否连接 默认是空值 只有收到了回调才会修改值
     */
    @Volatile
    private var isConnected: Boolean? = null

    /**
     * API 23 开始支持的 -- 当前连接的网络经过了验证的
     */
    @Volatile
    private var isValidated: Boolean? = null

    private fun getClassTag(): String = javaClass.simpleName

    /**
     * 当前网络状态
     */
    private var mNetworkType: NetworkType? = null


    // =============================下面是暴露给外部调用的API===========================================


    /**
     * 注册监听到系统
     */
    fun registerNetworkCallback() {
        val connMgr =
            Utils.getApp()?.applicationContext?.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
        connMgr?.registerNetworkCallback(NetworkRequest.Builder().build(), this)

        // 启动一个定时任务，获取实时网络状态
        startTimer()
    }

    /**
     * 从系统移除监听
     */
    fun unregisterNetworkCallback() {
        val connMgr =
            Utils.getApp()?.applicationContext?.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
        connMgr?.unregisterNetworkCallback(this)

        reset()
        mListeners.clear()
        mHandler.removeCallbacksAndMessages(null)
    }

    fun registerNetworkStatusChangedListener(listener: OnNetworkStatusChangedListener?) {
        listener?.let {
            // 避免业务多次调用
            if (!mListeners.contains(listener)) {
                mListeners.add(it)
            }
        }
    }

    fun unregisterNetworkStatusChangedListener(listener: OnNetworkStatusChangedListener?) {
        listener?.let {
            mListeners.remove(it)
        }
    }

    /**
     * 获取当前的网络状态
     * 0 -- 当前无网络
     * 1 -- 网络已连接
     * 2 -- 网络已连接&已验证
     * 3 -- 未知
     */
    fun getNetState(): Int {
        return when {
            isConnected() && isValidated() -> 2
            isConnected() -> 1
            isConnected().not() -> 0
            else -> {
                3
            }
        }
    }

    /**
     * 返回当前的网络连接状态
     *
     * 异步 + 同步
     */
    fun isConnected(): Boolean {
        // 使用缓存的 --> 如果此值是空的意味着网络状态回调还没有回来 --> 通过同步的方法获取网络状态。
        if (null != isConnected) {
            if (isConnected == false){
                return NetworkUtil.isNetworkConnected(Utils.getApp())
            }
            return isConnected == true
        }
        // 同步获取
        return NetworkUtil.isNetworkConnected(Utils.getApp())
    }

    /**
     * 返回当前的网络可用状态
     */
    private fun isValidated(): Boolean {
        // 使用缓存的 --> 如果此值是空的意味着网络状态回调还没有回来 --> 通过同步的方法获取网络状态。
        if (null != isValidated) {
            return isValidated == true
        }

        // 同步获取
        return NetworkUtil.isNetworkCapability(Utils.getApp())
    }

    /**
     * 获取当前网络状态
     */
    fun getNetworkType(): NetworkType? {
        if (null != mNetworkType) {
            return mNetworkType
        }

        if (NetworkUtil.isWifiConnected(Utils.getApp())) {
            mNetworkType = NetworkType.NETWORK_WIFI
            return mNetworkType
        }

        mNetworkType = getNetTypeMobile()
        return mNetworkType
    }


    // =========================下面是ConnectivityManager.NetworkCallback()回调=======================


    override fun onLost(network: Network) {
        super.onLost(network)
        // 当网络断开连接或不再满足此请求或回调时调用。
        // 如果回调是使用 requestNetwork() 或 registerDefaultNetworkCallback() 注册的，
        // 则仅当该网络丢失且没有其他网络满足请求条件时，才会针对 onAvailable() 返回的最后一个网络调用该回调。
        // 如果回调是通过 registerNetworkCallback() 注册的，那么将为每个不再满足回调条件的网络调用该回调。
        // 不要在此回调中调用ConnectivityManager.getNetworkCapabilities(android.net.Network)
        // 或 ConnectivityManager.getLinkProperties(android.net.Network)
        // 或其他同步 ConnectivityManager 方法，因为这很容易出现竞争条件；在回调中调用这些方法可能会返回过时的对象，甚至返回 null 对象。
        //

        // 这里需要注意 --> 如果同时连接了Wi-Fi和数据流量 --> 在断开其中一个的时候这里也会收到回调。
        // NetWorkCallback --> onLost --> dispatch --> 无网络 --> 网络已断开连接 --> 多次回调
        // 收到网络断开的回调的时候，需要同步校验一下，因为在Wi-Fi & 数据流量都接入的时候，断开其中一个网络也是会收到这个回调的。
        disposeOnLost()
    }

    override fun onCapabilitiesChanged(network: Network, networkCapabilities: NetworkCapabilities) {
        super.onCapabilitiesChanged(network, networkCapabilities)
        // 当与此请求相对应的网络更改功能但仍然满足请求的条件时调用。
        // 从Build.VERSION_CODES.O该方法开始保证在 后立即调用onAvailable(Network)。
        //
        // 不要ConnectivityManager.getLinkProperties(android.net.Network)在此回调中调用
        // 或其他同步 ConnectivityManager 方法，因为这很容易出现竞争条件：在回调中调用这些方法可能会返回过时的对象，甚至返回 null 对象。
        //
        // 网络状态的改变
        //
        // 这里会多次回调

        kotlin.runCatching {
            // 处理网络连接
            disposeConnect(network, networkCapabilities)
            // 处理网络上网能力
            disposeValidated(networkCapabilities)
            // 处理网络类型
            disposeNetType(networkCapabilities)
        }.getOrElse {
            Log.e(TAG, "${getClassTag()} --> it = $it")
        }
    }

    override fun onAvailable(network: Network) {
        super.onAvailable(network)
        // NetWorkCallback --> onAvailable ---> 有网络 --> 网络连接成功 --- 多个可用网络多次回调
    }

    override fun onLinkPropertiesChanged(network: Network, linkProperties: LinkProperties) {
        super.onLinkPropertiesChanged(network, linkProperties)
        // 当与该请求对应的网络发生变化时调用LinkProperties。
        //
        // 从Build.VERSION_CODES.O该方法开始保证在 后立即调用onAvailable(Network)。
        //
        // 不要ConnectivityManager.getNetworkCapabilities(android.net.Network)在此回调中调用或其他同步 ConnectivityManager 方法，
        // 因为这很容易出现竞争条件：在回调中调用这些方法可能会返回过时的对象，甚至返回 null 对象。
        //
        // NetWorkCallback --> onLinkPropertiesChanged --> 当网络连接属性发生变化时调用 --> 网络连接属性变化
    }

    override fun onBlockedStatusChanged(network: Network, blocked: Boolean) {
        super.onBlockedStatusChanged(network, blocked)
        // 当对指定网络的访问被阻止或解除阻止时调用。
        //
        // 不要在此回调中调用ConnectivityManager.getNetworkCapabilities(android.net.Network)
        // 或 ConnectivityManager.getLinkProperties(android.net.Network)
        // 或其他同步 ConnectivityManager 方法，因为这很容易出现竞争条件：
        // 在回调中调用这些方法可能会返回过时的对象，甚至返回 null 对象。
        //
        // NetWorkCallback --> onBlockedStatusChanged --> 当访问的网络被阻塞或者解除阻塞时调用 --> 访问的网络阻塞状态发生变化
    }

    override fun onLosing(network: Network, maxMsToLive: Int) {
        super.onLosing(network, maxMsToLive)
        // 当网络即将丢失时调用，通常是因为没有未完成的请求。这可以NetworkCallback#onAvailable与新的替代网络的呼叫配对以实现平稳切换。
        // 不保证在NetworkCallback#onLost调用之前调用此方法，例如在网络突然断开的情况下。
        //
        // 不要在此回调中调用ConnectivityManager.getNetworkCapabilities(android.net.Network)
        // 或 ConnectivityManager.getLinkProperties(android.net.Network)或其他同步 ConnectivityManager 方法，
        // 因为这很容易出现竞争条件；在回调中调用这些方法可能会返回过时的对象，甚至返回 null 对象。
        //
        // NetWorkCallback --> onLosing --> 有网络 --> 网络正在丢失连接 --> 不一定回调 --> 例如在网络突然断开的情况下。
    }

    override fun onUnavailable() {
        super.onUnavailable()
        // 如果在调用中指定的超时时间内未找到网络 ConnectivityManager.requestNetwork(android.net.NetworkRequest,
        // android.net.ConnectivityManager.NetworkCallback, int)或无法满足所请求的网络请求（无论是否指定超时），则调用。
        // 当调用此回调时，关联的 NetworkRequest将已被删除并释放，
        // 就像 ConnectivityManager.unregisterNetworkCallback(android.net.ConnectivityManager.NetworkCallback)已被调用一样。
        //
        // NetWorkCallback --> onUnavailable --> 无网络 --> 网络连接超时或网络不可达
    }


    // =============================================================================================

    /**
     * 数据重置
     */
    private fun reset() {
        isConnected = false
        isValidated = false
        mNetworkType = null
    }

    /**
     * 网络断开任务
     */
    private val lostRunnable = {
        // 延时之后会再次同步获取一下是否真的没有网络了 --> 不处理
        //                                      --> 开始分发
        val networkConnected = NetworkUtil.isNetworkConnected(Utils.getApp())
        isValidated = NetworkUtil.isNetworkCapability(Utils.getApp())
        val netState = getNetState()
        if (networkConnected) {
            isConnected = true // 恢复有网状态
            Log.i(
                TAG,
                "${getClassTag()} --> onLost() --> disposeOnLost --> lostRunnable --> networkConnected = $networkConnected netState = $netState --  网络重新连接了",
            )
            // 开始分发
            mListeners.forEach {
                // 在分发的过程中状态改变了，那就终止分发。
                if (isConnected == true) {
                    it.onConnected()
                } else {
                    return@forEach
                }
            }
        } else {
            // 在这一时刻检测到的网络状态是断开连接的。
            reset()

            Log.i(
                TAG,
                "${getClassTag()} --> onLost() --> disposeOnLost --> lostRunnable --> networkConnected = $networkConnected netState = $netState -- 网络断开了"
            )

            // 开始分发
            mListeners.forEach {
                // 在分发的过程中状态改变了，那就终止分发。
                if (isConnected == false) {
                    it.onDisconnected()
                } else {
                    return@forEach
                }
            }

        }
    }

    /**
     * 网络断开分发
     */
    private fun disposeOnLost() {
        // 这里的延迟主要是处理 --> 官方文档给出的建议
        //
        // 不要在此回调中调用ConnectivityManager.getNetworkCapabilities(android.net.Network)
        // 或 ConnectivityManager.getLinkProperties(android.net.Network)
        // 或其他同步 ConnectivityManager 方法，因为这很容易出现竞争条件；在回调中调用这些方法可能会返回过时的对象，甚至返回 null 对象。
        mHandler.removeCallbacks(lostRunnable)
        mHandler.postDelayed(lostRunnable, 1000)
        // 走了这个方法，说明网络断开了。原逻辑这里有个问题需要延迟一秒后才能改变网络状态。有个1秒的滞后，所以这里先设置无网，(测试有提bug)
        // 然后1秒后根据当时的网络状态，在来重新判断一次是否已经连接
        //isConnected = false 不可立即修改状态
    }

    /**
     * 网络重新连接分发
     */
    private fun disposeConnect(network: Network, networkCapabilities: NetworkCapabilities) {
        // hasCapability可以判断网络是否连接 --> 表示是否连接上了互联网（不关心是否可以上网）
        if (networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)) {
            // 设置当前网络连接状态，避免重复分发。
            if (isConnected == true) {
                return
            }
            isConnected = true

            Log.i(
                TAG,
                "${getClassTag()} --> onCapabilitiesChanged --> disposeConnect --> 表示是否连接上了互联网（不关心是否可以上网"
            )

            // 状态分发
            mHandler.post {
                mListeners.forEach {
                    // 再次校验如果这个时候状态被修改了那就终止分发状态
                    if (isConnected == true) {
                        it.onConnected(network, networkCapabilities)
                    } else {
                        return@forEach
                    }
                }
            }
        }
    }

    /**
     * 处理当前是否有网络能力
     */
    private fun disposeValidated(networkCapabilities: NetworkCapabilities) {
        // 表明此网络连接成功验证 --> 这个方式的验证会导致 --> 传音部分手机插入国内的卡无法识别网络状态
        // 业务上 手机提示已连接网络 --> App 判断无网络是不是也是矛盾的呢？
        // 是否具备上网的能力
        isValidated = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
        } else {
            true
        }
        Log.i(
            TAG,
            "${getClassTag()} --> disposeValidated() --> isValidated = $isValidated netState = ${getNetState()}"
        )
    }

    /**
     * 网络状态
     */
    private fun disposeNetType(networkCapabilities: NetworkCapabilities) {
        // 网络状态变化-- > 当与此请求对应的网络更改功能但仍满足请求的条件时调用。
        // hasTransport可以判断网络的类型
        val networkType =
            if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)) {
                NetworkType.NETWORK_WIFI
            } else if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)) {
                getNetTypeMobile()
            } else {
                NetworkType.NETWORK_UNKNOWN
            }
        if (mNetworkType == networkType) {
            return
        }
        mNetworkType = networkType
        Log.i(TAG, "当前网络类型是 mNetworkType = $mNetworkType")
    }

    /**
     * API24及以上获取网络类型
     */
    @SuppressLint("MissingPermission")
    private fun getNetTypeMobile(): NetworkType {
        val type: NetworkType?
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.N) {
            // 使用老版本的获取
            type = NetworkUtil.getNetworkType()
        } else {
            // 判断是否有权限
            if (ContextCompat.checkSelfPermission(
                    Utils.getApp(), Manifest.permission.READ_PHONE_STATE
                ) == PackageManager.PERMISSION_GRANTED
            ) {
                val telephonyManager =
                    Utils.getApp().getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager?
                type = when (telephonyManager?.dataNetworkType) {
                    TelephonyManager.NETWORK_TYPE_GPRS, TelephonyManager.NETWORK_TYPE_EDGE, TelephonyManager.NETWORK_TYPE_CDMA, TelephonyManager.NETWORK_TYPE_IDEN, TelephonyManager.NETWORK_TYPE_1xRTT -> NetworkType.NETWORK_2G

                    TelephonyManager.NETWORK_TYPE_EVDO_0, TelephonyManager.NETWORK_TYPE_EVDO_A, TelephonyManager.NETWORK_TYPE_EVDO_B, TelephonyManager.NETWORK_TYPE_HSDPA, TelephonyManager.NETWORK_TYPE_HSUPA, TelephonyManager.NETWORK_TYPE_HSPA, TelephonyManager.NETWORK_TYPE_HSPAP, TelephonyManager.NETWORK_TYPE_EHRPD -> NetworkType.NETWORK_3G

                    TelephonyManager.NETWORK_TYPE_LTE,
                        // 19 TelephonyManager.NETWORK_TYPE_LTE_CA,
                    19 -> NetworkType.NETWORK_4G

                    TelephonyManager.NETWORK_TYPE_NR -> NetworkType.NETWORK_5G

                    TelephonyManager.NETWORK_TYPE_UNKNOWN -> NetworkType.NETWORK_UNKNOWN
                    else -> NetworkType.NETWORK_UNKNOWN
                }
            } else {
                // 没有权限的时候使用 默认的方式获取
                type = NetworkUtil.getNetworkType()
            }
        }
        return type
    }


    /**
     * 定时任务
     */
    private val timerRunnable = {
        isValidated = NetworkUtil.isNetworkCapability(Utils.getApp())
        Log.i(TAG, "${getClassTag()} --> timerRunnable --> isValidated = $isValidated")
        startTimer()
    }

    /**
     * 开启定时任务
     */
    private fun startTimer() {
        Log.i(TAG, "${getClassTag()} --> startTimer() --> 开启定时任务")
        mHandler.postDelayed(timerRunnable, 2 * 60 * 1000)
    }

    private fun logNetWorkType(networkCapabilities: NetworkCapabilities?): String? {
        if (null == networkCapabilities) {
            return null
        }

        // 获取信号强度
        val signalStrength = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            " 信号强度 = ${networkCapabilities.signalStrength}"
        } else {
            " 当前版本不支持获取"
        }

        // 网络状态变化-- > 当与此请求对应的网络更改功能但仍满足请求的条件时调用。
        // hasTransport可以判断网络的类型
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)) {
            return "表示此网络使用蜂窝传输。 -- $signalStrength"
        }
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)) {
            return "表示此网络使用 Wi-Fi 传输。-- $signalStrength"
        }
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_BLUETOOTH)) {
            return "表示此网络使用蓝牙传输。-- $signalStrength"
        }
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET)) {
            return "表示此网络使用以太网传输。-- $signalStrength"
        }
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_VPN)) {
            return "表示此网络使用 VPN 传输。-- $signalStrength"
        }
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI_AWARE)) {
            return "表示此网络使用 Wi-Fi 感知传输。-- $signalStrength"
        }
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_LOWPAN)) {
            return "表示此网络使用 LoWPAN 传输。-- $signalStrength"
        }
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_USB)) {
            return "表示此网络使用 USB 传输。-- $signalStrength"
        }
        return null
    }

}