package com.unity3d.services.store

import com.unity3d.services.core.properties.ClientProperties
import com.unity3d.services.store.core.StoreExceptionHandler
import com.unity3d.services.store.core.StoreLifecycleListener
import com.unity3d.services.store.gpbl.StoreBilling
import com.unity3d.services.store.gpbl.listeners.BillingInitializationListener
import com.unity3d.services.store.gpbl.listeners.FeatureSupportedListener
import com.unity3d.services.store.gpbl.listeners.PurchaseHistoryResponseListener
import com.unity3d.services.store.gpbl.listeners.PurchasesResponseListener
import com.unity3d.services.store.gpbl.listeners.SkuDetailsResponseListener
import com.unity3d.services.store.gpbl.listeners.StoreEventListener
import kotlinx.coroutines.flow.MutableStateFlow

import java.lang.reflect.InvocationTargetException


class StoreMonitor(
    private val storeExceptionHandler: StoreExceptionHandler
) {
    private var storeBilling: StoreBilling? = null
    private var storeLifecycleListener: StoreLifecycleListener? = null

    private val _isInitialized = MutableStateFlow(false)
    val isInitialized: Boolean
        get() = _isInitialized.value

	@Throws(
        ClassNotFoundException::class,
        NoSuchMethodException::class,
        IllegalAccessException::class,
        InvocationTargetException::class
    )
    fun initialize(billingInitializationListener: BillingInitializationListener) {
        if (isInitialized) {
            // Already initialized
            billingInitializationListener.onIsAlreadyInitialized()
            return
        }
        storeBilling = StoreBilling(ClientProperties.getApplicationContext(), billingInitializationListener)
        storeBilling?.initialize(billingInitializationListener)
        _isInitialized.value = true
    }

	fun isFeatureSupported(operationId: Int, purchaseType: String?, featureSupportedListener: FeatureSupportedListener): Int {
        var result = -1
        try {
            result = storeBilling?.isFeatureSupported(purchaseType) ?: -1
            featureSupportedListener.onFeatureSupported(result)

        } catch (exception: Exception) {
            storeExceptionHandler.handleStoreException(
                StoreEvent.IS_FEATURE_SUPPORTED_REQUEST_ERROR,
                operationId,
                exception
            )
        }
        return result
    }

	fun getPurchases(operationId: Int, purchaseType: String?, purchasesResponseListener: PurchasesResponseListener) {
        try {
            checkNotNull(storeBilling)
            storeBilling?.getPurchases(
                purchaseType,
                purchasesResponseListener
            )
        } catch (exception: Exception) {
            storeExceptionHandler.handleStoreException(
                StoreEvent.PURCHASES_REQUEST_ERROR,
                operationId,
                exception
            )
        }
    }

	fun getPurchaseHistory(operationId: Int, purchaseType: String?, maxPurchases: Int, purchaseHistoryResponseListener: PurchaseHistoryResponseListener) {
        try {
            storeBilling?.getPurchaseHistory(
                purchaseType,
                maxPurchases, purchaseHistoryResponseListener)
        } catch (exception: Exception) {
            storeExceptionHandler.handleStoreException(
                StoreEvent.PURCHASE_HISTORY_LIST_REQUEST_ERROR,
                operationId,
                exception
            )
        }
    }

    fun getSkuDetails(operationId: Int, purchaseType: String?, skuList: List<String?>?, skuDetailsResponseListener: SkuDetailsResponseListener) {
        try {
            storeBilling?.getSkuDetails(purchaseType, skuList, skuDetailsResponseListener)
        } catch (exception: Exception) {
            storeExceptionHandler.handleStoreException(
                StoreEvent.SKU_DETAILS_LIST_REQUEST_ERROR,
                operationId,
                exception
            )
        }
    }

    fun startPurchaseTracking(purchaseTypes: ArrayList<String>, storeEventListener: StoreEventListener) {
        if (storeLifecycleListener != null) {
            stopPurchaseTracking()
        }
        storeLifecycleListener = StoreLifecycleListener(purchaseTypes, storeBilling!!, storeEventListener)
        ClientProperties.getApplication().registerActivityLifecycleCallbacks(storeLifecycleListener)
    }

	fun stopPurchaseTracking() {
        if (storeLifecycleListener != null) {
            ClientProperties.getApplication().unregisterActivityLifecycleCallbacks(
                storeLifecycleListener
            )
            storeLifecycleListener = null
        }
    }
}
