package com.unity3d.services.store.gpbl.bridges.billingclient.v6

import com.unity3d.services.store.gpbl.BillingResultResponseCode
import com.unity3d.services.store.gpbl.bridges.BillingResultBridge
import com.unity3d.services.store.gpbl.bridges.SkuDetailsBridge
import com.unity3d.services.store.gpbl.bridges.SkuDetailsParamsBridge
import com.unity3d.services.store.gpbl.bridges.billingclient.BillingClientAdapter
import com.unity3d.services.store.gpbl.listeners.BillingClientStateListener
import com.unity3d.services.store.gpbl.listeners.BillingInitializationListener
import com.unity3d.services.store.gpbl.listeners.ProductDetailsListener
import com.unity3d.services.store.gpbl.listeners.PurchasesResponseListener
import com.unity3d.services.store.gpbl.listeners.SkuDetailsResponseListener
import com.unity3d.services.store.gpbl.proxies.BillingClientStateListenerProxy
import com.unity3d.services.store.gpbl.proxies.PurchaseUpdatedListenerProxy
import com.unity3d.services.store.gpbl.proxies.PurchasesResponseListenerProxy
import com.unity3d.services.store.gpbl.proxies.SkuDetailsResponseListenerProxy

class BillingClientAdapter(
    private val billingClientBuilderBridge: BillingClientBuilderBridge,
    private val detectedVersion : Int = 6
) : BillingClientAdapter() {
    private lateinit var billingClient: BillingClientBridge

    override fun initialize(billingInitializationListener: BillingInitializationListener) {
        if (isInitialized) {
            // Already initialized (Should be revisited with UADSDK-4373) cause technically, it should be possible to re-initialize
            billingInitializationListener.onIsAlreadyInitialized()
            return
        }
        billingClientBuilderBridge.setListener(PurchaseUpdatedListenerProxy(billingInitializationListener))
        billingClientBuilderBridge.enablePendingPurchases()
        billingClient = billingClientBuilderBridge.build() as BillingClientBridge
        startConnection(billingInitializationListener)
        _isInitialized.value = true
    }

    override fun startConnection(billingClientStateListener: BillingClientStateListener?) {
        billingClient.startConnection(BillingClientStateListenerProxy(billingClientStateListener))
    }

    override fun isFeatureSupported(purchaseType: String?): BillingResultResponseCode? {
        return billingClient.isFeatureSupported(purchaseType)
    }

    override fun isReady(): Boolean {
        return billingClient.isReady
    }

    override fun queryPurchasesAsync(
        purchaseType: String?,
        purchasesResponseListener: PurchasesResponseListener?
    ) {
        billingClient.queryPurchasesAsync(purchaseType, PurchasesResponseListenerProxy(purchasesResponseListener))
    }

    override fun queryProductDetailsAsync(
        productId: String,
        productType: String,
        listener: ProductDetailsListener
    ) {
        // Bridge to SKU details as ProductDetails is not available in Billing Library v4
        val skuDetailsParamsBuilderBridge = SkuDetailsParamsBridge.newBuilder()
        val skuDetailsParamsBridge = skuDetailsParamsBuilderBridge.setSkuList(listOf(productId)).setType(productType).build()
        billingClient.querySkuDetailsAsync(skuDetailsParamsBridge, SkuDetailsResponseListenerProxy(object : SkuDetailsResponseListener {
                override fun onSkuDetailsUpdated(
                    billingResult: BillingResultBridge,
                    skuDetailsList: List<SkuDetailsBridge>?
                ) {
                    if (skuDetailsList.isNullOrEmpty()) {
                        // No products found (wrong product type, etc.)
                        listener.onProductDetailsResponse(billingResult, null)
                        return
                    } else {
                        listener.onProductDetailsResponse(
                            billingResult,
                            skuDetailsList[0].originalJson?.toString()
                        )
                    }
                }
            })
        )
    }

    override fun getAdapterVersion(): String = detectedVersion.toString()

}