package com.unity3d.ads.core.domain.scar

import com.google.protobuf.ByteString
import com.unity3d.ads.TokenConfiguration
import com.unity3d.ads.core.data.manager.ScarManager
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_COLLECTION_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_COLLECTION_STARTED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_COLLECTION_SUCCESS
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_UPLOAD_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_UPLOAD_SUCCESS
import com.unity3d.ads.core.extensions.elapsedMillis
import gatewayprotocol.v1.AdFormatOuterClass.AdFormat
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

class AndroidFetchSignalsAndSendUseCase(
    private val scope: CoroutineScope,
    private val scarManager: ScarManager,
    private val handleGetTokenRequest: HandleGetTokenRequest,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val scarEligibleEffectiveUseCase: ScarEligibleEffectiveUseCase
) : FetchSignalsAndSendUseCase {
    @OptIn(ExperimentalTime::class)
    override suspend fun invoke(tokenNumber: Int, tokenId: ByteString, tokenConfiguration: TokenConfiguration?) {
        scope.launch {
            val startTime = TimeSource.Monotonic.markNow()
            val requestedSignals = scarEligibleEffectiveUseCase(tokenConfiguration)
            if (requestedSignals.isEmpty()) { return@launch }
            sendDiagnosticEvent(SCAR_SIGNALS_COLLECTION_STARTED, tokenNumber = tokenNumber)
            val scarSignals = runCatching {
                scarManager.getSignals(requestedSignals)
            }.getOrNull()
            val signalsCollectionEvent = if (scarSignals != null) {
                SCAR_SIGNALS_COLLECTION_SUCCESS
            } else {
                SCAR_SIGNALS_COLLECTION_FAILURE
            }
            sendDiagnosticEvent(
                event = signalsCollectionEvent,
                value = startTime.elapsedMillis(),
                tags = getTags(requestedSignals),
                tokenNumber = tokenNumber
            )
            if (scarSignals == null) {
                return@launch
            }

            val uploadStartTime = TimeSource.Monotonic.markNow()
            sendDiagnosticEvent(SendDiagnosticEvent.SCAR_SIGNALS_UPLOAD_STARTED, tokenNumber = tokenNumber)

            val uploadResponse = handleGetTokenRequest(tokenId, scarSignals)
            val signalsUploadEvent = if (uploadResponse.hasError()) {
                SCAR_SIGNALS_UPLOAD_FAILURE
            } else {
                SCAR_SIGNALS_UPLOAD_SUCCESS
            }

            sendDiagnosticEvent(
                event = signalsUploadEvent,
                value = uploadStartTime.elapsedMillis(),
                tags = getTags(requestedSignals),
                tokenNumber = tokenNumber
            )
        }
    }

    private fun getTags(formats: List<AdFormat>): Map<String, String> {
        val tags = mutableMapOf<String, String>()

        if (formats.contains(AdFormat.AD_FORMAT_BANNER)) {
            tags["banner"] = "true"
        }
        if (formats.contains(AdFormat.AD_FORMAT_REWARDED)) {
            tags["rewarded"] = "true"
        }
        if (formats.contains(AdFormat.AD_FORMAT_INTERSTITIAL)) {
            tags["interstitial"] = "true"
        }

        return tags
    }
}