package com.unity3d.services.core.di

import android.content.Context
import android.os.Handler
import android.os.Looper
import androidx.datastore.core.DataMigration
import androidx.datastore.core.DataStore
import androidx.datastore.core.DataStoreFactory
import com.unity3d.ads.core.data.datasource.AndroidByteStringDataSource
import com.unity3d.ads.core.data.datasource.ByteStringDataSource
import com.unity3d.ads.core.data.model.ByteStringSerializer
import com.unity3d.ads.core.data.model.WebViewConfigurationStoreSerializer
import com.unity3d.ads.core.data.repository.DiagnosticEventRepository
import com.unity3d.ads.core.extensions.unityAdsDataStoreFile
import com.unity3d.ads.datastore.ByteStringStoreOuterClass.ByteStringStore
import com.unity3d.ads.datastore.WebviewConfigurationStore
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.ads.measurements.MeasurementsService
import com.unity3d.services.ads.token.AsyncTokenStorage
import com.unity3d.services.ads.token.InMemoryAsyncTokenStorage
import com.unity3d.services.ads.token.TokenStorage
import com.unity3d.services.ads.topics.TopicsService
import com.unity3d.services.core.device.StorageManager
import com.unity3d.services.core.device.StorageManager.StorageType
import com.unity3d.services.core.device.VolumeChange
import com.unity3d.services.core.device.VolumeChangeMonitor
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_GATEWAY_CACHE
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_GL_INFO
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_IAP_TRANSACTION
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_NATIVE_CONFIG
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_PRIVACY
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_PRIVACY_FSM
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_WEBVIEW_CONFIG
import com.unity3d.services.core.di.ServiceProvider.NAMED_GET_TOKEN_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_INIT_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_LOAD_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_OFFERWALL_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_OMID_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_SCAR_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_SHOW_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_TRANSACTION_SCOPE
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.domain.SDKDispatchers
import com.unity3d.services.core.misc.JsonStorage
import com.unity3d.services.core.properties.ClientProperties
import com.unity3d.services.core.request.metrics.SDKMetrics
import com.unity3d.services.core.request.metrics.SDKMetricsSender
import com.unity3d.services.core.webview.bridge.SharedInstances
import gatewayprotocol.v1.NativeConfigurationOuterClass.AdOperationsConfiguration
import gatewayprotocol.v1.NativeConfigurationOuterClass.NativeConfiguration
import gatewayprotocol.v1.NativeConfigurationOuterClass.RequestPolicy
import gatewayprotocol.v1.NativeConfigurationOuterClass.RequestRetryPolicy
import gatewayprotocol.v1.NativeConfigurationOuterClass.RequestTimeoutPolicy
import gatewayprotocol.v1.adOperationsConfiguration
import gatewayprotocol.v1.diagnosticEventsConfiguration
import gatewayprotocol.v1.featureFlags
import gatewayprotocol.v1.nativeConfiguration
import gatewayprotocol.v1.requestPolicy
import gatewayprotocol.v1.requestRetryPolicy
import gatewayprotocol.v1.requestTimeoutPolicy
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob

class UnityAdsModule {
    fun androidContext(): Context = ClientProperties.getApplicationContext()

    fun mainDispatcher(): CoroutineDispatcher = Dispatchers.Main

    fun defaultDispatcher(): CoroutineDispatcher = Dispatchers.Default

    fun ioDispatcher(): CoroutineDispatcher = Dispatchers.IO

    fun sdkDispatchers(): ISDKDispatchers = SDKDispatchers()

    fun sdkMetrics(): SDKMetricsSender = SDKMetrics.getInstance()

    fun initCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_INIT_SCOPE) + errorHandler)

    fun loadCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_LOAD_SCOPE) + errorHandler)

    fun showCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_SHOW_SCOPE) + errorHandler)

    fun transactionCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope =
        CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_TRANSACTION_SCOPE) + errorHandler)

    fun getTokenCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope =
        CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_GET_TOKEN_SCOPE) + errorHandler)

    fun scarSignalsCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_SCAR_SCOPE) + errorHandler)

    fun offerwallSignalsCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(
        NAMED_OFFERWALL_SCOPE) + errorHandler)

    fun omidCoroutineScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_OMID_SCOPE) + errorHandler)

    fun publicApiJob(
        diagnosticEventRepository: DiagnosticEventRepository,
    ): Job = Job().apply {
        invokeOnCompletion {
            diagnosticEventRepository.flush()
        }
    }

    fun gatewayDataStore(
        context: Context,
        dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_GATEWAY_CACHE)

    fun privacyDataStore(
        context: Context,
        dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_PRIVACY)

    fun privacyFsmDataStore(
        context: Context,
        dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_PRIVACY_FSM)

    fun nativeConfigurationDataStore(
        context: Context,
        dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_NATIVE_CONFIG)

    fun glInfoDataStore(
        context: Context, dispatcher: CoroutineDispatcher, fetchGLInfo: DataMigration<ByteStringStore>
    ): DataStore<ByteStringStore> = DataStoreFactory.create(
        serializer = ByteStringSerializer(),
        produceFile = { context.unityAdsDataStoreFile(DATA_STORE_GL_INFO) },
        migrations = listOf(fetchGLInfo),
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    fun iapTransactionDataStore(
        context: Context, dispatcher: CoroutineDispatcher
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_IAP_TRANSACTION)

    fun webViewConfigurationDataStore(
        context: Context, dispatcher: CoroutineDispatcher
    ): DataStore<WebviewConfigurationStore.WebViewConfigurationStore> = DataStoreFactory.create(
        serializer = WebViewConfigurationStoreSerializer(),
        produceFile = { context.unityAdsDataStoreFile(DATA_STORE_WEBVIEW_CONFIG) },
        corruptionHandler = null,
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    fun asyncTokenStorage(
        tokenStorage: TokenStorage,
        sdkMetricsSender: SDKMetricsSender,
    ): AsyncTokenStorage = InMemoryAsyncTokenStorage(
        null, Handler(Looper.getMainLooper()), sdkMetricsSender, tokenStorage
    )

    fun volumeChangeMonitor(
        volumeChange: VolumeChange,
    ): VolumeChangeMonitor = VolumeChangeMonitor(SharedInstances.webViewEventSender, volumeChange)

    fun publicJsonStorage(): JsonStorage = provideJsonStorage(StorageType.PUBLIC)

    fun privateJsonStorage(): JsonStorage = provideJsonStorage(StorageType.PRIVATE)

    fun memoryJsonStorage(): JsonStorage = provideJsonStorage(StorageType.MEMORY)

    fun defaultNativeConfiguration(): NativeConfiguration = nativeConfiguration {
        adOperations = getDefaultAdOperations()
        initPolicy = getDefaultRequestPolicy()
        adPolicy = getDefaultRequestPolicy()
        otherPolicy = getDefaultRequestPolicy()
        operativeEventPolicy = getDefaultRequestPolicy()
        diagnosticEvents = diagnosticEventsConfiguration {
            enabled = true
            maxBatchSize = 10
            maxBatchIntervalMs = 30000
            ttmEnabled = false
        }
        featureFlags = featureFlags {
            boldSdkNextSessionEnabled = true
        }
    }

    fun gatewayCacheDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun privacyDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun idfiDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun auidDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun privacyFsmDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun nativeConfigurationDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun glInfoDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun iapTransactionDataStore(dataStore: DataStore<ByteStringStore>) = provideByteStringDataSource(dataStore)

    fun measurementService(
        context: Context,
        dispatchers: ISDKDispatchers,
    ): MeasurementsService = MeasurementsService(context, dispatchers, SharedInstances.webViewEventSender)

    fun topicsService(
        context: Context, dispatchers: ISDKDispatchers
    ): TopicsService = TopicsService(context, dispatchers, SharedInstances.webViewEventSender)

    private fun provideJsonStorage(storageType: StorageType): JsonStorage {
        check(StorageManager.init(ClientProperties.getApplicationContext())) {
            "StorageManager failed to initialize"
        }
        return StorageManager.getStorage(storageType)
    }

    private fun provideByteStringDataSource(dataStore: DataStore<ByteStringStore>): ByteStringDataSource {
        return AndroidByteStringDataSource(dataStore)
    }

    private fun provideByteStringDataStore(
        context: Context, dispatcher: CoroutineDispatcher, dataStoreFile: String
    ): DataStore<ByteStringStore> {
        return DataStoreFactory.create(
            serializer = ByteStringSerializer(),
            produceFile = { context.unityAdsDataStoreFile(dataStoreFile) },
            corruptionHandler = null,
            scope = CoroutineScope(dispatcher + SupervisorJob())
        )
    }

    private fun getDefaultAdOperations(): AdOperationsConfiguration {
        return adOperationsConfiguration {
            loadTimeoutMs = UnityAdsConstants.AdOperations.LOAD_TIMEOUT_MS
            showTimeoutMs = UnityAdsConstants.AdOperations.SHOW_TIMEOUT_MS
            getTokenTimeoutMs = UnityAdsConstants.AdOperations.GET_TOKEN_TIMEOUT_MS
        }
    }

    private fun getDefaultRequestPolicy(): RequestPolicy {
        return requestPolicy {
            retryPolicy = getDefaultRequestRetryPolicy()
            timeoutPolicy = getDefaultRequestTimeoutPolicy()
        }
    }

    private fun getDefaultRequestRetryPolicy(): RequestRetryPolicy {
        return requestRetryPolicy {
            maxDuration = UnityAdsConstants.RequestPolicy.RETRY_MAX_DURATION
            retryWaitBase = UnityAdsConstants.RequestPolicy.RETRY_WAIT_BASE
            retryJitterPct = UnityAdsConstants.RequestPolicy.RETRY_JITTER_PCT
            shouldStoreLocally = UnityAdsConstants.RequestPolicy.SHOULD_STORE_LOCALLY
            retryMaxInterval = UnityAdsConstants.RequestPolicy.RETRY_MAX_INTERVAL
            retryScalingFactor = UnityAdsConstants.RequestPolicy.RETRY_SCALING_FACTOR
        }
    }

    private fun getDefaultRequestTimeoutPolicy(): RequestTimeoutPolicy {
        return requestTimeoutPolicy {
            connectTimeoutMs = UnityAdsConstants.RequestPolicy.CONNECT_TIMEOUT_MS
            readTimeoutMs = UnityAdsConstants.RequestPolicy.READ_TIMEOUT_MS
            writeTimeoutMs = UnityAdsConstants.RequestPolicy.WRITE_TIMEOUT_MS
            overallTimeoutMs = UnityAdsConstants.RequestPolicy.OVERALL_TIMEOUT_MS
        }
    }
}
