/*******************************************************************************
 * Copyright (c) 2000, 2009 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.aspectj.org.eclipse.jdt.internal.compiler.ast;

import org.aspectj.org.eclipse.jdt.internal.compiler.codegen.CodeStream;
import org.aspectj.org.eclipse.jdt.internal.compiler.codegen.ExceptionLabel;
import org.aspectj.org.eclipse.jdt.internal.compiler.lookup.BlockScope;
import org.aspectj.org.eclipse.jdt.internal.compiler.lookup.LocalVariableBinding;

/**
 * Extra behavior for statements which are generating subroutines
 */
public abstract class SubRoutineStatement extends Statement {

	public static void reenterAllExceptionHandlers(SubRoutineStatement[] subroutines, int max, CodeStream codeStream) {
		if (subroutines == null) return;
		if (max < 0) max = subroutines.length;
		for (int i = 0; i < max; i++) {
			SubRoutineStatement sub = subroutines[i];
			sub.enterAnyExceptionHandler(codeStream);
			sub.enterDeclaredExceptionHandlers(codeStream);
		}
	}

	ExceptionLabel anyExceptionLabel;
	protected SwitchExpression switchExpression = null;

	public ExceptionLabel enterAnyExceptionHandler(CodeStream codeStream) {

		if (this.anyExceptionLabel == null) {
			this.anyExceptionLabel = new ExceptionLabel(codeStream, null /*any exception*/);
		}
		this.anyExceptionLabel.placeStart();
		return this.anyExceptionLabel;
	}

	public void enterDeclaredExceptionHandlers(CodeStream codeStream) {
		// do nothing by default
	}

	public void exitAnyExceptionHandler() {
		if (this.anyExceptionLabel != null) {
			this.anyExceptionLabel.placeEnd();
		}
	}

	public void exitDeclaredExceptionHandlers(CodeStream codeStream) {
		// do nothing by default
	}


	/**
	 * Generate an invocation of a subroutine (e.g. jsr finally) in current context.
	 * @return boolean, <code>true</code> if the generated code will abrupt completion
	 */
	public abstract boolean generateSubRoutineInvocation(BlockScope currentScope, CodeStream codeStream, Object targetLocation, int stateIndex, LocalVariableBinding secretLocal);

	public abstract boolean isSubRoutineEscaping();

	public void placeAllAnyExceptionHandler() {
		this.anyExceptionLabel.place();
	}

	public SwitchExpression getSwitchExpression() {
		return this.switchExpression;
	}

	public void setSwitchExpression(SwitchExpression switchExpression) {
		this.switchExpression = switchExpression;
	}
}
