package xyz.erupt.core.controller;

import com.google.gson.JsonObject;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.web.bind.annotation.*;
import xyz.erupt.annotation.SceneEnum;
import xyz.erupt.annotation.fun.PowerObject;
import xyz.erupt.core.annotation.EruptRecordOperate;
import xyz.erupt.core.annotation.EruptRouter;
import xyz.erupt.core.config.GsonFactory;
import xyz.erupt.core.constant.EruptRestPath;
import xyz.erupt.core.event.EruptAddEvent;
import xyz.erupt.core.event.EruptDeleteEvent;
import xyz.erupt.core.event.EruptEditEvent;
import xyz.erupt.core.invoke.DataProcessorManager;
import xyz.erupt.core.invoke.DataProxyInvoke;
import xyz.erupt.core.naming.EruptRecordNaming;
import xyz.erupt.core.service.EruptCoreService;
import xyz.erupt.core.service.EruptModifyService;
import xyz.erupt.core.service.EruptService;
import xyz.erupt.core.service.IEruptDataService;
import xyz.erupt.core.util.EruptUtil;
import xyz.erupt.core.util.Erupts;
import xyz.erupt.core.util.ReflectUtil;
import xyz.erupt.core.view.EruptApiModel;
import xyz.erupt.core.view.EruptModel;
import xyz.erupt.core.view.R;

import javax.transaction.Transactional;
import java.util.List;

/**
 * Erupt 对数据的增删改查
 *
 * @author YuePeng
 * date 9/28/18.
 */
@Slf4j
@RestController
@RequestMapping(EruptRestPath.ERUPT_DATA_MODIFY)
@RequiredArgsConstructor
public class EruptModifyController {

    private final EruptService eruptService;

    private final EruptModifyService eruptModifyService;

    private final ApplicationEventPublisher applicationEventPublisher;

    @Transactional
    @PostMapping({"/{erupt}"})
    @EruptRecordOperate(value = "INSERT", dynamicConfig = EruptRecordNaming.class)
    @EruptRouter(skipAuthIndex = 3, authIndex = 1, verifyType = EruptRouter.VerifyType.ERUPT)
    public R<Void> addEruptData(@PathVariable("erupt") String erupt, @RequestBody JsonObject data) {
        EruptModel eruptModel = EruptCoreService.getErupt(erupt);
        Erupts.powerLegal(eruptModel, PowerObject::isAdd);
        EruptApiModel eruptApiModel = EruptUtil.validateEruptValue(eruptModel, data);
        if (eruptApiModel.getStatus() == EruptApiModel.Status.ERROR) return R.error(eruptApiModel.getMessage());
        Object obj = eruptModifyService.eruptInsertDataProcess(eruptModel, data);
        DataProxyInvoke.invoke(eruptModel, (dataProxy -> dataProxy.beforeAdd(obj)));
        DataProcessorManager.getEruptDataProcessor(eruptModel.getClazz()).addData(eruptModel, obj);
        eruptModifyService.modifyLog(eruptModel, "ADD", GsonFactory.getGson().toJson(obj));
        DataProxyInvoke.invoke(eruptModel, (dataProxy -> dataProxy.afterAdd(obj)));
        applicationEventPublisher.publishEvent(new EruptAddEvent<>(eruptModel.getClazz(), obj));
        return R.ok();
    }


    @SneakyThrows
    public void batchAddEruptData(EruptModel eruptModel, List<Object> list) {
        for (Object data : list) {
            DataProxyInvoke.invoke(eruptModel, (dataProxy -> dataProxy.beforeAdd(data)));
        }
        DataProcessorManager.getEruptDataProcessor(eruptModel.getClazz()).batchAddData(eruptModel, list);
        for (Object obj : list) {
            eruptModifyService.modifyLog(eruptModel, "ADD", GsonFactory.getGson().toJson(obj));
            DataProxyInvoke.invoke(eruptModel, (dataProxy -> dataProxy.afterAdd(obj)));
            applicationEventPublisher.publishEvent(new EruptAddEvent<>(eruptModel.getClazz(), obj));
        }
    }

    @PostMapping("/{erupt}/update")
    @EruptRecordOperate(value = "UPDATE", dynamicConfig = EruptRecordNaming.class)
    @EruptRouter(skipAuthIndex = 3, authIndex = 1, verifyType = EruptRouter.VerifyType.ERUPT)
    @Transactional
    public R<Void> updateEruptData(@PathVariable("erupt") String erupt, @RequestBody JsonObject data) {
        return this.putUpdateEruptData(erupt, data);
    }

    @PostMapping("/{erupt}/delete")
    @EruptRouter(skipAuthIndex = 3, authIndex = 1, verifyType = EruptRouter.VerifyType.ERUPT)
    @EruptRecordOperate(value = "DELETE", dynamicConfig = EruptRecordNaming.class)
    @Transactional
    public R<Void> deleteEruptData(@PathVariable("erupt") String erupt, @RequestBody String[] ids) {
        return this.deleteEruptDataList(erupt, ids);
    }


    @Deprecated
    @PutMapping("/{erupt}")
    @EruptRecordOperate(value = "UPDATE", dynamicConfig = EruptRecordNaming.class)
    @EruptRouter(skipAuthIndex = 3, authIndex = 1, verifyType = EruptRouter.VerifyType.ERUPT)
    @Transactional
    @SneakyThrows
    public R<Void> putUpdateEruptData(@PathVariable("erupt") String erupt, @RequestBody JsonObject data) {
        EruptModel eruptModel = EruptCoreService.getErupt(erupt);
        Erupts.powerLegal(eruptModel, PowerObject::isEdit);
        EruptApiModel eruptApiModel = EruptUtil.validateEruptValue(eruptModel, data);
        if (eruptApiModel.getStatus() == EruptApiModel.Status.ERROR) return R.error(eruptApiModel.getMessage());
        eruptService.verifyIdPermissions(eruptModel, data.get(eruptModel.getErupt().primaryKeyCol()).getAsString());
        Object o = GsonFactory.getGson().fromJson(data.toString(), eruptModel.getClazz());
        EruptUtil.clearObjectDefaultValueByJson(o, data);
        Object old = DataProcessorManager.getEruptDataProcessor(eruptModel.getClazz()).findDataById(eruptModel, ReflectUtil.findClassField(eruptModel.getClazz(), eruptModel.getErupt().primaryKeyCol()).get(o));
        Object realOld = eruptModel.getClazz().getConstructor().newInstance();
        BeanUtils.copyProperties(old, realOld);
        Object obj = EruptUtil.dataTarget(eruptModel, o, old, SceneEnum.EDIT);
        DataProxyInvoke.invoke(eruptModel, (dataProxy -> dataProxy.beforeUpdate(obj)));
        DataProcessorManager.getEruptDataProcessor(eruptModel.getClazz()).editData(eruptModel, obj);
        eruptModifyService.modifyLog(eruptModel, "EDIT", data.toString());
        DataProxyInvoke.invoke(eruptModel, (dataProxy -> dataProxy.afterUpdate(obj)));
        applicationEventPublisher.publishEvent(new EruptEditEvent<>(eruptModel.getClazz(), obj, realOld));
        return R.ok();
    }

    @Deprecated
    @DeleteMapping("/{erupt}/{id}")
    @EruptRecordOperate(value = "DELETE", dynamicConfig = EruptRecordNaming.class)
    @EruptRouter(skipAuthIndex = 3, authIndex = 1, verifyType = EruptRouter.VerifyType.ERUPT)
    @Transactional
    public EruptApiModel deleteEruptData(@PathVariable("erupt") String erupt, @PathVariable("id") String id) {
        EruptModel eruptModel = EruptCoreService.getErupt(erupt);
        Erupts.powerLegal(eruptModel, PowerObject::isDelete);
        eruptService.verifyIdPermissions(eruptModel, id);
        IEruptDataService dataService = DataProcessorManager.getEruptDataProcessor(eruptModel.getClazz());
        //获取对象数据信息用于DataProxy函数中
        Object obj = dataService.findDataById(eruptModel, EruptUtil.toEruptId(eruptModel, id));
        DataProxyInvoke.invoke(eruptModel, (dataProxy -> dataProxy.beforeDelete(obj)));
        dataService.deleteData(eruptModel, obj);
        eruptModifyService.modifyLog(eruptModel, "DELETE", id);
        DataProxyInvoke.invoke(eruptModel, (dataProxy -> dataProxy.afterDelete(obj)));
        applicationEventPublisher.publishEvent(new EruptDeleteEvent<>(eruptModel.getClazz(), obj));
        return EruptApiModel.successApi();
    }

    @Deprecated
    @DeleteMapping("/{erupt}")
    @EruptRouter(skipAuthIndex = 3, authIndex = 1, verifyType = EruptRouter.VerifyType.ERUPT)
    @EruptRecordOperate(value = "DELETES", dynamicConfig = EruptRecordNaming.class)
    @Transactional
    public R<Void> deleteEruptDataList(@PathVariable("erupt") String erupt, @RequestParam("ids") String[] ids) {
        for (String id : ids) {
            EruptApiModel eruptApiModel = this.deleteEruptData(erupt, id);
            if (eruptApiModel.getStatus() == EruptApiModel.Status.ERROR) {
                return R.error(eruptApiModel.getMessage());
            }
        }
        return R.ok();
    }

}
