/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.api.variant

import org.gradle.api.Incubating
import org.gradle.api.Named
import org.gradle.api.Task
import org.gradle.api.file.Directory
import org.gradle.api.provider.Provider
import org.gradle.api.tasks.TaskProvider

/**
 * Represents all the source folders for a source type in the variant.
 *
 * since 7.2
 */
@Incubating
interface SourceDirectories: Named {

    /**
     * Add the output of a custom task to the list of source directories.
     *
     * The [Directory] is the output of a Task [T] that has been registered using the Gradle's Task
     * manager.
     *
     * The [Directory] is added last to the variant's list of source directories. In case there is
     * merging for the source type, the [Directory] will have the highest priority.
     *
     * @param taskProvider the [TaskProvider] returned by Gradle's Task manager when registering the
     * Task of type [T].
     * @param wiredWith the method reference returning the [T] task's output to use as a source
     * directory. The generated source directory location is automatically determined by the
     * Android Gradle Plugin
     */
    fun <T: Task> add(taskProvider: TaskProvider<T>, wiredWith: (T) -> Provider<Directory>)

    /**
     * Add a source directory with sources already present to the variant.
     *
     * The directory will be added last in the list of source folders for the variant. In case there
     * is merging for the source type, [srcDir] will be the folder with the highest priority.
     *
     * Do not use [addSrcDir] to add sources that are generated by a task, instead use [add]
     *
     * @param srcDir the source directory path, that will be resolved using the [Directory.dir] API
     * relative to the Gradle project directory.
     */
    fun addSrcDir(srcDir: String)

    /**
     * Get all registered source folders and files as a [List] of [Directory].
     *
     * Some source types do not have the concept of overriding, while others require a merging step to
     * ensure only one source file is used when processing begins.
     *
     * The returned [Provider] can be used directly in a [org.gradle.api.tasks.InputFiles] annotated
     * property of a [Task]
     */
    val all: Provider<List<Directory>>
}
