package com.applovin.sdk;

import android.content.Context;

import com.applovin.impl.privacy.ComplianceManager;
import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.Logger;

import androidx.annotation.Nullable;

/**
 * This class contains privacy settings for AppLovin.
 */
public class AppLovinPrivacySettings
{
    private static final String TAG = "AppLovinPrivacySettings";

    /**
     * Set whether or not user has provided consent for information sharing.
     *
     * @param hasUserConsent {@code true} if the user has provided consent for information sharing.
     * @param context        Parent activity or application context. Must not be {@code null}.
     */
    public static void setHasUserConsent(final boolean hasUserConsent, final Context context)
    {
        Logger.userDebug( TAG, "setHasUserConsent()" );

        final boolean isValueChanged = ComplianceManager.setHasUserConsent( hasUserConsent, context );
        if ( isValueChanged )
        {
            AppLovinSdk.reinitializeAll( hasUserConsent, null, null );
        }
    }

    /**
     * Check if user has provided consent for information sharing.
     *
     * @param context Parent activity or application context. Must not be {@code null}.
     *
     * @return {@code true} if user has provided consent for information sharing. {@code false} if the user declined to share information or the consent value has not been set (see {@link #isUserConsentSet(Context)}).
     */
    public static boolean hasUserConsent(final Context context)
    {
        Logger.userDebug( TAG, "hasUserConsent()" );

        final Boolean hasUserConsent = ComplianceManager.hasUserConsent().getValue( context );
        if ( hasUserConsent != null )
        {
            return hasUserConsent;
        }
        else
        {
            return false;
        }
    }

    /**
     * Check if user has set consent for information sharing.
     *
     * @param context Parent activity or application context. Must not be {@code null}.
     *
     * @return {@code true} if user has set a value of consent for information sharing.
     */
    public static boolean isUserConsentSet(final Context context)
    {
        Logger.userDebug( TAG, "isUserConsentSet()" );

        return ComplianceManager.hasUserConsent().getValue( context ) != null;
    }

    /**
     * Mark user as age restricted (i.e. under 16).
     *
     * @param isAgeRestrictedUser {@code true} if the user is age restricted (i.e. under 16).
     * @param context             Parent activity or application context. Must not be {@code null}.
     */
    public static void setIsAgeRestrictedUser(final boolean isAgeRestrictedUser, final Context context)
    {
        Logger.userDebug( TAG, "setIsAgeRestrictedUser()" );

        final boolean isValueChanged = ComplianceManager.setIsAgeRestrictedUser( isAgeRestrictedUser, context );
        if ( isValueChanged )
        {
            AppLovinSdk.reinitializeAll( null, isAgeRestrictedUser, null );
        }
    }

    /**
     * Check if user is age restricted.
     *
     * @param context Parent activity or application context. Must not be {@code null}.
     *
     * @return {@code true} if the user is age-restricted. {@code false} if the user is not age-restricted or the age-restriction value has not been set (see {@link #isAgeRestrictedUserSet(Context)}).
     */
    public static boolean isAgeRestrictedUser(final Context context)
    {
        Logger.userDebug( TAG, "isAgeRestrictedUser()" );

        final Boolean isAgeRestrictedUser = ComplianceManager.isAgeRestrictedUser().getValue( context );
        if ( isAgeRestrictedUser != null )
        {
            return isAgeRestrictedUser;
        }
        else
        {
            return false;
        }
    }

    /**
     * Check if user has set its age restricted settings.
     *
     * @param context Parent activity or application context. Must not be {@code null}.
     *
     * @return {@code true} if user has set its age restricted settings.
     */
    public static boolean isAgeRestrictedUserSet(final Context context)
    {
        Logger.userDebug( TAG, "isAgeRestrictedUserSet()" );

        return ComplianceManager.isAgeRestrictedUser().getValue( context ) != null;
    }

    /**
     * Set whether or not user has opted out of the sale of their personal information.
     *
     * @param doNotSell {@code true} if the user has opted out of the sale of their personal information.
     * @param context   Parent activity or application context. Must not be {@code null}.
     */
    public static void setDoNotSell(final boolean doNotSell, final Context context)
    {
        Logger.userDebug( TAG, "setDoNotSell()" );

        final boolean isValueChanged = ComplianceManager.setDoNotSell( doNotSell, context );
        if ( isValueChanged )
        {
            AppLovinSdk.reinitializeAll( null, null, doNotSell );
        }
    }

    /**
     * Check if the user has opted out of the sale of their personal information.
     *
     * @param context Parent activity or application context. Must not be {@code null}.
     *
     * @return {@code true} if user has opted out of the sale of their personal information. {@code false} if the user opted in to the sale of their personal information or the value has not been set (see {@link #isDoNotSellSet(Context)}).
     */
    public static boolean isDoNotSell(final Context context)
    {
        Logger.userDebug( TAG, "isDoNotSell()" );

        final Boolean isDoNotSell = ComplianceManager.doNotSell().getValue( context );
        if ( isDoNotSell != null )
        {
            return isDoNotSell;
        }
        else
        {
            return false;
        }
    }

    /**
     * Check if the user has set the option to sell their personal information.
     *
     * @param context Parent activity or application context. Must not be {@code null}.
     *
     * @return {@code true} if user has chosen an option to sell their personal information.
     */
    public static boolean isDoNotSellSet(final Context context)
    {
        Logger.userDebug( TAG, "isDoNotSellSet()" );

        return ComplianceManager.doNotSell().getValue( context ) != null;
    }

    /**
     * Parses the IABTCF_VendorConsents string to determine the consent status of the IAB vendor with the provided ID.
     * <p>
     * NOTE: Must be called after AppLovin MAX SDK has been initialized.
     *
     * @param vendorId Vendor ID as defined in the Global Vendor List.
     *
     * @return {@code true} if the vendor has consent, {@code false} if not, or {@code null} if TC data is not available on disk.
     * @see <a href="https://vendor-list.consensu.org/v3/vendor-list.json">Current Version of Global Vendor List</a>
     */
    @Nullable
    public static Boolean getTcfVendorConsentStatus(final int vendorId)
    {
        if ( CoreSdk.MAIN_SDK == null )
        {
            Logger.userError( TAG, "AppLovinPrivacySettings.getTcfVendorConsentStatus(...) called when AppLovin MAX SDK is not initialized yet" );
            return null;
        }

        return CoreSdk.MAIN_SDK.getTcfManager().getTcfVendorConsentStatus( vendorId );
    }

    /**
     * Parses the IABTCF_AddtlConsent string to determine the consent status of the advertising entity with the provided Ad Technology Provider (ATP) ID.
     * <p>
     * NOTE: Must be called after AppLovin MAX SDK has been initialized.
     *
     * @param atpId ATP ID of the advertising entity (e.g. 89 for Meta Audience Network).
     *
     * @return {@code true} if the advertising entity has consent, {@code false} if not, or {@code null} if no AC string is available on disk or the ATP network was not listed in the CMP flow.
     * @see <a href="https://support.google.com/admanager/answer/9681920">Google’s Additional Consent Mode technical specification</a>
     * @see <a href="https://storage.googleapis.com/tcfac/additional-consent-providers.csv">List of Google ATPs and their IDs</a>
     */
    @Nullable
    public static Boolean getAdditionalConsentStatus(final int atpId)
    {
        if ( CoreSdk.MAIN_SDK == null )
        {
            Logger.userError( TAG, "AppLovinPrivacySettings.getAdditionalConsentStatus(...) called when AppLovin MAX SDK is not initialized yet" );
            return null;
        }

        return CoreSdk.MAIN_SDK.getTcfManager().getAdditionalConsentStatus( atpId );
    }

    /**
     * Parses the IABTCF_PurposeConsents String to determine the consent status of the IAB defined data processing purpose.
     * <p>
     * NOTE: Must be called after AppLovin MAX SDK has been initialized.
     *
     * @param purposeId Purpose ID.
     *
     * @return {@code true} if the purpose has consent, {@code false} if not, or {@code null} if TC data is not available on disk.
     * @see <a href="https://iabeurope.eu/iab-europe-transparency-consent-framework-policies">IAB Europe Transparency & Consent Framework Policies (Appendix A)</a> for purpose definitions.
     */
    @Nullable
    public static Boolean getPurposeConsentStatus(final int purposeId)
    {
        if ( CoreSdk.MAIN_SDK == null )
        {
            Logger.userError( TAG, "AppLovinPrivacySettings.getPurposeConsentStatus(...) called when AppLovin MAX SDK is not initialized yet" );
            return null;
        }

        return CoreSdk.MAIN_SDK.getTcfManager().getPurposeConsentStatus( purposeId );
    }

    /**
     * Parses the IABTCF_SpecialFeaturesOptIns String to determine the opt-in status of the IAB defined special feature.
     * <p>
     * NOTE: Must be called after AppLovin MAX SDK has been initialized.
     *
     * @param specialFeatureId Special feature ID.
     *
     * @return {@code true} if the user opted in for the special feature, {@code false} if not, or {@code null} if TC data is not available on disk.
     * @see <a href="https://iabeurope.eu/iab-europe-transparency-consent-framework-policies">IAB Europe Transparency & Consent Framework Policies (Appendix A)</a> for special features definitions.
     */
    @Nullable
    public static Boolean getSpecialFeatureOptInStatus(final int specialFeatureId)
    {
        if ( CoreSdk.MAIN_SDK == null )
        {
            Logger.userError( TAG, "AppLovinPrivacySettings.getSpecialFeatureOptInStatus(...) called when AppLovin MAX SDK is not initialized yet" );
            return null;
        }

        return CoreSdk.MAIN_SDK.getTcfManager().getSpecialFeatureOptInStatus( specialFeatureId );
    }
}
