package com.applovin.mediation.adapters;

import android.app.Activity;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.ViewGroup;
import android.widget.RelativeLayout;

import com.applovin.mediation.MaxAdFormat;
import com.applovin.mediation.MaxReward;
import com.applovin.mediation.adapter.MaxAdViewAdapter;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.MaxInterstitialAdapter;
import com.applovin.mediation.adapter.MaxRewardedAdapter;
import com.applovin.mediation.adapter.listeners.MaxAdViewAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxRewardedAdapterListener;
import com.applovin.mediation.adapter.parameters.MaxAdapterInitializationParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterResponseParameters;
import com.applovin.mediation.adapters.inneractive.BuildConfig;
import com.applovin.sdk.AppLovinSdk;
import com.applovin.sdk.AppLovinSdkConfiguration;
import com.fyber.inneractive.sdk.external.ImpressionData;
import com.fyber.inneractive.sdk.external.InneractiveAdManager;
import com.fyber.inneractive.sdk.external.InneractiveAdRequest;
import com.fyber.inneractive.sdk.external.InneractiveAdSpot;
import com.fyber.inneractive.sdk.external.InneractiveAdSpotManager;
import com.fyber.inneractive.sdk.external.InneractiveAdViewEventsListenerWithImpressionData;
import com.fyber.inneractive.sdk.external.InneractiveAdViewUnitController;
import com.fyber.inneractive.sdk.external.InneractiveErrorCode;
import com.fyber.inneractive.sdk.external.InneractiveFullScreenAdRewardedListener;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenAdEventsListenerWithImpressionData;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenUnitController;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenVideoContentController;
import com.fyber.inneractive.sdk.external.InneractiveUnitController;
import com.fyber.inneractive.sdk.external.VideoContentListener;

import java.lang.reflect.Method;
import java.util.concurrent.atomic.AtomicBoolean;

public class InneractiveMediationAdapter
        extends MediationAdapterBase
        implements MaxInterstitialAdapter, MaxRewardedAdapter, MaxAdViewAdapter
{
    private static final AtomicBoolean INITIALIZED = new AtomicBoolean();

    private InneractiveAdSpot interstitialSpot;
    private InneractiveAdSpot rewardedSpot;
    private InneractiveAdSpot adViewSpot;

    private ViewGroup adViewGroup;
    private boolean   hasGrantedReward;

    // Explicit default constructor declaration
    public InneractiveMediationAdapter(final AppLovinSdk sdk) { super( sdk ); }

    @Override
    public void initialize(final MaxAdapterInitializationParameters parameters, final Activity activity, final OnCompletionListener onCompletionListener)
    {
        // Check existence of SDK classes
        checkExistence( InneractiveAdManager.class, InneractiveAdSpot.class, InneractiveFullscreenUnitController.class, InneractiveAdViewUnitController.class );

        if ( INITIALIZED.compareAndSet( false, true ) )
        {
            final String appId = parameters.getServerParameters().getString( "app_id", null );
            log( "Initializing Inneractive SDK with app id: " + appId + "..." );

            InneractiveAdManager.initialize( activity, appId );
            InneractiveAdManager.setUserId( getWrappingSdk().getUserIdentifier() );
            log( "Inneractive SDK initialized" );
        }

        onCompletionListener.onCompletion( InitializationStatus.DOES_NOT_APPLY, null );
    }

    @Override
    public String getSdkVersion()
    {
        return InneractiveAdManager.getVersion();
    }

    @Override
    public String getAdapterVersion()
    {
        return BuildConfig.VERSION_NAME;
    }

    @Override
    public void loadInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        log( "Loading interstitial ad..." );

        final InneractiveFullscreenVideoContentController videoContentController = new InneractiveFullscreenVideoContentController();

        final InneractiveFullscreenUnitController controller = new InneractiveFullscreenUnitController();
        controller.addContentController( videoContentController );
        controller.setEventsListener( new InneractiveFullscreenAdEventsListenerWithImpressionData()
        {
            @Override
            public void onAdImpression(final InneractiveAdSpot inneractiveAdSpot) { }

            @Override
            public void onAdImpression(final InneractiveAdSpot inneractiveAdSpot, final ImpressionData impressionData)
            {
                log( "Interstitial shown" );

                // Passing extra info such as creative id supported in 9.15.0+
                String creativeId = impressionData.getCreativeId();
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( creativeId ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", creativeId );

                    listener.onInterstitialAdDisplayed( extraInfo );
                }
                else
                {
                    listener.onInterstitialAdDisplayed();
                }
            }

            @Override
            public void onAdClicked(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "Interstitial clicked" );
                listener.onInterstitialAdClicked();
            }

            @Override
            public void onAdDismissed(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "Interstitial hidden" );
                listener.onInterstitialAdHidden();
            }

            @Override
            public void onAdEnteredErrorState(final InneractiveAdSpot inneractiveAdSpot, final InneractiveUnitController.AdDisplayError adDisplayError)
            {
                log( "Interstitial failed to show: " + adDisplayError );

                final MaxAdapterError adapterError = new MaxAdapterError( MaxAdapterError.UNSPECIFIED.getErrorCode(), MaxAdapterError.UNSPECIFIED.getErrorMessage(), 0, adDisplayError.toString() );
                listener.onInterstitialAdDisplayFailed( adapterError );
            }

            @Override
            public void onAdWillOpenExternalApp(final InneractiveAdSpot inneractiveAdSpot) {}

            @Override
            public void onAdWillCloseInternalBrowser(final InneractiveAdSpot inneractiveAdSpot) {}
        } );

        interstitialSpot = InneractiveAdSpotManager.get().createSpot();
        interstitialSpot.addUnitController( controller );
        interstitialSpot.setRequestListener( new InneractiveAdSpot.RequestListener()
        {
            @Override
            public void onInneractiveSuccessfulAdRequest(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "Interstitial loaded" );
                listener.onInterstitialAdLoaded();
            }

            @Override
            public void onInneractiveFailedAdRequest(final InneractiveAdSpot inneractiveAdSpot, final InneractiveErrorCode inneractiveErrorCode)
            {
                log( "Interstitial failed to load with Inneractive error: " + inneractiveErrorCode + " " + inneractiveErrorCode.toString() );
                listener.onInterstitialAdLoadFailed( toMaxError( inneractiveErrorCode ) );
            }
        } );

        final InneractiveAdRequest request = createAdRequest( parameters );
        interstitialSpot.requestAd( request );
    }

    @Override
    public void showInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        log( "Showing interstitial ad..." );

        if ( interstitialSpot.isReady() )
        {
            final InneractiveFullscreenUnitController controller = (InneractiveFullscreenUnitController) interstitialSpot.getSelectedUnitController();
            controller.show( activity );
        }
        else
        {
            log( "Interstitial ad not ready" );
            listener.onInterstitialAdDisplayFailed( MaxAdapterError.AD_NOT_READY );
        }
    }

    @Override
    public void loadRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        log( "Loading rewarded ad..." );

        final InneractiveFullscreenVideoContentController videoContentController = new InneractiveFullscreenVideoContentController();
        videoContentController.setEventsListener( new VideoContentListener()
        {
            @Override
            public void onProgress(final int totalDurationMillis, final int positionMillis)
            {
                if ( positionMillis == 0 )
                {
                    log( "Rewarded video started" );
                }
            }

            @Override
            public void onCompleted()
            {
                log( "Rewarded video completed" );
                listener.onRewardedAdVideoCompleted();
            }

            @Override
            public void onPlayerError()
            {
                log( "Rewarded video failed to display for unspecified error" );
                listener.onRewardedAdDisplayFailed( MaxAdapterError.UNSPECIFIED );
            }
        } );

        final InneractiveFullscreenUnitController controller = new InneractiveFullscreenUnitController();
        controller.addContentController( videoContentController );
        controller.setEventsListener( new InneractiveFullscreenAdEventsListenerWithImpressionData()
        {
            @Override
            public void onAdImpression(final InneractiveAdSpot inneractiveAdSpot) { }

            @Override
            public void onAdImpression(final InneractiveAdSpot inneractiveAdSpot, final ImpressionData impressionData)
            {
                log( "Rewarded ad shown" );

                // Passing extra info such as creative id supported in 9.15.0+
                String creativeId = impressionData.getCreativeId();
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( creativeId ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", creativeId );

                    listener.onRewardedAdDisplayed( extraInfo );
                }
                else
                {
                    listener.onRewardedAdDisplayed();
                }

                // `VideoContentListener.onProgress()` is called before this, but we want the video started callback after the cimp.
                listener.onRewardedAdVideoStarted();
            }

            @Override
            public void onAdClicked(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "Rewarded ad clicked" );
                listener.onRewardedAdClicked();
            }

            @Override
            public void onAdDismissed(final InneractiveAdSpot inneractiveAdSpot)
            {
                if ( hasGrantedReward || shouldAlwaysRewardUser() )
                {
                    final MaxReward reward = getReward();
                    log( "Rewarded user with reward: " + reward );
                    listener.onUserRewarded( reward );
                }

                log( "Rewarded ad hidden" );
                listener.onRewardedAdHidden();
            }

            @Override
            public void onAdEnteredErrorState(final InneractiveAdSpot inneractiveAdSpot, final InneractiveUnitController.AdDisplayError adDisplayError)
            {
                log( "Rewarded ad failed to show: " + adDisplayError );

                final MaxAdapterError adapterError = new MaxAdapterError( MaxAdapterError.UNSPECIFIED.getErrorCode(), MaxAdapterError.UNSPECIFIED.getErrorMessage(), 0, adDisplayError.toString() );
                listener.onRewardedAdDisplayFailed( adapterError );
            }

            @Override
            public void onAdWillOpenExternalApp(final InneractiveAdSpot inneractiveAdSpot) {}

            @Override
            public void onAdWillCloseInternalBrowser(final InneractiveAdSpot inneractiveAdSpot) {}
        } );

        controller.setRewardedListener( new InneractiveFullScreenAdRewardedListener()
        {
            @Override
            public void onAdRewarded(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "User earned reward." );
                hasGrantedReward = true;
            }
        } );

        rewardedSpot = InneractiveAdSpotManager.get().createSpot();
        rewardedSpot.addUnitController( controller );
        rewardedSpot.setRequestListener( new InneractiveAdSpot.RequestListener()
        {
            @Override
            public void onInneractiveSuccessfulAdRequest(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "Rewarded ad loaded" );
                listener.onRewardedAdLoaded();
            }

            @Override
            public void onInneractiveFailedAdRequest(final InneractiveAdSpot inneractiveAdSpot, final InneractiveErrorCode inneractiveErrorCode)
            {
                log( "Rewarded ad failed to load with Inneractive error: " + inneractiveErrorCode + " " + inneractiveErrorCode.toString() );
                listener.onRewardedAdLoadFailed( toMaxError( inneractiveErrorCode ) );
            }
        } );

        final InneractiveAdRequest request = createAdRequest( parameters );
        rewardedSpot.requestAd( request );
    }

    @Override
    public void showRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        log( "Showing rewarded ad..." );

        if ( rewardedSpot.isReady() )
        {
            // Configure userReward from server.
            configureReward( parameters );

            final InneractiveFullscreenUnitController controller = (InneractiveFullscreenUnitController) rewardedSpot.getSelectedUnitController();
            controller.show( activity );
        }
        else
        {
            log( "Rewarded ad not ready" );
            listener.onRewardedAdDisplayFailed( MaxAdapterError.AD_NOT_READY );
        }
    }

    @Override
    public void loadAdViewAd(final MaxAdapterResponseParameters parameters, final MaxAdFormat adFormat, final Activity activity, final MaxAdViewAdapterListener listener)
    {
        log( "Loading " + adFormat.getLabel() + " AdView ad..." );

        final InneractiveAdViewUnitController controller = new InneractiveAdViewUnitController();
        controller.setEventsListener( new InneractiveAdViewEventsListenerWithImpressionData()
        {
            @Override
            public void onAdImpression(final InneractiveAdSpot inneractiveAdSpot) { }

            @Override
            public void onAdImpression(final InneractiveAdSpot inneractiveAdSpot, final ImpressionData impressionData)
            {
                log( "AdView shown" );

                // Passing extra info such as creative id supported in 9.15.0+
                String creativeId = impressionData.getCreativeId();
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( creativeId ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", creativeId );

                    listener.onAdViewAdDisplayed( extraInfo );
                }
                else
                {
                    listener.onAdViewAdDisplayed();
                }
            }

            @Override
            public void onAdClicked(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "AdView clicked" );
                listener.onAdViewAdClicked();
            }

            @Override
            public void onAdExpanded(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "AdView expanded" );
                listener.onAdViewAdExpanded();
            }

            @Override
            public void onAdCollapsed(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "AdView collapsed" );
                listener.onAdViewAdCollapsed();
            }

            @Override
            public void onAdEnteredErrorState(final InneractiveAdSpot inneractiveAdSpot, final InneractiveUnitController.AdDisplayError adDisplayError)
            {
                log( "AdView failed to show: " + adDisplayError );

                final MaxAdapterError adapterError = new MaxAdapterError( MaxAdapterError.UNSPECIFIED.getErrorCode(), MaxAdapterError.UNSPECIFIED.getErrorMessage(), 0, adDisplayError.toString() );
                listener.onAdViewAdDisplayFailed( adapterError );
            }

            @Override
            public void onAdResized(final InneractiveAdSpot inneractiveAdSpot) {}

            @Override
            public void onAdWillCloseInternalBrowser(final InneractiveAdSpot inneractiveAdSpot) {}

            @Override
            public void onAdWillOpenExternalApp(final InneractiveAdSpot inneractiveAdSpot) {}
        } );

        adViewGroup = new RelativeLayout( activity );

        adViewSpot = InneractiveAdSpotManager.get().createSpot();
        adViewSpot.addUnitController( controller );
        adViewSpot.setRequestListener( new InneractiveAdSpot.RequestListener()
        {
            @Override
            public void onInneractiveSuccessfulAdRequest(final InneractiveAdSpot inneractiveAdSpot)
            {
                log( "AdView loaded" );
                controller.bindView( adViewGroup );
                listener.onAdViewAdLoaded( adViewGroup );
            }

            @Override
            public void onInneractiveFailedAdRequest(final InneractiveAdSpot inneractiveAdSpot, final InneractiveErrorCode inneractiveErrorCode)
            {
                log( "AdView failed to load with Inneractive error: " + inneractiveErrorCode + " " + inneractiveErrorCode.toString() );
                listener.onAdViewAdLoadFailed( toMaxError( inneractiveErrorCode ) );
            }
        } );

        final InneractiveAdRequest request = createAdRequest( parameters );
        adViewSpot.requestAd( request );
    }

    @Override
    public void onDestroy()
    {
        if ( interstitialSpot != null )
        {
            interstitialSpot.destroy();
            interstitialSpot = null;
        }

        if ( rewardedSpot != null )
        {
            rewardedSpot.destroy();
            rewardedSpot = null;
        }

        if ( adViewSpot != null )
        {
            adViewSpot.destroy();
            adViewSpot = null;
        }
    }

    private InneractiveAdRequest createAdRequest(final MaxAdapterResponseParameters parameters)
    {
        InneractiveAdManager.setUserId( getWrappingSdk().getUserIdentifier() );

        if ( getWrappingSdk().getConfiguration().getConsentDialogState() == AppLovinSdkConfiguration.ConsentDialogState.APPLIES )
        {
            Boolean hasUserConsent = getPrivacySetting( "hasUserConsent", parameters );
            if ( hasUserConsent != null )
            {
                InneractiveAdManager.setGdprConsent( hasUserConsent );
            }
        }

        InneractiveAdRequest request = new InneractiveAdRequest( parameters.getThirdPartyAdPlacementId() );

        Bundle serverParameters = parameters.getServerParameters();
        if ( serverParameters.containsKey( "is_muted" ) ) // Introduced in 9.10.0
        {
            request.setMuteVideo( serverParameters.getBoolean( "is_muted" ) );
        }

        return request;
    }

    private Boolean getPrivacySetting(final String privacySetting, final MaxAdapterParameters parameters)
    {
        try
        {
            // Use reflection because compiled adapters have trouble fetching `boolean` from old SDKs and `Boolean` from new SDKs (above 9.14.0)
            Class<?> parametersClass = parameters.getClass();
            Method privacyMethod = parametersClass.getMethod( privacySetting );
            return (Boolean) privacyMethod.invoke( parameters );
        }
        catch ( Exception exception )
        {
            log( "Error getting privacy setting " + privacySetting + " with exception: ", exception );
            return ( AppLovinSdk.VERSION_CODE >= 9140000 ) ? null : false;
        }
    }

    private static MaxAdapterError toMaxError(final InneractiveErrorCode inneractiveErrorCode)
    {
        final MaxAdapterError adapterError;
        if ( inneractiveErrorCode == InneractiveErrorCode.NO_FILL )
        {
            adapterError = MaxAdapterError.NO_FILL;
        }
        else if ( inneractiveErrorCode == InneractiveErrorCode.SERVER_INTERNAL_ERROR )
        {
            adapterError = MaxAdapterError.SERVER_ERROR;
        }
        else if ( inneractiveErrorCode == InneractiveErrorCode.SERVER_INVALID_RESPONSE )
        {
            adapterError = MaxAdapterError.BAD_REQUEST;
        }
        else if ( inneractiveErrorCode == InneractiveErrorCode.SDK_INTERNAL_ERROR )
        {
            adapterError = MaxAdapterError.INTERNAL_ERROR;
        }
        else if ( inneractiveErrorCode == InneractiveErrorCode.CONNECTION_TIMEOUT || inneractiveErrorCode == InneractiveErrorCode.LOAD_TIMEOUT )
        {
            adapterError = MaxAdapterError.TIMEOUT;
        }
        else if ( inneractiveErrorCode == InneractiveErrorCode.CONNECTION_ERROR )
        {
            adapterError = MaxAdapterError.NO_CONNECTION;
        }
        else if ( inneractiveErrorCode == InneractiveErrorCode.UNKNOWN_APP_ID )
        {
            adapterError = MaxAdapterError.NOT_INITIALIZED;
        }
        else if ( inneractiveErrorCode == InneractiveErrorCode.ERROR_CONFIGURATION_MISMATCH )
        {
            adapterError = MaxAdapterError.INVALID_CONFIGURATION;
        }
        else if ( inneractiveErrorCode == InneractiveErrorCode.SPOT_DISABLED )
        {
            adapterError = MaxAdapterError.INVALID_CONFIGURATION;
        }
        else
        {
            adapterError = MaxAdapterError.UNSPECIFIED;
        }

        final int adapterErrorCode;
        final String adapterErrorStr;
        if ( inneractiveErrorCode != null )
        {
            adapterErrorCode = inneractiveErrorCode.ordinal();
            adapterErrorStr = inneractiveErrorCode.name();
        }
        else
        {
            adapterErrorCode = 0;
            adapterErrorStr = "";
        }
        return new MaxAdapterError( adapterError.getErrorCode(), adapterError.getErrorMessage(), adapterErrorCode, adapterErrorStr );
    }
}
