package com.applovin.mediation.adapters;

import android.app.Activity;
import android.content.SharedPreferences;
import android.preference.PreferenceManager;
import android.text.TextUtils;

import com.applovin.mediation.MaxReward;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.MaxInterstitialAdapter;
import com.applovin.mediation.adapter.MaxRewardedAdapter;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxRewardedAdapterListener;
import com.applovin.mediation.adapter.parameters.MaxAdapterInitializationParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterResponseParameters;
import com.applovin.sdk.AppLovinSdk;
import com.hyprmx.android.sdk.core.HyprMX;
import com.hyprmx.android.sdk.core.HyprMXErrors;
import com.hyprmx.android.sdk.core.HyprMXIf;
import com.hyprmx.android.sdk.core.HyprMXState;
import com.hyprmx.android.sdk.placement.Placement;
import com.hyprmx.android.sdk.placement.PlacementListener;
import com.hyprmx.android.sdk.placement.RewardedPlacementListener;
import com.hyprmx.android.sdk.utility.HyprMXLog;

import java.util.Locale;
import java.util.UUID;

public class HyprMXMediationAdapter
        extends MediationAdapterBase
        implements MaxInterstitialAdapter, MaxRewardedAdapter
{
    private static final String KEY_RANDOM_HYPRMX_USER_ID = "com.applovin.sdk.mediation.random_hyprmx_user_id";

    // Interstitial
    private Placement            interstitialAd;
    private InterstitialListener interstitialAdListener;

    // Rewarded
    private Placement          rewardedAd;
    private RewardedAdListener rewardedAdListener;

    // Explicit default constructor declaration
    public HyprMXMediationAdapter(final AppLovinSdk appLovinSdk)
    {
        super( appLovinSdk );
    }

    @Override
    public String getSdkVersion()
    {
        return getVersionString( com.hyprmx.android.BuildConfig.class, "VERSION_NAME" );
    }

    @Override
    public String getAdapterVersion()
    {
        return com.applovin.mediation.adapters.hyprmx.BuildConfig.VERSION_NAME;
    }

    @Override
    public void onDestroy()
    {
        interstitialAd = null;
        interstitialAdListener = null;

        rewardedAd = null;
        rewardedAdListener = null;
    }

    @Override
    public void initialize(final MaxAdapterInitializationParameters parameters, final Activity activity, final OnCompletionListener onCompletionListener)
    {
        if ( HyprMX.INSTANCE.getInitializationState() == HyprMXState.NOT_INITIALIZED )
        {
            final String distributorId = parameters.getServerParameters().getString( "distributor_id" );

            // HyprMX requires userId to initialize -> product team suggested generating a random one
            String userId = getWrappingSdk().getUserIdentifier();
            if ( TextUtils.isEmpty( userId ) )
            {
                SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences( activity.getApplicationContext() );
                userId = sharedPreferences.getString( KEY_RANDOM_HYPRMX_USER_ID, null );
                if ( TextUtils.isEmpty( userId ) )
                {
                    userId = UUID.randomUUID().toString().toLowerCase( Locale.US ); // Guarantee lower-case even if it's supposed to be default, b/c Android;
                    sharedPreferences.edit().putString( KEY_RANDOM_HYPRMX_USER_ID, userId ).apply();
                }
            }

            log( "Initializing HyprMX SDK with distributor id: " + distributorId );

            HyprMXLog.enableDebugLogs( parameters.isTesting() );

            // NOTE: HyprMX deals with user consent and CCPA via their UI and don't support GDPR. Backend will filter HyprMX out in EU region.
            HyprMX.INSTANCE.initialize( activity.getApplicationContext(), distributorId, userId, new HyprMXIf.HyprMXInitializationListener()
            {
                @Override
                public void initializationComplete()
                {
                    log( "HyprMX SDK initialized" );
                    onCompletionListener.onCompletion( InitializationStatus.INITIALIZED_SUCCESS, null );
                }

                @Override
                public void initializationFailed()
                {
                    log( "HyprMX SDK failed to initialize" );
                    onCompletionListener.onCompletion( InitializationStatus.INITIALIZED_FAILURE, null );
                }
            } );
        }
        else
        {
            if ( HyprMX.INSTANCE.getInitializationState() == HyprMXState.INITIALIZATION_COMPLETE )
            {
                onCompletionListener.onCompletion( InitializationStatus.INITIALIZED_SUCCESS, null );
            }
            else if ( HyprMX.INSTANCE.getInitializationState() == HyprMXState.INITIALIZATION_FAILED )
            {
                onCompletionListener.onCompletion( InitializationStatus.INITIALIZED_FAILURE, null );
            }
            else if ( HyprMX.INSTANCE.getInitializationState() == HyprMXState.INITIALIZING )
            {
                onCompletionListener.onCompletion( InitializationStatus.INITIALIZING, null );
            }
            else
            {
                onCompletionListener.onCompletion( InitializationStatus.INITIALIZED_UNKNOWN, null );
            }
        }
    }

    @Override
    public void loadInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading interstitial ad for placement: " + placementId );

        interstitialAdListener = new InterstitialListener( listener );
        interstitialAd = createFullscreenAd( placementId, parameters, interstitialAdListener );

        interstitialAd.loadAd();
    }

    @Override
    public void showInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        log( "Showing interstitial ad" );

        if ( interstitialAd.isAdAvailable() )
        {
            interstitialAd.showAd();
        }
        else
        {
            log( "Interstitial ad not ready" );
            listener.onInterstitialAdDisplayFailed( MaxAdapterError.AD_NOT_READY );
        }
    }

    @Override
    public void loadRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading rewarded ad for placement: " + placementId );

        rewardedAdListener = new RewardedAdListener( listener );
        rewardedAd = createFullscreenAd( placementId, parameters, rewardedAdListener );

        rewardedAd.loadAd();
    }

    @Override
    public void showRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        log( "Showing rewarded ad" );

        if ( rewardedAd.isAdAvailable() )
        {
            // Configure reward from server.
            configureReward( parameters );

            rewardedAd.showAd();
        }
        else
        {
            log( "Rewarded ad not ready" );
            listener.onRewardedAdDisplayFailed( MaxAdapterError.AD_NOT_READY );
        }
    }

    //region Helper Methods

    private Placement createFullscreenAd(String placementId, MaxAdapterResponseParameters parameters, PlacementListener listener)
    {
        Placement fullscreenPlacement = HyprMX.INSTANCE.getPlacement( placementId );
        fullscreenPlacement.setPlacementListener( listener );

        return fullscreenPlacement;
    }

    private MaxAdapterError getAdapterError()
    {
        if ( HyprMX.INSTANCE.getInitializationState() != HyprMXState.INITIALIZATION_COMPLETE )
        {
            return MaxAdapterError.NOT_INITIALIZED;
        }
        else
        {
            return new MaxAdapterError( MaxAdapterError.NO_FILL.getErrorCode(), MaxAdapterError.NO_FILL.getErrorMessage(), HyprMXErrors.NO_FILL.ordinal(), HyprMXErrors.NO_FILL.name() );
        }
    }
    //endregion

    private class InterstitialListener
            implements PlacementListener
    {
        final MaxInterstitialAdapterListener listener;

        InterstitialListener(MaxInterstitialAdapterListener listener)
        {
            this.listener = listener;
        }

        @Override
        public void onAdAvailable(Placement placement)
        {
            log( "Interstitial ad loaded: " + placement.getName() );
            listener.onInterstitialAdLoaded();
        }

        @Override
        public void onAdNotAvailable(Placement placement)
        {
            log( "Interstitial failed to load: " + placement.getName() );
            listener.onInterstitialAdLoadFailed( getAdapterError() );
        }

        @Override
        public void onAdExpired(Placement placement)
        {
            log( "Interstitial expired: " + placement.getName() );
        }

        @Override
        public void onAdStarted(Placement placement)
        {
            log( "Interstitial did show: " + placement.getName() );
            listener.onInterstitialAdDisplayed();
        }

        @Override
        public void onAdClosed(Placement placement, boolean finished)
        {
            log( "Interstitial ad hidden with finished state: " + finished + " for placement: " + placement.getName() );
            listener.onInterstitialAdHidden();
        }

        @Override
        public void onAdDisplayError(Placement placement, HyprMXErrors hyprMXError)
        {
            log( "Interstitial failed to display with error: " + hyprMXError.ordinal() + ", for placement: " + placement.getName() );

            MaxAdapterError adapterError = new MaxAdapterError( MaxAdapterError.UNSPECIFIED.getErrorCode(), MaxAdapterError.UNSPECIFIED.getErrorMessage(), hyprMXError.ordinal(), hyprMXError.name() );
            listener.onInterstitialAdDisplayFailed( adapterError );
        }
    }

    private class RewardedAdListener
            implements PlacementListener, RewardedPlacementListener
    {
        final MaxRewardedAdapterListener listener;

        private boolean hasGrantedReward;

        RewardedAdListener(MaxRewardedAdapterListener listener)
        {
            this.listener = listener;
        }

        @Override
        public void onAdAvailable(Placement placement)
        {
            log( "Rewarded ad loaded: " + placement.getName() );
            listener.onRewardedAdLoaded();
        }

        @Override
        public void onAdNotAvailable(Placement placement)
        {
            log( "Rewarded ad failed to load: " + placement.getName() );
            listener.onRewardedAdLoadFailed( getAdapterError() );
        }

        @Override
        public void onAdExpired(Placement placement)
        {
            log( "Rewarded ad expired: " + placement.getName() );
        }

        @Override
        public void onAdStarted(Placement placement)
        {
            log( "Rewarded ad did show: " + placement.getName() );

            listener.onRewardedAdDisplayed();
            listener.onRewardedAdVideoStarted();
        }

        @Override
        public void onAdClosed(Placement placement, boolean finished)
        {
            listener.onRewardedAdVideoCompleted();

            if ( hasGrantedReward || shouldAlwaysRewardUser() )
            {
                final MaxReward reward = getReward();
                log( "Rewarded user with reward: " + reward );
                listener.onUserRewarded( reward );
            }

            log( "Rewarded ad hidden with finished state: " + finished + " for placement: " + placement.getName() );
            listener.onRewardedAdHidden();
        }

        @Override
        public void onAdDisplayError(Placement placement, HyprMXErrors hyprMXError)
        {
            log( "Rewarded ad failed to display with error: " + hyprMXError.ordinal() + ", for placement: " + placement.getName() );

            MaxAdapterError adapterError = new MaxAdapterError( MaxAdapterError.UNSPECIFIED.getErrorCode(), MaxAdapterError.UNSPECIFIED.getErrorMessage(), hyprMXError.ordinal(), hyprMXError.name() );
            listener.onRewardedAdDisplayFailed( adapterError );
        }

        @Override
        public void onAdRewarded(Placement placement, String rewardName, int rewardValue)
        {
            log( "Rewarded ad for placement: " + placement.getName() + " granted reward with rewardName: " + rewardName + " rewardValue: " + rewardValue );
            hasGrantedReward = true;
        }
    }
}
