package com.applovin.mediation.adapters;

import android.app.Activity;
import android.os.Bundle;
import android.text.TextUtils;

import com.applovin.mediation.MaxAdFormat;
import com.applovin.mediation.adapter.MaxAdViewAdapter;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.MaxInterstitialAdapter;
import com.applovin.mediation.adapter.MaxRewardedAdapter;
import com.applovin.mediation.adapter.MaxSignalProvider;
import com.applovin.mediation.adapter.listeners.MaxAdViewAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxRewardedAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxSignalCollectionListener;
import com.applovin.mediation.adapter.parameters.MaxAdapterInitializationParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterResponseParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterSignalCollectionParameters;
import com.applovin.mediation.adapters.mintegral.BuildConfig;
import com.applovin.sdk.AppLovinSdk;
import com.mbridge.msdk.MBridgeConstans;
import com.mbridge.msdk.MBridgeSDK;
import com.mbridge.msdk.interstitialvideo.out.InterstitialVideoListener;
import com.mbridge.msdk.interstitialvideo.out.MBBidInterstitialVideoHandler;
import com.mbridge.msdk.interstitialvideo.out.MBInterstitialVideoHandler;
import com.mbridge.msdk.mbbid.out.BidManager;
import com.mbridge.msdk.out.BannerAdListener;
import com.mbridge.msdk.out.BannerSize;
import com.mbridge.msdk.out.MBBannerView;
import com.mbridge.msdk.out.MBBidRewardVideoHandler;
import com.mbridge.msdk.out.MBConfiguration;
import com.mbridge.msdk.out.MBRewardVideoHandler;
import com.mbridge.msdk.out.MBridgeSDKFactory;
import com.mbridge.msdk.out.RewardVideoListener;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

public class MintegralMediationAdapter
        extends MediationAdapterBase
        implements MaxInterstitialAdapter, MaxRewardedAdapter, MaxAdViewAdapter, MaxSignalProvider
{
    private static final AtomicBoolean INITIALIZED = new AtomicBoolean();

    private static final String APP_ID_PARAMETER  = "app_id";
    private static final String APP_KEY_PARAMETER = "app_key";

    // Possible ad load error messages received from Mintegral in an email
    private final static String NOT_INITIALIZED  = "init error";
    private final static String NO_FILL_1        = "no ads available can show";
    private final static String NO_FILL_2        = "no ads available";
    private final static String NO_FILL_3        = "no server ads available";
    private final static String NO_FILL_4        = "no ads source";
    private final static String NETWORK_ERROR    = "network exception";
    private final static String BAD_REQUEST      = "request parameter is null";
    private final static String TIMEOUT          = "load timeout";
    private final static String UNIT_ID_EMPTY    = "UnitId is null";
    private final static String NETWORK_IO_ERROR = "Network error,I/O exception";

    // List of Mintegral error codes not defined in API, but in their docs
    //
    // http://cdn-adn.rayjump.com/cdn-adn/v2/markdown_v2/index.html?file=sdk-m_sdk-android&lang=en#faqs
    //
    private final static String EXCEPTION_RETURN_EMPTY            = "EXCEPTION_RETURN_EMPTY"; // ads no fill
    private final static String EXCEPTION_TIMEOUT                 = "EXCEPTION_TIMEOUT"; // request timeout
    private final static String EXCEPTION_IV_RECALLNET_INVALIDATE = "EXCEPTION_IV_RECALLNET_INVALIDATE"; // The network status at the time of the request is incorrect. Generally， because of the SDK initialization is not completed yet when the request has been sent.
    private final static String EXCEPTION_SIGN_ERROR              = "EXCEPTION_SIGN_ERROR"; // AppID and appKey do not match correctly
    private final static String EXCEPTION_UNIT_NOT_FOUND          = "EXCEPTION_UNIT_NOT_FOUND"; // Can not find the unitID in dashboard
    private final static String EXCEPTION_UNIT_ID_EMPTY           = "EXCEPTION_UNIT_ID_EMPTY"; // unitID is empty
    private final static String EXCEPTION_UNIT_NOT_FOUND_IN_APP   = "EXCEPTION_UNIT_NOT_FOUND_IN_APP"; // Can not find the unitID of the appID
    private final static String EXCEPTION_UNIT_ADTYPE_ERROR       = "EXCEPTION_UNIT_ADTYPE_ERROR"; // The adtype of the unitID is wrong
    private final static String EXCEPTION_APP_ID_EMPTY            = "EXCEPTION_APP_ID_EMPTY"; // appID is empty
    private final static String EXCEPTION_APP_NOT_FOUND           = "EXCEPTION_APP_NOT_FOUND"; // Can not find the appId

    private static String sSdkVersion;

    // Mintegral suggested we keep a map of unit id -> handler to prevent re-creation / high error rates - https://app.asana.com/0/573104092700345/1166998599374502
    private static final Map<String, MBInterstitialVideoHandler>    mbInterstitialVideoHandlers    = new HashMap<>();
    private static final Map<String, MBBidInterstitialVideoHandler> mbBidInterstitialVideoHandlers = new HashMap<>();
    private static final Map<String, MBRewardVideoHandler>          mbRewardVideoHandlers          = new HashMap<>();
    private static final Map<String, MBBidRewardVideoHandler>       mbBidRewardVideoHandlers       = new HashMap<>();

    // Supports video, interactive, and banner ad formats
    private MBInterstitialVideoHandler    mbInterstitialVideoHandler;
    private MBBidInterstitialVideoHandler mbBidInterstitialVideoHandler;
    private MBRewardVideoHandler          mbRewardVideoHandler;
    private MBBidRewardVideoHandler       mbBidRewardVideoHandler;
    private MBBannerView                  mbBannerView;

    // Explicit default constructor declaration
    public MintegralMediationAdapter(final AppLovinSdk sdk) { super( sdk ); }

    @Override
    public void initialize(final MaxAdapterInitializationParameters parameters, final Activity activity, final OnCompletionListener onCompletionListener)
    {
        MBridgeConstans.DEBUG = parameters.isTesting();

        if ( INITIALIZED.compareAndSet( false, true ) )
        {
            final String appId = parameters.getServerParameters().getString( APP_ID_PARAMETER );
            final String appKey = parameters.getServerParameters().getString( APP_KEY_PARAMETER );
            log( "Initializing Mintegral SDK with app id: " + appId + " and app key: " + appKey + "..." );

            final MBridgeSDK mBridgeSDK = MBridgeSDKFactory.getMBridgeSDK();

            // Communicated over email, GDPR status can only be set before SDK initialization
            Boolean hasUserConsent = getPrivacySetting( "hasUserConsent", parameters );
            if ( hasUserConsent != null )
            {
                int consent = hasUserConsent ? MBridgeConstans.IS_SWITCH_ON : MBridgeConstans.IS_SWITCH_OFF;
                mBridgeSDK.setUserPrivateInfoType( activity, MBridgeConstans.AUTHORITY_ALL_INFO, consent );
                mBridgeSDK.setConsentStatus( activity, consent );
            }

            // Has to be _before_ their SDK init as well
            if ( AppLovinSdk.VERSION_CODE >= 91100 )
            {
                Boolean isDoNotSell = getPrivacySetting( "isDoNotSell", parameters );
                if ( isDoNotSell != null && isDoNotSell )
                {
                    mBridgeSDK.setDoNotTrackStatus( true );
                }
            }

            // Mintegral Docs - "It is recommended to use the API in the main thread"
            final Map<String, String> map = mBridgeSDK.getMBConfigurationMap( appId, appKey );
            mBridgeSDK.init( map, activity );
        }

        onCompletionListener.onCompletion( InitializationStatus.DOES_NOT_APPLY, null );
    }

    @Override
    public String getSdkVersion()
    {
        // Do not use `MBConfiguration.SDK_VERSION` as this will hardcode the version when the adapter is built
        if ( sSdkVersion == null )
        {
            sSdkVersion = getVersionString( MBConfiguration.class, "SDK_VERSION" );
        }

        return sSdkVersion;
    }

    @Override
    public String getAdapterVersion()
    {
        return BuildConfig.VERSION_NAME;
    }

    @Override
    public void collectSignal(final MaxAdapterSignalCollectionParameters parameters, final Activity activity, final MaxSignalCollectionListener callback)
    {
        // DOCS: https://app.asana.com/0/20387143076905/1126882455204927
        log( "Collecting signal..." );

        //  "3.2.5 - This step is only required for developers using the server to server bidding architecture"
        final String signal = BidManager.getBuyerUid( activity );
        callback.onSignalCollected( signal );
    }

    @Override
    public void loadInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        InterstitialVideoListener adListener = new InterstitialVideoListener()
        {
            @Override
            public void onVideoLoadSuccess(final String placementId, final String unitId)
            {
                // Ad has loaded and video has been downloaded
                log( "Interstitial successfully loaded and video has been downloaded" );

                String requestId;
                if ( mbBidInterstitialVideoHandler != null )
                {
                    requestId = mbBidInterstitialVideoHandler.getRequestId();
                }
                else
                {
                    requestId = mbInterstitialVideoHandler.getRequestId();
                }

                // Passing extra info such as creative id supported in 9.15.0+
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( requestId ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", requestId );

                    listener.onInterstitialAdLoaded( extraInfo );
                }
                else
                {
                    listener.onInterstitialAdLoaded();
                }
            }

            @Override
            public void onLoadSuccess(final String placementId, final String unitId)
            {
                // Ad has loaded but video still needs to be downloaded
                log( "Interstitial successfully loaded but video still needs to be downloaded" );
            }

            @Override
            public void onVideoLoadFail(final String errorMsg)
            {
                log( "Interstitial failed to load: " + errorMsg );
                listener.onInterstitialAdLoadFailed( toMaxError( errorMsg ) );
            }

            @Override
            public void onAdShow()
            {
                log( "Interstitial displayed" );
                listener.onInterstitialAdDisplayed();
            }

            @Override
            public void onShowFail(final String errorMsg)
            {
                log( "Interstitial failed to show: " + errorMsg );
                listener.onInterstitialAdDisplayFailed( toMaxError( errorMsg ) );
            }

            @Override
            public void onVideoAdClicked(final String placementId, final String unitId)
            {
                log( "Interstitial clicked" );
                listener.onInterstitialAdClicked();
            }

            @Override
            public void onAdClose(final boolean isCompleteView)
            {
                log( "Interstitial hidden" );
                listener.onInterstitialAdHidden();
            }

            @Override
            public void onVideoComplete(final String placementId, final String unitId)
            {
                log( "Interstitial video completed" );
            }

            @Override
            public void onAdCloseWithIVReward(final boolean isComplete, final int rewardAlertStatus)
            {
                // This is the ad close callback for when the interstitial has a reward.
            }

            @Override
            public void onEndcardShow(final String placementId, final String unitId)
            {
                log( "Interstitial endcard shown" );
            }
        };

        final boolean shouldUpdateMuteState = parameters.getServerParameters().containsKey( "is_muted" ); // Introduced in 9.10.0
        final int muteState = parameters.getServerParameters().getBoolean( "is_muted" ) ? MBridgeConstans.REWARD_VIDEO_PLAY_MUTE : MBridgeConstans.INTER_ACTIVE_VIDEO_PLAY_NOT_MUTE;

        final String unitId = parameters.getThirdPartyAdPlacementId();
        final String placementId = parameters.getServerParameters().getString( "placement_id" );

        if ( !TextUtils.isEmpty( parameters.getBidResponse() ) )
        {
            log( "Loading bidding interstitial ad for unit id: " + unitId + " and placement id: " + placementId + "..." );

            if ( mbBidInterstitialVideoHandlers.containsKey( unitId ) )
            {
                mbBidInterstitialVideoHandler = mbBidInterstitialVideoHandlers.get( unitId );
            }
            else
            {
                mbBidInterstitialVideoHandler = new MBBidInterstitialVideoHandler( activity, placementId, unitId );
                mbBidInterstitialVideoHandlers.put( unitId, mbBidInterstitialVideoHandler );
            }

            mbBidInterstitialVideoHandler.setInterstitialVideoListener( adListener );

            if ( mbBidInterstitialVideoHandler.isBidReady() )
            {
                log( "A bidding interstitial ad is ready already" );

                // Passing extra info such as creative id supported in 9.15.0+
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( mbBidInterstitialVideoHandler.getRequestId() ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", mbBidInterstitialVideoHandler.getRequestId() );

                    listener.onInterstitialAdLoaded( extraInfo );
                }
                else
                {
                    listener.onInterstitialAdLoaded();
                }
            }
            else
            {
                // Update mute state if configured by backend
                if ( shouldUpdateMuteState ) mbBidInterstitialVideoHandler.playVideoMute( muteState );

                mbBidInterstitialVideoHandler.loadFromBid( parameters.getBidResponse() );
            }
        }
        else
        {
            log( "Loading mediated interstitial ad for unit id: " + unitId + " and placement id: " + placementId + "..." );

            if ( mbInterstitialVideoHandlers.containsKey( unitId ) )
            {
                mbInterstitialVideoHandler = mbInterstitialVideoHandlers.get( unitId );
            }
            else
            {
                mbInterstitialVideoHandler = new MBInterstitialVideoHandler( activity, placementId, unitId );
                mbInterstitialVideoHandlers.put( unitId, mbInterstitialVideoHandler );
            }

            mbInterstitialVideoHandler.setInterstitialVideoListener( adListener );

            if ( mbInterstitialVideoHandler.isReady() )
            {
                log( "A mediated interstitial ad is ready already" );

                // Passing extra info such as creative id supported in 9.15.0+
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( mbInterstitialVideoHandler.getRequestId() ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", mbInterstitialVideoHandler.getRequestId() );

                    listener.onInterstitialAdLoaded( extraInfo );
                }
                else
                {
                    listener.onInterstitialAdLoaded();
                }
            }
            else
            {
                // Update mute state if configured by backend
                if ( shouldUpdateMuteState ) mbInterstitialVideoHandler.playVideoMute( muteState );

                mbInterstitialVideoHandler.load();
            }
        }
    }

    @Override
    public void showInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        if ( mbBidInterstitialVideoHandler != null && mbBidInterstitialVideoHandler.isBidReady() )
        {
            log( "Showing bidding interstitial..." );
            mbBidInterstitialVideoHandler.showFromBid();
        }
        else if ( mbInterstitialVideoHandler != null && mbInterstitialVideoHandler.isReady() )
        {
            log( "Showing mediated interstitial..." );
            mbInterstitialVideoHandler.show();
        }
        else
        {
            log( "Unable to show interstitial - no ad loaded..." );

            // Ad load failed
            listener.onInterstitialAdDisplayFailed( MaxAdapterError.AD_NOT_READY );
        }
    }

    @Override
    public void loadRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        RewardVideoListener adListener = new RewardVideoListener()
        {
            @Override
            public void onVideoLoadSuccess(final String placementId, final String unitId)
            {
                // Ad has loaded and video has been downloaded
                log( "Rewarded ad successfully loaded and video has been downloaded" );

                String requestId;
                if ( mbBidRewardVideoHandler != null )
                {
                    requestId = mbBidRewardVideoHandler.getRequestId();
                }
                else
                {
                    requestId = mbRewardVideoHandler.getRequestId();
                }

                // Passing extra info such as creative id supported in 9.15.0+
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( requestId ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", requestId );

                    listener.onRewardedAdLoaded( extraInfo );
                }
                else
                {
                    listener.onRewardedAdLoaded();
                }
            }

            @Override
            public void onLoadSuccess(final String placementId, final String unitId)
            {
                // Ad has loaded but video still needs to be downloaded
                log( "Rewarded ad successfully loaded but video still needs to be downloaded" );
            }

            @Override
            public void onVideoLoadFail(final String errorMsg)
            {
                log( "Rewarded ad failed to load: " + errorMsg );
                listener.onRewardedAdLoadFailed( toMaxError( errorMsg ) );
            }

            @Override
            public void onAdShow()
            {
                log( "Rewarded ad displayed" );
                listener.onRewardedAdDisplayed();
                listener.onRewardedAdVideoStarted();
            }

            @Override
            public void onShowFail(final String errorMsg)
            {
                log( "Rewarded ad failed to show: " + errorMsg );
                listener.onRewardedAdDisplayFailed( toMaxError( errorMsg ) );
            }

            @Override
            public void onVideoAdClicked(final String placementId, final String unitId)
            {
                log( "Rewarded ad clicked" );
                listener.onRewardedAdClicked();
            }

            @Override
            public void onAdClose(final boolean isCompleteView, final String rewardName, final float rewardAmount)
            {
                log( "Rewarded ad hidden" );

                if ( isCompleteView )
                {
                    listener.onRewardedAdVideoCompleted();
                    listener.onUserRewarded( getReward() );
                }
                else if ( shouldAlwaysRewardUser() )
                {
                    listener.onUserRewarded( getReward() );
                }

                listener.onRewardedAdHidden();
            }

            @Override
            public void onVideoComplete(final String placementId, final String unitId)
            {
                log( "Rewarded ad video completed" );
            }

            @Override
            public void onEndcardShow(final String placementId, final String unitId)
            {
                log( "Rewarded ad endcard shown" );
            }
        };

        final boolean shouldUpdateMuteState = parameters.getServerParameters().containsKey( "is_muted" ); // Introduced in 9.10.0
        final int muteState = parameters.getServerParameters().getBoolean( "is_muted" ) ? MBridgeConstans.REWARD_VIDEO_PLAY_MUTE : MBridgeConstans.INTER_ACTIVE_VIDEO_PLAY_NOT_MUTE;

        final String unitId = parameters.getThirdPartyAdPlacementId();
        final String placementId = parameters.getServerParameters().getString( "placement_id" );

        if ( !TextUtils.isEmpty( parameters.getBidResponse() ) )
        {
            log( "Loading bidding rewarded ad for unit id: " + unitId + " and placement id: " + placementId + "..." );

            if ( mbBidRewardVideoHandlers.containsKey( unitId ) )
            {
                mbBidRewardVideoHandler = mbBidRewardVideoHandlers.get( unitId );
            }
            else
            {
                mbBidRewardVideoHandler = new MBBidRewardVideoHandler( activity, placementId, unitId );
                mbBidRewardVideoHandlers.put( unitId, mbBidRewardVideoHandler );
            }

            mbBidRewardVideoHandler.setRewardVideoListener( adListener );

            if ( mbBidRewardVideoHandler.isBidReady() )
            {
                log( "A bidding rewarded ad is ready already" );

                // Passing extra info such as creative id supported in 9.15.0+
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( mbBidRewardVideoHandler.getRequestId() ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", mbBidRewardVideoHandler.getRequestId() );

                    listener.onRewardedAdLoaded( extraInfo );
                }
                else
                {
                    listener.onRewardedAdLoaded();
                }
            }
            else
            {
                // Update mute state if configured by backend
                if ( shouldUpdateMuteState ) mbBidRewardVideoHandler.playVideoMute( muteState );

                mbBidRewardVideoHandler.loadFromBid( parameters.getBidResponse() );
            }
        }
        else
        {
            log( "Loading mediated rewarded ad for unit id: " + unitId + " and placement id: " + placementId + "..." );

            if ( mbRewardVideoHandlers.containsKey( unitId ) )
            {
                mbRewardVideoHandler = mbRewardVideoHandlers.get( unitId );
            }
            else
            {
                mbRewardVideoHandler = new MBRewardVideoHandler( activity, placementId, unitId );
                mbRewardVideoHandlers.put( unitId, mbRewardVideoHandler );
            }

            mbRewardVideoHandler.setRewardVideoListener( adListener );

            if ( mbRewardVideoHandler.isReady() )
            {
                log( "A mediated rewarded ad is ready already" );

                // Passing extra info such as creative id supported in 9.15.0+
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( mbRewardVideoHandler.getRequestId() ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", mbRewardVideoHandler.getRequestId() );

                    listener.onRewardedAdLoaded( extraInfo );
                }
                else
                {
                    listener.onRewardedAdLoaded();
                }
            }
            else
            {

                // Update mute state if configured by backend
                if ( shouldUpdateMuteState ) mbRewardVideoHandler.playVideoMute( muteState );

                mbRewardVideoHandler.load();
            }
        }
    }

    @Override
    public void showRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        // Configure userReward from server.
        configureReward( parameters );

        final Bundle serverParameters = parameters.getServerParameters();
        final String rewardId = serverParameters.getString( "reward_id", "" );
        final String userId = serverParameters.getString( "user_id", "" );

        if ( mbBidRewardVideoHandler != null && mbBidRewardVideoHandler.isBidReady() )
        {
            log( "Showing bidding rewarded ad..." );
            mbBidRewardVideoHandler.showFromBid( rewardId, userId );
        }
        else if ( mbRewardVideoHandler != null && mbRewardVideoHandler.isReady() )
        {
            log( "Showing mediated rewarded ad..." );
            mbRewardVideoHandler.show( rewardId, userId );
        }
        else
        {
            log( "Unable to show rewarded ad - no ad loaded..." );

            // Ad load failed
            listener.onRewardedAdDisplayFailed( MaxAdapterError.AD_NOT_READY );
        }
    }

    @Override
    public void loadAdViewAd(final MaxAdapterResponseParameters parameters, final MaxAdFormat adFormat, final Activity activity, final MaxAdViewAdapterListener listener)
    {
        BannerSize size = toBannerSize( adFormat );

        final String unitId = parameters.getThirdPartyAdPlacementId();
        final String placementId = parameters.getServerParameters().getString( "placement_id" );

        mbBannerView = new MBBannerView( activity );
        mbBannerView.init( size, placementId, unitId );
        mbBannerView.setAllowShowCloseBtn( false );
        mbBannerView.setRefreshTime( 0 );

        mbBannerView.setBannerAdListener( new BannerAdListener()
        {
            @Override
            public void onLoadSuccessed()
            {
                log( "Banner ad loaded" );

                // Passing extra info such as creative id supported in 9.15.0+
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( mbBannerView.getRequestId() ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", mbBannerView.getRequestId() );

                    listener.onAdViewAdLoaded( mbBannerView, extraInfo );
                }
                else
                {
                    listener.onAdViewAdLoaded( mbBannerView );
                }
            }

            @Override
            public void onLoadFailed(final String msg)
            {
                log( "Banner ad failed to load: " + msg );
                listener.onAdViewAdLoadFailed( toMaxError( msg ) );
            }

            @Override
            public void onLogImpression()
            {
                log( "Banner ad displayed" );
                listener.onAdViewAdDisplayed();
            }

            @Override
            public void onClick()
            {
                log( "Banner ad clicked" );
                listener.onAdViewAdClicked();
            }

            @Override
            public void onLeaveApp()
            {
                log( "Banner ad will leave application" );
            }

            @Override
            public void showFullScreen()
            {
                log( "Banner ad expanded" );
                listener.onAdViewAdExpanded();
            }

            @Override
            public void closeFullScreen()
            {
                log( "Banner ad collapsed" );
                listener.onAdViewAdCollapsed();
            }

            @Override
            public void onCloseBanner()
            {
                log( "Banner ad closed" );
            }
        } );

        if ( !TextUtils.isEmpty( parameters.getBidResponse() ) )
        {
            log( "Loading bidding banner ad for unit id: " + unitId + " and placement id: " + placementId + "..." );
            mbBannerView.loadFromBid( parameters.getBidResponse() );
        }
        else
        {
            log( "Loading mediated banner ad for unit id: " + unitId + " and placement id: " + placementId + "..." );
            mbBannerView.load();
        }
    }

    @Override
    public void onDestroy()
    {
        if ( mbInterstitialVideoHandler != null )
        {
            mbInterstitialVideoHandler.setInterstitialVideoListener( null );
            mbInterstitialVideoHandler = null;
        }

        if ( mbBidInterstitialVideoHandler != null )
        {
            mbBidInterstitialVideoHandler.setInterstitialVideoListener( null );
            mbBidInterstitialVideoHandler = null;
        }

        if ( mbRewardVideoHandler != null )
        {
            mbRewardVideoHandler.setRewardVideoListener( null );
            mbRewardVideoHandler = null;
        }

        if ( mbBidRewardVideoHandler != null )
        {
            mbBidRewardVideoHandler.setRewardVideoListener( null );
            mbBidRewardVideoHandler = null;
        }

        if ( mbBannerView != null )
        {
            mbBannerView.release();
            mbBannerView = null;
        }
    }

    private Boolean getPrivacySetting(final String privacySetting, final MaxAdapterParameters parameters)
    {
        try
        {
            // Use reflection because compiled adapters have trouble fetching `boolean` from old SDKs and `Boolean` from new SDKs (above 9.14.0)
            Class<?> parametersClass = parameters.getClass();
            Method privacyMethod = parametersClass.getMethod( privacySetting );
            return (Boolean) privacyMethod.invoke( parameters );
        }
        catch ( Exception exception )
        {
            log( "Error getting privacy setting " + privacySetting + " with exception: ", exception );
            return ( AppLovinSdk.VERSION_CODE >= 9140000 ) ? null : false;
        }
    }

    private static MaxAdapterError toMaxError(final String mintegralError)
    {
        // Note: we are using `contains()` in some cases b/c Mintegral prepends the message with `data load failed, errorMsg is `...

        final MaxAdapterError adapterError;
        if ( NOT_INITIALIZED.equals( mintegralError ) || mintegralError.contains( EXCEPTION_IV_RECALLNET_INVALIDATE ) )
        {
            adapterError = MaxAdapterError.NOT_INITIALIZED;
        }
        else if ( NO_FILL_1.contains( mintegralError ) || NO_FILL_2.contains( mintegralError )
                || NO_FILL_3.contains( mintegralError ) || NO_FILL_4.contains( mintegralError ) || mintegralError.contains( EXCEPTION_RETURN_EMPTY ) )
        {
            adapterError = MaxAdapterError.NO_FILL;
        }
        else if ( NETWORK_ERROR.equalsIgnoreCase( mintegralError ) || mintegralError.contains( NETWORK_IO_ERROR ) )
        {
            adapterError = MaxAdapterError.NO_CONNECTION;
        }
        else if ( BAD_REQUEST.equalsIgnoreCase( mintegralError ) )
        {
            adapterError = MaxAdapterError.BAD_REQUEST;
        }
        else if ( TIMEOUT.equalsIgnoreCase( mintegralError ) || mintegralError.contains( EXCEPTION_TIMEOUT ) )
        {
            adapterError = MaxAdapterError.TIMEOUT;
        }
        else if ( mintegralError.contains( EXCEPTION_SIGN_ERROR ) || mintegralError.contains( EXCEPTION_UNIT_NOT_FOUND ) || mintegralError.contains( EXCEPTION_UNIT_ID_EMPTY ) || mintegralError.contains( EXCEPTION_UNIT_NOT_FOUND_IN_APP ) || mintegralError.contains( EXCEPTION_UNIT_ADTYPE_ERROR ) || mintegralError.contains( EXCEPTION_APP_ID_EMPTY ) || mintegralError.contains( EXCEPTION_APP_NOT_FOUND ) || mintegralError.contains( UNIT_ID_EMPTY ) )
        {
            adapterError = MaxAdapterError.INVALID_CONFIGURATION;
        }
        else
        {
            adapterError = MaxAdapterError.UNSPECIFIED;
        }

        return new MaxAdapterError( adapterError.getErrorCode(), adapterError.getErrorMessage(), 0, mintegralError );
    }

    private BannerSize toBannerSize(final MaxAdFormat adFormat)
    {
        if ( adFormat == MaxAdFormat.BANNER || adFormat == MaxAdFormat.LEADER )
        {
            // Last two parameters are for custom width and height, so we can just use 0.
            return new BannerSize( BannerSize.SMART_TYPE, 0, 0 );
        }
        else if ( adFormat == MaxAdFormat.MREC )
        {
            return new BannerSize( BannerSize.MEDIUM_TYPE, 0, 0 );
        }
        else
        {
            throw new IllegalArgumentException( "Unsupported ad format: " + adFormat );
        }
    }
}