package com.hyprmx.android.sdk.activity

import android.annotation.SuppressLint
import android.app.AlertDialog
import android.content.Context
import android.content.Intent
import android.content.pm.ActivityInfo
import android.content.res.Configuration
import android.graphics.Color
import android.os.Build
import android.os.Bundle
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.widget.RelativeLayout
import androidx.annotation.CallSuper
import androidx.annotation.RequiresApi
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.lifecycle.LifecycleObserver
import com.hyprmx.android.R
import com.hyprmx.android.sdk.api.data.Ad
import com.hyprmx.android.sdk.api.data.Orientation.ANY
import com.hyprmx.android.sdk.api.data.Orientation.LANDSCAPE
import com.hyprmx.android.sdk.api.data.Orientation.PORTRAIT
import com.hyprmx.android.sdk.api.data.PresentDialog
import com.hyprmx.android.sdk.core.HyprMXReInit
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.fullscreen.FullScreenPresenterAdapter
import com.hyprmx.android.sdk.fullscreen.FullScreenPresenterInterface
import com.hyprmx.android.sdk.fullscreen.FullScreenSharedInterface
import com.hyprmx.android.sdk.jsAlertDialog.JSAlertDialogContract
import com.hyprmx.android.sdk.jsAlertDialog.JSAlertDialogPresenter
import com.hyprmx.android.sdk.jsAlertDialog.JSAlertDialogView
import com.hyprmx.android.sdk.jsinterface.AppJSDelegate
import com.hyprmx.android.sdk.mvp.LifecycleEvent
import com.hyprmx.android.sdk.mvp.LifecycleEventAdapter
import com.hyprmx.android.sdk.mvp.LifecycleEventHandler
import com.hyprmx.android.sdk.network.NetworkConnectionListener
import com.hyprmx.android.sdk.network.NetworkConnectionMonitor
import com.hyprmx.android.sdk.om.OpenMeasurementController
import com.hyprmx.android.sdk.overlay.HyprMXOverlay
import com.hyprmx.android.sdk.overlay.HyprMXOverlayAdapter
import com.hyprmx.android.sdk.overlay.ImageCapture
import com.hyprmx.android.sdk.overlay.ImageCapturer
import com.hyprmx.android.sdk.powersavemode.PowerSaveModeListener
import com.hyprmx.android.sdk.presentation.ActivityResultListener
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.utility.DetachableClickListener
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.InternetConnectionDialog
import com.hyprmx.android.sdk.utility.PermissionResponse
import com.hyprmx.android.sdk.utility.PermissionResult
import com.hyprmx.android.sdk.utility.Utils
import com.hyprmx.android.sdk.utility.convertDpToPixel
import com.hyprmx.android.sdk.utility.convertPixelsToDp
import com.hyprmx.android.sdk.utility.toArrayList
import com.hyprmx.android.sdk.webview.HyprMXWebView
import com.hyprmx.android.sdk.webview.WebViewFactoryIf
import com.iab.omid.library.jungroup.adsession.FriendlyObstructionPurpose
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancelChildren
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.json.JSONArray
import org.json.JSONException
import kotlin.coroutines.CoroutineContext
import kotlin.math.floor

/**
 * This class provides the basic implementation of creating a layout to display on an activity.
 */
internal abstract class HyprMXBaseViewController(
  val activity: AppCompatActivity,
  private val savedInstanceState: Bundle?,
  val hyprMXBaseViewControllerListener: HyprMXBaseViewControllerListener,
  private val activityResultListener: ActivityResultListener,
  val powerSaveMode: PowerSaveModeListener,
  val webViewFactory: WebViewFactoryIf,
  val openMeasurementController: OpenMeasurementController?,
  protected val baseAd: Ad,
  private val scope: CoroutineScope,
  private var mainDispatcher: CoroutineDispatcher = Dispatchers.Main,
  val networkConnectionMonitor: NetworkConnectionMonitor,
  val internetConnectionDialog: InternetConnectionDialog,
  private val parentJob: Job? = scope.coroutineContext[Job],
  private val job: Job = SupervisorJob(parentJob),
  eventPublisher: PresentationEventPublisher,
  lifecycleEventAdapter: LifecycleEventHandler = LifecycleEventAdapter(eventPublisher, scope),
  private val hyprMXOverlay: HyprMXOverlay = HyprMXOverlayAdapter(
    overlayContext = activity,
    monitorPlatformOverlay = true,
  ),
  imageCapturer: ImageCapture = ImageCapturer(),
  val fullScreenSharedConnector: FullScreenSharedInterface,
) :
  ViewTreeObserver.OnGlobalLayoutListener,
  LifecycleObserver,
  JSAlertDialogContract.AdDialogPresenter,
  JSAlertDialogContract.WebViewPresenter,
  CoroutineScope,
  InternetConnectionDialog by internetConnectionDialog,
  NetworkConnectionListener,
  LifecycleEventHandler by lifecycleEventAdapter,
  PermissionResponse,
  ImageCapture by imageCapturer,
  FullScreenPresenterInterface,
  FullScreenSharedInterface by fullScreenSharedConnector,
  HyprMXReInit {

  override val coroutineContext: CoroutineContext
    get() {
      return job + mainDispatcher + CoroutineName("HyprMXBaseViewController")
    }

  private lateinit var layout: RelativeLayout
  private lateinit var adViewLayout: RelativeLayout.LayoutParams
  var jsAlertDialogPresenter: JSAlertDialogContract.Presenter =
    JSAlertDialogPresenter(JSAlertDialogView(), this, this)

  var webView: HyprMXWebView

  init {
    attach(FullScreenPresenterAdapter(this, this))
    hyprmxDelegate.hyprMXController?.registerSDKReInitListener(this)
    webView = webViewFactory.createWebView(viewModelIdentifier, this.baseAd.userAgent).apply {
      this.containingActivity = activity
      this.initialize(viewModelIdentifier, this@HyprMXBaseViewController.baseAd.userAgent)
    }
  }

  var adFinished = false
  var offerCancelAlertDialog: AlertDialog? = null

  private var containerWidth: Int = -1
  private var containerHeight: Int = -1

  private var appJSHandler: AppJSDelegate? = null

  private var activityResultListenerCalled = false

  private var errorDialog: AlertDialog? = null

  val context: Context
    get() = activity.baseContext

  val rootLayout: ViewGroup
    get() {
      return layout
    }

  val rootLayoutParams: RelativeLayout.LayoutParams
    get() {
      return adViewLayout
    }

  abstract fun getOfferRootLayout(): ViewGroup

  interface HyprMXBaseViewControllerListener {
    fun onSetRequestedOrientation(requestedOrientation: Int)
  }

  @CallSuper
  open fun onCreate() {
    onLifecycleEvent(LifecycleEvent.ON_CREATE)

    setupLayout()
    setupOrientation()

    // either load the thank you page if available or exit with ad cancel.
    if (savedInstanceState != null) {
      restoreState(savedInstanceState)
    } else {
      loadAd()
    }
  }

  @CallSuper
  open fun loadAd() {
  }

  private fun setupOrientation() {
    when (baseAd.allowedOrientation) {
      PORTRAIT -> hyprMXBaseViewControllerListener.onSetRequestedOrientation(
        ActivityInfo.SCREEN_ORIENTATION_PORTRAIT,
      )
      LANDSCAPE -> hyprMXBaseViewControllerListener.onSetRequestedOrientation(
        ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE,
      )
      ANY -> {
      }
    } // don't lock in orientation
  }

  @CallSuper
  open fun setupLayout() {
    layout = RelativeLayout(activity)
    layout.id = R.id.hyprmx_root_layout
    layout.setBackgroundColor(Color.BLACK)
    adViewLayout = RelativeLayout.LayoutParams(
      RelativeLayout.LayoutParams.MATCH_PARENT,
      RelativeLayout.LayoutParams.MATCH_PARENT,
    )
    adViewLayout.addRule(RelativeLayout.CENTER_IN_PARENT)
    activity.setContentView(layout, adViewLayout)
  }

  open fun onBackPressed() {
    onBackButtonPressed()
  }

  @CallSuper
  open fun onPause() {
    onLifecycleEvent(LifecycleEvent.ON_PAUSE)
    webView.pauseJSExecution()
  }

  @CallSuper
  open fun onResume() {
    onLifecycleEvent(LifecycleEvent.ON_RESUME)
    containerVisibilityChanged(true)
    hyprMXOverlay.isOverlayPresented = false
    webView.resumeJSExecution()
  }

  open fun onStart() {
    onLifecycleEvent(LifecycleEvent.ON_START)
    if (!payoutComplete) {
      networkConnectionMonitor.startMonitoring(this)
    }
    layout.viewTreeObserver.addOnGlobalLayoutListener(this)
  }

  @SuppressLint("NewApi")
  open fun onStop() {
    onLifecycleEvent(LifecycleEvent.ON_STOP)
    containerVisibilityChanged(false)
    networkConnectionMonitor.stopMonitoring(this)
    layout.viewTreeObserver.removeOnGlobalLayoutListener(this)
    if (adFinished) notifyAdResultListener()
  }

  fun notifyAdResultListener() {
    launch {
      if (!activityResultListenerCalled) {
        if (payoutComplete) {
          activityResultListener.onAdRewarded()
        }
        activityResultListener.onAdDismissed(adCompleted)
        activityResultListenerCalled = true
      }
    }
  }

  @CallSuper
  open fun onDestroy() {
    onLifecycleEvent(LifecycleEvent.ON_DESTROY)
    jsAlertDialogPresenter.dismissDialog()
    offerCancelAlertDialog?.dismiss()
    internetConnectionDialog.dismissInternetConnectionErrorDialog()

    // This is for cases where a publishers activity is set to launch in singleTask mode.
    // In this case finishWithResult is never called and thus the javascript is never notified that ad finished.
    notifyAdResultListener()

    appJSHandler?.appJSHandler = null
    appJSHandler = null

    launch {
      destroy()

      // Open Measurement requires us to hold the webview open for 1s before destroying it
      delay(1_000)

      cleanupWebView()
      job.cancelChildren()
    }
  }

  private fun cleanupWebView() {
    if (webView.parent != null) {
      rootLayout.removeView(webView)
    }
    webView.cleanup()
  }

  /**
   * Sets resultCode and then closes the activity.
   */
  @CallSuper
  open fun finishWithResult() {
    onClose()
    adFinished = true
    openMeasurementController?.endSession()
    activity.finish()
  }

  @CallSuper
  open fun restoreState(savedInstanceState: Bundle) {
  }

  /**
   * Called when an activity you launched exits.
   *
   * @param requestCode The integer request code originally supplied to
   * startActivityForResult(), allowing you to identify who this
   * result came from.
   * @param resultCode The integer result code returned by the child activity
   * through its setResult().
   * @param data An Intent, which can return result data to the caller
   */
  open fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
    launch {
      processActivityResult(
        activity.applicationContext,
        requestCode,
        resultCode,
        data,
        this@HyprMXBaseViewController,
      )
    }
  }

  /**
   * Notifies the host application that an Android permission has been granted.
   */
  override fun onPermissionResponse(permissionResults: List<PermissionResult>, requestCode: Int) {
    HyprMXLog.d("onPermissionResponse - $requestCode")
    permissionResult(permissionResults, requestCode)
  }

  override suspend fun reportPowerState(): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("Reporting power state to webview")
    powerSaveMode.sendPowerStateEvent(webView)
  }

  override suspend fun dismissOfferCancellationDialog(): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("dismissOfferCancellationDialog")
    offerCancelAlertDialog?.dismiss()
    offerCancelAlertDialog = null
  }

  override suspend fun setClosable(closable: Boolean): Unit = withContext(Dispatchers.Main) {
    // Does nothing
  }

  override suspend fun showDialog(jsonConfiguration: String): Unit =
    withContext(Dispatchers.Main) {
      HyprMXLog.d("showDialog")

      val presentDialog: PresentDialog = try {
        PresentDialog.fromJsonString(jsonConfiguration)
      } catch (e: JSONException) {
        HyprMXLog.e(e.message)
        return@withContext
      }

      if (!activity.isFinishing) {
        jsAlertDialogPresenter.createAndShowDialog(activity, presentDialog)
      }
    }

  @CallSuper
  override suspend fun startOMSession(sessionData: String): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("startOMSession")
    openMeasurementController?.startJavascriptOMSession(sessionData, webView.webView)
  }

  override suspend fun endOMSession(): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("endOMSession")
    openMeasurementController?.endSession()
  }

  override suspend fun showCancelDialog(
    message: String,
    exitButton: String,
    continueButton: String,
  ) {
    HyprMXLog.d("Displaying offerCancelAlertDialog")
    val clickListener = DetachableClickListener.wrap { dialog, _ ->
      if (offerCancelAlertDialog?.isShowing == true) {
        dialog.dismiss()
      }

      this@HyprMXBaseViewController.launch {
        cancelDialogExitPressed()
      }
    }

    val dialogBuilder = AlertDialog.Builder(activity)
    offerCancelAlertDialog = dialogBuilder
      .setMessage(message)
      .setPositiveButton(continueButton, null)
      .setNegativeButton(exitButton, clickListener)
      .setCancelable(true)
      .setOnCancelListener { it.dismiss() }
      .create().apply {
        setCanceledOnTouchOutside(true)
        if (!activity.isFinishing) {
          HyprMXLog.d("Displaying offerCancelAlertDialog")
          show()
        } else {
          HyprMXLog.d("Not displaying offerCancelAlertDialog because activity is finishing")
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
          clickListener.clearOnDetach(this)
        }
      }
  }

  override fun handleInternetConnectivityChange(active: Boolean) {
    if (!active) {
      HyprMXLog.d("No internet connection detected.")
      internetLossDetected()
    }
  }

  /**
   * Displays network error dialog with OK button.
   * Calls page ready timeout to be removed.
   */
  @CallSuper
  open fun showNetworkErrorDialog() {
    showInternetConnectionErrorDialog(activity) {
      onErrorDialogOKPressed()
    }
  }

  @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  override suspend fun showErrorDialog(message: String): Unit = withContext(Dispatchers.Main) {
    val clickListener = DetachableClickListener.wrap { dialog, _ ->
      dialog.dismiss()
      onErrorDialogOKPressed()
    }

    errorDialog = AlertDialog.Builder(activity)
      .setMessage(message)
      .setNegativeButton(activity.getString(android.R.string.ok), clickListener)
      .setCancelable(false)
      .create().apply {
        setCanceledOnTouchOutside(false)
        if (!activity.isFinishing) {
          show()
        }
        clickListener.clearOnDetach(this)
      }
  }

  /**
   * Called when ad container layout state changes.
   */
  override fun onGlobalLayout() {
    val newContainerWidth = rootLayout.width
    val newContainerHeight = rootLayout.height
    if (containerHeight != newContainerHeight || containerWidth != newContainerWidth) {
      containerHeight = newContainerHeight
      containerWidth = newContainerWidth
      containerSizedChanged(
        width = floor(containerWidth.convertPixelsToDp(context)).toInt(),
        height = floor(containerHeight.convertPixelsToDp(context)).toInt(),
      )
    }
  }

  override suspend fun storePicture(url: String): Unit = withContext(Dispatchers.Main) {
    hyprMXOverlay.asyncSavePhoto(url)
  }

  override suspend fun captureImage(): Unit = withContext(Dispatchers.Main) {
    dispatchImageCaptureIntent(activity)
  }

  override suspend fun permissionRequest(permissions: String, permissionId: Int): Unit =
    withContext(Dispatchers.Main) {
      val perms = JSONArray(permissions).toArrayList()
      ActivityCompat.requestPermissions(
        activity,
        perms.toTypedArray(),
        permissionId,
      )
    }

  fun open(url: String) {
    windowOpenAttempt(url)
  }

  override suspend fun setOrientationProperties(
    allowOrientationChange: Boolean,
    forceOrientation: String,
  ): Unit = withContext(Dispatchers.Main) {
    if (forceOrientation == "portrait") {
      hyprMXBaseViewControllerListener.onSetRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT)
    } else if (forceOrientation == "landscape") {
      hyprMXBaseViewControllerListener.onSetRequestedOrientation(
        ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE,
      )
    } else if (!allowOrientationChange) {
      val currentOrientation = Utils.getCurrentScreenOrientation(activity)
      hyprMXBaseViewControllerListener.onSetRequestedOrientation(currentOrientation)
    } else if (forceOrientation == "none") {
      hyprMXBaseViewControllerListener.onSetRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR)
    }
  }

  override suspend fun useCustomClose(enable: Boolean): Unit = withContext(Dispatchers.Main) {
    // setup 1x1 pixel close button
    if (enable) {
      if (rootLayout.findViewById<View>(R.id.hyprmx_custom_close) != null) {
        HyprMXLog.d("Custom close already enabled.")
        return@withContext
      }
      val close1x1Pixel = View(activity)
      close1x1Pixel.id = R.id.hyprmx_custom_close
      close1x1Pixel.setOnClickListener {
        nativeClosePressed()
      }
      val close1x1PixelLayoutParam = RelativeLayout.LayoutParams(
        1.convertDpToPixel(context),
        1.convertDpToPixel(context),
      )
      close1x1PixelLayoutParam.addRule(RelativeLayout.ALIGN_PARENT_TOP)
      close1x1PixelLayoutParam.addRule(RelativeLayout.ALIGN_PARENT_RIGHT)
      close1x1PixelLayoutParam.setMargins(
        0,
        15.convertDpToPixel(activity),
        15.convertDpToPixel(activity),
        0,
      )
      rootLayout.addView(close1x1Pixel, close1x1PixelLayoutParam)

      openMeasurementController?.addFriendlyObstruction(
        close1x1Pixel,
        FriendlyObstructionPurpose.CLOSE_AD,
        "1x1 Close Ad Tracking Pixel",
      )
    } else {
      rootLayout.findViewById<View>(R.id.hyprmx_custom_close)?.let { closePixel ->
        rootLayout.removeView(closePixel)
        openMeasurementController?.removeFriendlyObstruction(closePixel)
      }
    }
  }

  open fun onConfigurationChanged(newConfig: Configuration) {
    // Reset the scroll to 0 on rotation.
    webView.webView.scrollTo(0, 0)
  }

  override fun onDialogDisplayed() {
    webView.pauseJSExecution()
  }

  override fun onDialogDismissed() {
    webView.resumeJSExecution()
  }

  override suspend fun closeAdExperience(): Unit = withContext(Dispatchers.Main) {
    finishWithResult()
  }

  override suspend fun hyprMXBrowserClosed(): Unit = withContext(Dispatchers.Main) {
    hyprMXOverlay.hyprMXBrowserClosed()
  }

  override suspend fun showHyprMXBrowser(viewModelIdentifier: String): Unit =
    withContext(Dispatchers.Main) {
      hyprMXOverlay.showHyprMXBrowser(viewModelIdentifier)
    }

  override suspend fun showPlatformBrowser(url: String): Unit = withContext(Dispatchers.Main) {
    hyprMXOverlay.showPlatformBrowser(url)
    clearBrowserRequest()
  }

  override suspend fun openShareSheet(data: String): Unit = withContext(Dispatchers.Main) {
    hyprMXOverlay.openShareSheet(data)
  }

  override suspend fun createCalendarEvent(data: String): Unit = withContext(Dispatchers.Main) {
    hyprMXOverlay.createCalendarEvent(data)
  }

  override suspend fun showToast(resourceId: Int): Unit = withContext(Dispatchers.Main) {
    hyprMXOverlay.showToast(resourceId)
  }

  override suspend fun openOutsideApplication(url: String): Unit = withContext(Dispatchers.Main) {
    hyprMXOverlay.openOutsideApplication(url)
  }

  override fun onSDKReInit() {
    activity.finish()
  }
}
