package com.hyprmx.android.sdk.activity

import android.annotation.TargetApi
import android.app.DatePickerDialog
import android.app.DatePickerDialog.OnDateSetListener
import android.app.Dialog
import android.content.ActivityNotFoundException
import android.content.res.ColorStateList
import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.PorterDuffColorFilter
import android.graphics.Rect
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.text.InputFilter
import android.text.Spanned
import android.text.method.DigitsKeyListener
import android.view.Gravity
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.view.ViewGroup.LayoutParams
import android.view.ViewTreeObserver
import android.view.inputmethod.EditorInfo
import android.view.inputmethod.InputMethodManager
import android.widget.Button
import android.widget.DatePicker
import android.widget.EditText
import android.widget.LinearLayout
import android.widget.ProgressBar
import android.widget.RadioButton
import android.widget.RadioGroup
import android.widget.ScrollView
import android.widget.TextView
import android.widget.Toast
import androidx.activity.addCallback
import androidx.annotation.RequiresApi
import androidx.appcompat.app.AppCompatActivity
import androidx.browser.customtabs.CustomTabsIntent
import androidx.core.content.ContextCompat
import androidx.core.content.getSystemService
import com.hyprmx.android.R
import com.hyprmx.android.sdk.api.data.DateRequirement
import com.hyprmx.android.sdk.api.data.NaturalNumberRequirement
import com.hyprmx.android.sdk.api.data.RequiredInformation
import com.hyprmx.android.sdk.api.data.SingleSelectSetRequirement
import com.hyprmx.android.sdk.api.data.UiComponents
import com.hyprmx.android.sdk.api.data.UserInfoForm
import com.hyprmx.android.sdk.core.HyprMXReInit
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.footer.FooterContract
import com.hyprmx.android.sdk.footer.FooterFragment
import com.hyprmx.android.sdk.footer.FooterPresenter
import com.hyprmx.android.sdk.utility.HyprMXLog
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Locale
import kotlin.collections.set

/**
 * This package-private class implements the 'Required Information' activity.
 */
class HyprMXRequiredInformationActivity :
  AppCompatActivity(),
  FooterContract.URLPresenter,
  FooterContract.NavigationPresenter,
  HyprMXReInit {

  private lateinit var formContainer: ViewGroup
  private lateinit var scrollView: ScrollView
  private lateinit var titleView: TextView
  private lateinit var submitButton: Button
  private lateinit var progressView: ProgressBar
  private lateinit var datePickerDate: DatePickerDate
  private lateinit var calendar: Calendar
  private lateinit var footerFragment: FooterFragment
  private lateinit var uiComponents: UiComponents
  private lateinit var footerPresenter: FooterPresenter

  private lateinit var requiredInfoController: HyprMXRequiredInfoViewController

  private var datePickerDialog: DatePickerDialog? = null

  private var density: Float = 0.toFloat()

  // Don't allow the offersAvailable request to get sent multiple times.
  // No matter what, we finish when the request completes (or fails).
  private var requestSent = false

  private lateinit var requiredInformations: List<RequiredInformation>

  private var backPressed: Boolean = false

  @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
  override fun onCreate(savedInstanceState: Bundle?) {
    super.onCreate(savedInstanceState)

    // If ActivityResultListener instance is null exit Activity.
    if (ActivityDependencyHolder.requiredInfoControllerFactory == null) {
      HyprMXLog.w("Cancelling ad. Cannot recreate HyprMXRequiredInformationActivity.")
      finish()
      return
    }

    hyprmxDelegate.hyprMXController?.registerSDKReInitListener(this)

    requiredInfoController = ActivityDependencyHolder.requiredInfoControllerFactory!!
      .createRequiredActivityController(this)

    density = resources.displayMetrics.density

    // This fix is to protect SDK from crashing due to issue found in PLAYER-10081.
    if (savedInstanceState != null) {
      HyprMXLog.d("Cancelling ad because activity was destroyed.")
      requiredInfoController.requiredInfoPresentationCancelled()
      finish()
      return
    }

    uiComponents = requiredInfoController.uiComponents
    requiredInformations = requiredInfoController.requiredInformation

    setContentView(R.layout.hyprmx_prequal_layout)

    calendar = Calendar.getInstance()

    datePickerDate = DatePickerDate(
      calendar.get(Calendar.YEAR),
      calendar.get(Calendar.MONTH),
      calendar.get(Calendar.DAY_OF_MONTH),
    )

    // setup linearlayout for info background
    scrollView = findViewById(R.id.hyprmx_scroller)
    formContainer = findViewById(R.id.hyprmx_form_container)
    titleView = findViewById(R.id.hyprmx_title_textview)
    progressView = findViewById(R.id.hyprmx_progress)

    // Setup views regard to Required Information
    createRequiredInfoContainer(requiredInformations)

    footerFragment =
      supportFragmentManager.findFragmentById(R.id.hyprmx_footer_fragment) as FooterFragment

    footerPresenter = FooterPresenter(
      this,
      this,
      uiComponents.userInfoForm.footer,
      footerFragment,
      false,
      requiredInfoController.imageCacheManager,
    )

    bindUserInfoFormWithViews(uiComponents.userInfoForm)

    onBackPressedDispatcher.addCallback {
      backPressed = true
      requiredInfoController.requiredInfoPresentationCancelled()
      finish()
    }
  }

  override fun onStop() {
    datePickerDialog?.dismiss()
    super.onStop()
  }

  /**
   * This fixes PLAYER-12886, where the app's activity that initializes our SDK has
   * launchMode set to singleTask.
   */
  override fun onDestroy() {
    try {
      if (!backPressed && !requestSent) {
        requiredInfoController.onAdDismissed(false)
      }
    } catch (exception: Exception) {
      HyprMXLog.e("Error cleaning up required info activity.")
    }
    super.onDestroy()
  }

  override fun dispatchTouchEvent(event: MotionEvent): Boolean {
    if (event.action == MotionEvent.ACTION_DOWN) {
      val v = currentFocus
      if (v is EditText) {
        val outRect = Rect()
        v.getGlobalVisibleRect(outRect)
        if (!outRect.contains(event.rawX.toInt(), event.rawY.toInt())) {
          v.clearFocus()

          getSystemService<InputMethodManager>()?.hideSoftInputFromWindow(v.windowToken, 0)
        }
      }
    }
    return super.dispatchTouchEvent(event)
  }

  @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
  private fun createRequiredInfoContainer(requiredInformation: List<RequiredInformation>) {
    for (requirement in requiredInformation) {
      val textView = TextView(this)
      textView.text = requirement.title
      textView.textSize = 18f
      val params = LinearLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT)
      params.setMargins(
        (10 * density).toInt(),
        (5 * density).toInt(),
        0,
        (5 * density).toInt(),
      )

      submitButton = findViewById(R.id.hyprmx_submit_button)

      when (requirement) {
        is DateRequirement -> {
          val editText = EditText(this)

          editText.contentDescription = requirement.name
          editText.tag = requirement
          editText.isFocusable = false
          editText.isFocusableInTouchMode = false
          editText.isClickable = true
          editText.setOnClickListener {
            val setListener =
              DetachableSetListener.wrap { _, year, monthOfYear, dayOfMonth ->
                val sdf = SimpleDateFormat("yyyy-MM-dd", Locale.US)
                calendar.clear()
                calendar.set(year, monthOfYear, dayOfMonth)
                val date = calendar.time
                editText.setText(sdf.format(date))
                datePickerDate.year = year
                datePickerDate.month = monthOfYear
                datePickerDate.day = dayOfMonth
              }

            datePickerDialog = DatePickerDialog(
              this@HyprMXRequiredInformationActivity,
              setListener,
              datePickerDate.year,
              datePickerDate.month,
              datePickerDate.day,
            )

            datePickerDialog!!.setTitle(requirement.title)
            if (!this@HyprMXRequiredInformationActivity.isFinishing) {
              datePickerDialog!!.show()
            }
            setListener.clearOnDetach(datePickerDialog!!)
          }
          val newParams =
            LinearLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT)
          newParams.setMargins(
            (10 * density).toInt(),
            (5 * density).toInt(),
            (10 * density).toInt(),
            (5 * density).toInt(),
          )
          formContainer.addView(textView, params)
          formContainer.addView(editText, newParams)
        }
        is SingleSelectSetRequirement -> {
          val group = RadioGroup(this)
          group.tag = requirement
          for (value in requirement.values) {
            val button = RadioButton(this)
            button.tag = value.value
            button.text = value.label
            button.setTextColor(Color.BLACK)
            button.buttonTintList =
              ContextCompat.getColorStateList(this, R.color.hyprmx_prequal_radio_button)
            group.addView(button)
          }
          formContainer.addView(textView, params)
          formContainer.addView(group, params)
        }
        is NaturalNumberRequirement -> {
          val editTextWithError =
            layoutInflater.inflate(
              R.layout.hyprmx_edit_text_with_error,
              null,
            ) as LinearLayout
          editTextWithError.tag = requirement
          val editText = editTextWithError
            .findViewById<EditText>(R.id.hyprmx_editText)
          val titleView = editTextWithError
            .findViewById<TextView>(R.id.hyprmx_titleView)
          val errorView = editTextWithError
            .findViewById<TextView>(R.id.hyprmx_errorView)
          titleView.text = requirement.title
          editText.contentDescription = requirement.name
          editText.imeOptions = EditorInfo.IME_FLAG_NO_EXTRACT_UI
          editText.hint = requirement.hint
          editText.keyListener = DigitsKeyListener.getInstance("0123456789")
          editText.filters = arrayOf(
            InputFilterMax(requirement.max),
            InputFilter.LengthFilter(requirement.max.toString().length),
          )

          val editTextParams =
            LinearLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT)
          editTextParams.setMargins(
            (10 * density).toInt(),
            (5 * density).toInt(),
            (10 * density).toInt(),
            0,
          )
          val errorTextParams =
            LinearLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT)
          errorTextParams.setMargins(
            (14 * density).toInt(),
            0,
            (10 * density).toInt(),
            (5 * density).toInt(),
          )
          titleView.layoutParams = params
          editText.layoutParams = editTextParams
          errorView.layoutParams = errorTextParams
          formContainer.addView(editTextWithError)
        }
      }
    }

    // Lollipop's implementation of setBackgroundTintList() is broken, doesn't add a tint.
    if (Build.VERSION.SDK_INT == Build.VERSION_CODES.LOLLIPOP) {
      submitButton.background.colorFilter =
        PorterDuffColorFilter(
          ContextCompat.getColor(this, R.color.hyprmx_submit_red),
          PorterDuff.Mode.SRC,
        )
    } else {
      submitButton.backgroundTintList =
        ColorStateList.valueOf(ContextCompat.getColor(this, R.color.hyprmx_submit_red))
    }
    submitButton.setTextColor(Color.WHITE)
    submitButton.setOnClickListener { onSubmitClicked(requiredInformation, it) }
  }

  private fun valueForRequirement(requiredInformation: NaturalNumberRequirement): String {
    val editTextWithError = formContainer
      .findViewWithTag<LinearLayout>(requiredInformation)
    return editTextWithError.findViewById<EditText>(R.id.hyprmx_editText).text.toString()
  }

  private fun valueForRequirement(requiredInformation: SingleSelectSetRequirement): String? {
    var value: String? = null
    val group = formContainer.findViewWithTag<RadioGroup>(requiredInformation)
    for (count in 0 until group.childCount) {
      val child = group.getChildAt(count) as RadioButton
      if (child.isChecked) {
        value = child.tag as String
        break
      }
    }
    return value
  }

  private fun valueForRequirement(requiredInformation: DateRequirement): String {
    return formContainer.findViewWithTag<EditText>(requiredInformation).text.toString()
  }

  private fun onSubmitClicked(
    requiredInformation: List<RequiredInformation>,
    submitButtonView: View,
  ) {
    var allInformationGathered = true
    val requiredInfo = HashMap<String, String>()
    label@ for (requirement in requiredInformation) {
      val value = when (requirement) {
        is DateRequirement -> valueForRequirement(requirement)
        is SingleSelectSetRequirement -> valueForRequirement(requirement)
        is NaturalNumberRequirement -> valueForRequirement(requirement)
        else -> {
          HyprMXLog.e("Requirement type not supported")
          null
        }
      }

      if (value.isNullOrEmpty()) {
        HyprMXLog.v("RequiredInformation not entered: ${requirement.name}")
        allInformationGathered = false
        break
      }

      when (requirement) {
        is DateRequirement -> requiredInfo[requirement.name] = value
        is SingleSelectSetRequirement -> requiredInfo[requirement.name] = value
        is NaturalNumberRequirement -> {
          val editTextWithError = formContainer
            .findViewWithTag<LinearLayout>(requirement)
          // val numberField = editTextWithError.findViewById<EditText>(R.id.hyprmx_editText)
          val errorView = editTextWithError
            .findViewById<TextView>(R.id.hyprmx_errorView)
          try {
            // will throw NumberFormatException if value cannot be converted to integer
            val number = Integer.parseInt(value)
            if (number >= requirement.min && number <= requirement.max) {
              requiredInfo[requirement.name] = value
            } else {
              throw NumberFormatException("RequiredInformation not entered: " + requirement.name)
            }
          } catch (nfe: NumberFormatException) {
            HyprMXLog.v(nfe.localizedMessage)
            allInformationGathered = false
            errorView.text = requirement.invalidAnswerMsg
            errorView.visibility = View.VISIBLE
            break@label
          }
        }
      }
    }

    if (allInformationGathered && !requestSent) {
      requestSent = true

      requiredInfoController.requiredInfoPresentationCompletedWithParams(
        requiredInfo,
      )
      finish()
    } else {
      val yOffset = submitButtonView.top + submitButtonView.height
      val message =
        if (requestSent) {
          resources.getString(R.string.hyprmx_MSG_PLEASE_WAIT)
        } else {
          resources.getString(R.string.hyprmx_MSG_PLEASE_FILL_IN_ALL_FIELDS)
        }
      val toast = Toast.makeText(applicationContext, message, Toast.LENGTH_LONG)
      toast.setGravity(Gravity.TOP or Gravity.CENTER_HORIZONTAL, 0, yOffset)
      toast.show()
    }
  }

  /**
   * Bind UserInfoForm data with their view, (setting text, loading their images)
   */
  private fun bindUserInfoFormWithViews(form: UserInfoForm) {
    // set the title information
    titleView.text = form.title
    titleView.textSize = form.title_size.toFloat()

    val layoutParams =
      LinearLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT)
    submitButton.layoutParams = layoutParams
    submitButton.setPadding(
      (10 * density + 0.5f).toInt(),
      (10 * density + 0.5f).toInt(),
      (10 * density + 0.5f).toInt(),
      (10 * density + 0.5f).toInt(),
    )

    scrollView.visibility = View.VISIBLE
    progressView.visibility = View.GONE
  }

  /**
   * Wrapper class for DatePickerDialog.OnDateSetListener to clear the reference to the listener
   * when the dialog window is detached. This fixes the issue of potential memory leaks
   * on OS prior to Android Lollipop when using dialogs.
   */
  class DetachableSetListener private constructor(
    private var dateSetListener: OnDateSetListener?,
  ) : OnDateSetListener {

    override fun onDateSet(view: DatePicker, year: Int, monthOfYear: Int, dayOfMonth: Int) {
      dateSetListener?.onDateSet(view, year, monthOfYear, dayOfMonth)
    }

    @TargetApi(Build.VERSION_CODES.KITKAT)
    internal fun clearOnDetach(dialog: Dialog) {
      dialog.window!!
        .decorView
        .viewTreeObserver
        .addOnWindowAttachListener(
          object : ViewTreeObserver.OnWindowAttachListener {
            override fun onWindowAttached() {}

            override fun onWindowDetached() {
              dateSetListener = null
            }
          },
        )
    }

    companion object {

      fun wrap(delegate: OnDateSetListener): DetachableSetListener {
        return DetachableSetListener(delegate)
      }
    }
  }

  internal class InputFilterMax(private val max: Int) : InputFilter {

    override fun filter(
      source: CharSequence,
      start: Int,
      end: Int,
      dest: Spanned,
      dstart: Int,
      dend: Int,
    ): CharSequence? {
      try {
        val replacement = source.subSequence(start, end).toString()

        val newVal = dest.toString().substring(0, dstart) + replacement +
          dest.toString().substring(
            dend,
            dest.toString().length,
          )

        val input = Integer.parseInt(newVal)

        if (input <= max) {
          return null
        }
      } catch (nfe: NumberFormatException) {
        HyprMXLog.d("NumberFormatException for EditText field input: " + nfe.localizedMessage)
      }

      return ""
    }
  }

  internal inner class DatePickerDate(var year: Int, var month: Int, var day: Int)

  override fun didTapURL(url: String) {
    val builder = CustomTabsIntent.Builder()
    val customTabsIntent = builder.build()
    try {
      customTabsIntent.launchUrl(this, Uri.parse(url))
    } catch (ex: ActivityNotFoundException) {
      HyprMXLog.d("Could not launch activity")
    }
  }

  // FooterContract.NavigationPresenter implementation for footer DAA icon support
  override fun didTapForward() {
  }
  override fun didTapBack() {
  }
  override fun didTapIcon(index: Int) {
    if (index == 0) {
      uiComponents.userInfoForm.footer.icon1?.let { image ->
        didTapURL(image.link)
      }
    } else if (index == 1) {
      uiComponents.userInfoForm.footer.icon2?.let { image ->
        didTapURL(image.link)
      }
    }
  }

  override fun onSDKReInit() {
    finish()
  }
}
