package com.hyprmx.android.sdk.api.data

import com.hyprmx.android.sdk.utility.Result
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.util.ArrayList

/**
 * Representation of Required Information
 */
internal interface RequiredInformation {
  val name: String
  val title: String

  enum class Type {
    Date,
    SingleSelectSet,
    NaturalNumber,
  }

  companion object {
    fun fromJson(jsonString: String): Result<RequiredInformation> {
      return try {
        val json = JSONObject(jsonString)
        val reqInfo = when (Type.valueOf(json.getString(FIELD_TYPE))) {
          Type.Date -> {
            DateRequirement.fromJSON(json)
          }
          Type.SingleSelectSet -> SingleSelectSetRequirement.fromJSON(json)
          Type.NaturalNumber -> NaturalNumberRequirement.fromJSON(json)
        }

        Result.Success(reqInfo)
      } catch (exception: Exception) {
        Result.Failure("Exception parsing required information.", 0, exception)
      }
    }

    @JvmStatic
    fun fromRequiredInfoJSON(requiredInfoString: String?): Result<List<RequiredInformation>> {
      if (requiredInfoString.isNullOrEmpty()) {
        return Result.Failure("No required info to parse.", 0)
      }

      return try {
        val requirementsJsonArray = JSONArray(requiredInfoString)
        val requiredInformations = ArrayList<RequiredInformation>()
        for (i in 0 until requirementsJsonArray.length()) {
          when (val result = fromJson(requirementsJsonArray.get(i).toString())) {
            is Result.Success -> {
              requiredInformations.add(result.value)
            }
            is Result.Failure -> {
              return Result.Failure(result.message, result.code, result.throwable)
            }
          }
        }
        return Result.Success(requiredInformations)
      } catch (exception: JSONException) {
        Result.Failure("Exception parsing required information.", 1, exception)
      }
    }

    private const val FIELD_TYPE = "Type"
  }
}

const val FIELD_NAME = "Name"
const val FIELD_TITLE = "Title"

private class BaseRequiredInfo(override val title: String, override val name: String) :
  RequiredInformation

internal class DateRequirement(private val requiredInfo: RequiredInformation) :
  RequiredInformation by requiredInfo {
  companion object {
    fun fromJSON(jsonObject: JSONObject): DateRequirement {
      val title = jsonObject.getString(FIELD_TITLE)
      val name = jsonObject.getString(FIELD_NAME)
      return DateRequirement(BaseRequiredInfo(title, name))
    }
  }
}

internal class SingleSelectSetRequirement(
  private val requiredInfo: RequiredInformation,
  val values: List<Value>,
) : RequiredInformation by requiredInfo {
  companion object {
    fun fromJSON(jsonObject: JSONObject): SingleSelectSetRequirement {
      val title = jsonObject.getString(FIELD_TITLE)
      val name = jsonObject.getString(FIELD_NAME)
      val values = ArrayList<Value>()
      val valuesJsonArray = jsonObject.optJSONArray(FIELD_VALUES)
      valuesJsonArray?.let {
        for (i in 0 until valuesJsonArray.length()) {
          values.add(Value.fromJson(valuesJsonArray.get(i).toString()))
        }
      }
      return SingleSelectSetRequirement(BaseRequiredInfo(title, name), values)
    }

    private const val FIELD_VALUES = "Values"
  }
}

internal class NaturalNumberRequirement(
  private val requiredInfo: RequiredInformation,
  val hint: String,
  val min: Int,
  val max: Int,
  val invalidAnswerMsg: String,
) : RequiredInformation by requiredInfo {
  companion object {
    fun fromJSON(jsonObject: JSONObject): NaturalNumberRequirement {
      val title = jsonObject.getString(FIELD_TITLE)
      val name = jsonObject.getString(FIELD_NAME)

      val hint = jsonObject.optString(FIELD_HINT, "-")
      val min = jsonObject.optInt(FIELD_MIN, 0)
      val max = jsonObject.optInt(FIELD_MAX, 199)
      val invalidAnswerMessage = jsonObject.optString(
        FIELD_INVALID_ANSWER_MSG,
        "Please enter a number between $min and $max.",
      )
      return NaturalNumberRequirement(
        BaseRequiredInfo(title, name),
        hint,
        min,
        max,
        invalidAnswerMessage,
      )
    }

    private const val FIELD_INVALID_ANSWER_MSG = "invalid_answer_message"
    private const val FIELD_MIN = "min"
    private const val FIELD_MAX = "max"
    private const val FIELD_HINT = "hint"
  }
}
