package com.hyprmx.android.sdk.audio

import android.content.Context
import android.database.ContentObserver
import android.os.Handler
import android.os.Looper
import android.os.RemoteException
import android.provider.Settings
import androidx.core.content.getSystemService
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.utility.HyprMXLog

internal class HyprMXAudioManager(
  private val applicationContext: Context,
  private val sharedAM: AudioManagerShared,
  jsEngine: JSEngine,
) : AudioManager, AudioManagerShared by sharedAM {

  init {
    AudioManagerNativeAdapter(jsEngine, this)
  }

  private val audioObserver by lazy { AudioObserver(applicationContext, Handler(Looper.getMainLooper())) }

  override fun setActive(isActive: Boolean) {
    if (isActive) {
      applicationContext.contentResolver.registerContentObserver(Settings.System.CONTENT_URI, true, audioObserver)
    } else {
      applicationContext.contentResolver.unregisterContentObserver(audioObserver)
    }
  }

  override fun getVolume(): Double = audioObserver.getVolume()

  override fun getIsMuted(): Boolean = audioObserver.getIsMuted()

  override fun getAudioCategory(): String = audioObserver.getAudioCategory()

  /**
   * This is currently unused on Android.  On iOS this call is used to resume the audio session.
   */
  override fun startAudioSession(): String? {
    return null
  }

  inner class AudioObserver internal constructor(private val context: Context, handler: Handler?) :
    ContentObserver(handler) {

    private val am by lazy { context.getSystemService<android.media.AudioManager>()!! }
    private var previousVolume: Int = am.getCurrentVolume()
    private val volume = Volume(previousVolume, am.getMaxVolume())

    override fun onChange(selfChange: Boolean) {
      super.onChange(selfChange)
      sharedAM.onVolumeChange(volume.copy(currentVolume = am.getCurrentVolume()))
    }

    fun getVolume(): Double {
      return (am.getCurrentVolume().toDouble() / am.getMaxVolume().toDouble())
    }

    fun getIsMuted(): Boolean {
      return false
    }

    fun getAudioCategory(): String = "STREAM_MUSIC"

    private fun android.media.AudioManager.getCurrentVolume() = try {
      getStreamVolume(android.media.AudioManager.STREAM_MUSIC)
    } catch (ex: RemoteException) {
      HyprMXLog.e("There was an error getting the current volume.")
      0
    }

    private fun android.media.AudioManager.getMaxVolume() = try {
      getStreamMaxVolume(android.media.AudioManager.STREAM_MUSIC)
    } catch (ex: RemoteException) {
      HyprMXLog.e("There was an error getting stream's max volume.")
      1
    }
  }
}
