package com.hyprmx.android.sdk.banner

import com.hyprmx.android.sdk.core.HyprMXErrors
import com.hyprmx.android.sdk.core.HyprMXReInit
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.mvp.LifecycleEventAdapter
import com.hyprmx.android.sdk.mvp.LifecycleEventHandler
import com.hyprmx.android.sdk.presentation.BannerPresentationEventPublisher
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.utility.HyprMXLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

internal class HyprMXBannerPresenter(
  override var view: HyprMXBannerContract.View?,
  private val placementName: String,
  jsEngine: JSEngine,
  coroutineScope: CoroutineScope,
  eventPublisher: PresentationEventPublisher = BannerPresentationEventPublisher(
    jsEngine = jsEngine,
    placementName = placementName,
  ),
  lifecycleEventAdapter: LifecycleEventHandler = LifecycleEventAdapter(
    eventPublisher,
    coroutineScope,
  ),
  private val sharedInterface: BannerSharedInterface = BannerSharedConnector(eventPublisher),
) : HyprMXBannerContract.Presenter,
  CoroutineScope by coroutineScope,
  LifecycleEventHandler by lifecycleEventAdapter,
  BannerPresenterInterface,
  BannerSharedInterface by sharedInterface,
  HyprMXReInit {

  init {
    attach(BannerPresenterAdapter(this, coroutineScope))
    hyprmxDelegate.hyprMXController?.registerSDKReInitListener(this)
  }

  override fun cleanup() {
    destroy()
    view = null
  }

  override fun loadAd(
    definedSize: HyprMXBannerSize,
    actualWidth: Float,
    actualHeight: Float,
    bidResponse: String?,
  ) {
    if (viewModelIdentifier.isBlank()) {
      HyprMXLog.d("ViewModel does not exist.")
      view?.loadAdFailure(HyprMXErrors.NO_FILL)
      return
    }

    sharedInterface.loadAd(
      definedSize = definedSize,
      actualWidth = actualWidth,
      actualHeight = actualHeight,
      bidResponse = bidResponse,
    )
  }

  override suspend fun loadAdSuccess(): Unit = withContext(Dispatchers.Main) {
    view?.loadAdSuccess()
  }

  override suspend fun loadAdFailed(error: String): Unit = withContext(Dispatchers.Main) {
    val hyprMXError = try { enumValueOf(error) } catch (ex: IllegalArgumentException) {
      HyprMXErrors.UNKNOWN
    }
    view?.loadAdFailure(hyprMXError)
  }

  override suspend fun onAdClicked(): Unit = withContext(Dispatchers.Main) {
    view?.onAdClicked()
  }

  override suspend fun hyprMXBrowserClosed(): Unit = withContext(Dispatchers.Main) {
    view?.hyprMXBrowserClosed()
  }

  override suspend fun showHyprMXBrowser(viewModelIdentifier: String): Unit = withContext(Dispatchers.Main) {
    view?.showHyprMXBrowser(viewModelIdentifier)
  }

  override suspend fun showPlatformBrowser(url: String): Unit = withContext(Dispatchers.Main) {
    view?.showPlatformBrowser(url)
  }

  override suspend fun storePicture(url: String): Unit = withContext(Dispatchers.Main) {
    view?.onAdClicked()
    view?.storePicture(url)
  }

  override suspend fun createCalendarEvent(data: String): Unit = withContext(Dispatchers.Main) {
    view?.onAdClicked()
    view?.createCalendarEvent(data)
  }

  override suspend fun openOutsideApplication(data: String) {
    view?.onAdClicked()
    view?.openOutsideApplication(data)
  }

  override fun startVisibilityTracking(rate: Long, opacityThresholdPercent: Int) {
    view?.startVisibilityTracking(rate, opacityThresholdPercent)
  }

  override fun stopVisibilityTracking() {
    view?.stopVisibilityTracking()
  }

  override fun startOMSession() {
    view?.startOMSession()
  }

  override fun finishOMSession() {
    view?.finishOMSession()
  }

  override fun onAdImpression() {
    view?.onAdImpression()
  }

  override fun onSDKReInit() {
    HyprMXLog.d("Banner - onSDKReInit")
    view?.removePresenter()
    view?.reloadWebView()
    cleanup()
  }
}
