package com.hyprmx.android.sdk.core

import android.content.Context
import android.os.PowerManager
import com.hyprmx.android.sdk.activity.AdControllerFactory
import com.hyprmx.android.sdk.activity.DefaultAdViewControllerFactory
import com.hyprmx.android.sdk.activity.DefaultHyprMXNoAdViewControllerFactory
import com.hyprmx.android.sdk.activity.DefaultHyprMXRequiredInfoControllerFactory
import com.hyprmx.android.sdk.activity.HyprMXNoAdViewControllerFactory
import com.hyprmx.android.sdk.activity.HyprMXRequiredInfoControllerFactory
import com.hyprmx.android.sdk.analytics.ClientErrorController
import com.hyprmx.android.sdk.analytics.ClientErrorControllerIf
import com.hyprmx.android.sdk.analytics.EventController
import com.hyprmx.android.sdk.analytics.ParameterController
import com.hyprmx.android.sdk.analytics.ParameterControllerImpl
import com.hyprmx.android.sdk.api.data.Ad
import com.hyprmx.android.sdk.api.data.RequiredInformation
import com.hyprmx.android.sdk.api.data.UiComponents
import com.hyprmx.android.sdk.audio.AudioEventPublisher
import com.hyprmx.android.sdk.audio.AudioManager
import com.hyprmx.android.sdk.audio.AudioManagerShared
import com.hyprmx.android.sdk.audio.AudioManagerSharedImpl
import com.hyprmx.android.sdk.audio.HyprAudioEventPublisher
import com.hyprmx.android.sdk.audio.HyprMXAudioManager
import com.hyprmx.android.sdk.bidding.BiddingController
import com.hyprmx.android.sdk.consent.ConsentController
import com.hyprmx.android.sdk.consent.ConsentControllerIf
import com.hyprmx.android.sdk.consent.ConsentStatus
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.fullscreen.FullScreenSharedInterface
import com.hyprmx.android.sdk.initialization.InitializationController
import com.hyprmx.android.sdk.initialization.InitializationControllerIf
import com.hyprmx.android.sdk.initialization.UpdateController
import com.hyprmx.android.sdk.initialization.UpdateControllerIf
import com.hyprmx.android.sdk.network.HttpNetworkController
import com.hyprmx.android.sdk.network.JSNetworkController
import com.hyprmx.android.sdk.network.NetworkController
import com.hyprmx.android.sdk.network.NetworkControllerJSInterface
import com.hyprmx.android.sdk.om.OpenMeasurementController
import com.hyprmx.android.sdk.placement.PlacementController
import com.hyprmx.android.sdk.placement.PlacementControllerImpl
import com.hyprmx.android.sdk.powersavemode.PowerSaveModeListener
import com.hyprmx.android.sdk.preferences.LocalStorageController
import com.hyprmx.android.sdk.preferences.PreferencesController
import com.hyprmx.android.sdk.preferences.PreferencesControllerIf
import com.hyprmx.android.sdk.preload.PreloadController
import com.hyprmx.android.sdk.preload.PreloadControllerIf
import com.hyprmx.android.sdk.presentation.ActivityResultListener
import com.hyprmx.android.sdk.presentation.DefaultPresentationFactory
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.presentation.PresentationFactory
import com.hyprmx.android.sdk.utility.ConsoleLogJSInterface
import com.hyprmx.android.sdk.utility.DefaultConsoleLog
import com.hyprmx.android.sdk.utility.HyprMXConnection
import com.hyprmx.android.sdk.utility.HyprMXNativeTimer
import com.hyprmx.android.sdk.utility.HyprMXNativeTimerJSInterface
import com.hyprmx.android.sdk.utility.ImageCacheManager
import com.hyprmx.android.sdk.utility.ImageCacheManagerIf
import com.hyprmx.android.sdk.utility.StorePictureManager
import com.hyprmx.android.sdk.utility.StorePictureManagerImpl
import com.hyprmx.android.sdk.webview.WebViewFactory
import com.hyprmx.android.sdk.webview.WebViewFactoryIf
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.plus

internal interface ApplicationModule {
  val applicationContext: Context
  val distributorId: String
  val scope: CoroutineScope
  val networkController: NetworkController
  val consoleLog: ConsoleLogJSInterface
  val timerController: HyprMXNativeTimerJSInterface
  val jsEngine: JSEngine
  val errorCaptureController: ClientErrorControllerIf
  val powerSaveModeListener: PowerSaveModeListener
  val initializationController: InitializationControllerIf
  val preferenceController: PreferencesControllerIf
  val consentStatus: ConsentStatus
  val consentController: ConsentControllerIf
  val storageHelper: StorageHelperIf
  val localStorageController: LocalStorageController
  val eventController: EventController
  val placementController: PlacementController
  val parameterController: ParameterController
  val audioManagerShared: AudioManagerShared
  val audioManager: AudioManager
  val imageCacheManager: ImageCacheManagerIf
  val preloadController: PreloadControllerIf
  val updateController: UpdateControllerIf
  var openMeasurementController: OpenMeasurementController?
  val storePictureManager: StorePictureManager
  val jsNetworkController: NetworkControllerJSInterface
  val connectionInfo: HyprMXConnection
  val biddingController: BiddingController
  val presenterFactory: PresentationFactory
  val webViewFactory: WebViewFactoryIf
  val audioEventPublisher: AudioEventPublisher

  fun createViewControllerFactory(
    applicationModule: ApplicationModule,
    ad: Ad,
    activityResultListener: ActivityResultListener,
    eventPublisher: PresentationEventPublisher,
    fullScreenSharedConnector: FullScreenSharedInterface,
  ): AdControllerFactory

  fun createNoAdControllerFactory(
    activityResultListener: ActivityResultListener,
    uiComponents: UiComponents,
  ): HyprMXNoAdViewControllerFactory

  fun createRequiredInformationControllerFactory(
    activityResultListener: ActivityResultListener,
    imageCacheManager: ImageCacheManagerIf,
    uiComponents: UiComponents,
    requiredInformation: List<RequiredInformation>,
  ): HyprMXRequiredInfoControllerFactory
}

internal class DefaultApplicationModule(
  override val applicationContext: Context,
  override val distributorId: String,
  override val scope: CoroutineScope = MainScope() + CoroutineName("HyprMXController"),
  override val networkController: NetworkController =
    HttpNetworkController(context = applicationContext),
  override val connectionInfo: HyprMXConnection = HyprMXConnection(applicationContext),
  override val jsEngine: JSEngine = JSEngine(),
  override val errorCaptureController: ClientErrorControllerIf = ClientErrorController(
    jsEngine = jsEngine,
  ).also {
    jsEngine.errorController = it
  },
  override val powerSaveModeListener: PowerSaveModeListener = PowerSaveModeListener(
    applicationContext,
    powerManager = applicationContext.getSystemService(Context.POWER_SERVICE) as PowerManager,
    scope = scope,
  ),
  override val initializationController: InitializationControllerIf = InitializationController(
    jsEngine,
    errorCaptureController = errorCaptureController,
    context = applicationContext,
    scope = scope,
  ),
  override val preferenceController: PreferencesControllerIf = PreferencesController(
    applicationContext,
    jsEngine,
    scope,
  ),
  override val consentStatus: ConsentStatus = ConsentStatus.CONSENT_STATUS_UNKNOWN,
  override val consentController: ConsentControllerIf = ConsentController(
    jsEngine,
    consentStatus,
    scope,
  ),
  override val storageHelper: StorageHelperIf = StorageHelper(applicationContext),
  override val localStorageController: LocalStorageController = LocalStorageController(
    applicationContext,
    jsEngine,
  ),
  override val eventController: EventController = EventController(
    jsEngine,
    errorCaptureController,
    networkController,
  ),
  override val placementController: PlacementController = PlacementControllerImpl(
    jsEngine,
  ),
  override val parameterController: ParameterController = ParameterControllerImpl(
    applicationContext,
    jsEngine,
    powerSaveModeListener,
    connectionInfo,
    scope = scope,
  ),
  override val audioManagerShared: AudioManagerShared = AudioManagerSharedImpl(
    jsEngine,
  ),
  override val audioManager: AudioManager = HyprMXAudioManager(
    applicationContext,
    audioManagerShared,
    jsEngine,
  ),
  override val imageCacheManager: ImageCacheManagerIf = ImageCacheManager(
    applicationContext,
    networkController,
  ),
  override val preloadController: PreloadControllerIf = PreloadController(
    jsEngine,
    imageCacheManager,
    applicationContext = applicationContext,
    coroutineScope = scope,
  ),
  override val updateController: UpdateControllerIf = UpdateController(
    networkController,
    storageHelper,
    errorCaptureController,
  ),
  override var openMeasurementController: OpenMeasurementController? = null,
  override val storePictureManager: StorePictureManager = StorePictureManagerImpl(
    context = applicationContext,
    networkController = networkController,
  ),
  override val consoleLog: ConsoleLogJSInterface = DefaultConsoleLog(jsEngine),
  override val timerController: HyprMXNativeTimerJSInterface = HyprMXNativeTimer(jsEngine, scope),
  override val jsNetworkController: NetworkControllerJSInterface = JSNetworkController(networkController, jsEngine, scope),
  override val biddingController: BiddingController = BiddingController(jsEngine),
  override val presenterFactory: PresentationFactory = DefaultPresentationFactory(jsEngine, scope),
  override val webViewFactory: WebViewFactoryIf = WebViewFactory(applicationContext, preloadController),
  override val audioEventPublisher: AudioEventPublisher = HyprAudioEventPublisher(jsEngine, scope),
) : ApplicationModule {
  override fun createViewControllerFactory(
    applicationModule: ApplicationModule,
    ad: Ad,
    activityResultListener: ActivityResultListener,
    eventPublisher: PresentationEventPublisher,
    fullScreenSharedConnector: FullScreenSharedInterface,
  ): AdControllerFactory {
    return DefaultAdViewControllerFactory(
      DefaultViewControllerModule(
        applicationModule = applicationModule,
        ad = ad,
        activityResultListener = activityResultListener,
        eventPublisher = eventPublisher,
        fullScreenSharedConnector = fullScreenSharedConnector,
      ),
    )
  }

  override fun createNoAdControllerFactory(
    activityResultListener: ActivityResultListener,
    uiComponents: UiComponents,
  ): HyprMXNoAdViewControllerFactory {
    return DefaultHyprMXNoAdViewControllerFactory(
      activityResultListener = activityResultListener,
      uiComponents = uiComponents,
      scope = scope,
    )
  }

  override fun createRequiredInformationControllerFactory(
    activityResultListener: ActivityResultListener,
    imageCacheManager: ImageCacheManagerIf,
    uiComponents: UiComponents,
    requiredInformation: List<RequiredInformation>,
  ): HyprMXRequiredInfoControllerFactory {
    return DefaultHyprMXRequiredInfoControllerFactory(
      activityResultListener = activityResultListener,
      imageCacheManager = imageCacheManager,
      uiComponents = uiComponents,
      requiredInformation = requiredInformation,
      scope = scope,
    )
  }
}
