package com.hyprmx.android.sdk.core

import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.utility.HyprMXLog
import kotlinx.coroutines.suspendCancellableCoroutine
import java.util.UUID
import kotlin.coroutines.Continuation
import kotlin.coroutines.resume

internal class DefaultContinuationController(
  private val jsEngine: JSEngine,
) : ContinuationJS, ContinuationNativeInterface {
  private val continuationMap = mutableMapOf<String, Continuation<Any?>>()

  init {
    jsEngine.addJavascriptInterface(this, JS_INTERFACE_NAME)
  }

  @RetainMethodSignature
  override fun resume(callerId: String, result: Any?) {
    HyprMXLog.d("resume $callerId")
    continuationMap[callerId]?.resume(result)
    continuationMap.remove(callerId)
  }

  @RetainMethodSignature
  override fun resumeWithError(callerId: String, error: Any?) {
    HyprMXLog.e("resumeWithError $callerId")
    continuationMap[callerId]?.resume(null)
    continuationMap.remove(callerId)
  }

  override suspend fun asyncEvaluateScriptForResponse(
    method: String,
    caller: String,
    args: String,
  ): Any? = suspendCancellableCoroutine { continuation ->
    val newId = UUID.randomUUID().toString()
    continuationMap[newId] = continuation
    val asyncWrapperScript =
      """
        globalThis.callAsync(
          "$newId",
          "$method",
          $caller,
          $args
        )
      """.trimMargin()

    jsEngine.evaluate(asyncWrapperScript)
    continuation.invokeOnCancellation {
      HyprMXLog.d("asyncEvaluateScriptForResponse cancelled for Id: $newId")
      resumeWithError(newId, it)
    }
  }

  companion object {
    private const val JS_INTERFACE_NAME = "NativeContinuationController"
  }
}

internal interface ContinuationJS {
  suspend fun asyncEvaluateScriptForResponse(
    method: String,
    caller: String,
    args: String,
  ): Any?
}

interface ContinuationNativeInterface {
  fun resume(callerId: String, result: Any?)
  fun resumeWithError(callerId: String, error: Any?)
}
