package com.hyprmx.android.sdk.core

import android.content.Context
import androidx.annotation.Keep
import com.hyprmx.android.sdk.consent.ConsentStatus
import com.hyprmx.android.sdk.placement.Placement

/**
 * This represents the interface to interact with the HyprMX SDK
 */
@Keep
interface HyprMXIf : HyprMXMediation {

  /**
   * [context] The context
   * [distributorId] The distribution id provided to you by HyprMX.
   * [listener] The HyprMX Initialization Listener. HyprMX will callback to this when initialization is complete
   */
  fun initialize(
    context: Context,
    distributorId: String,
    listener: (InitResult) -> Unit,
  ) {
    initialize(
      context = context,
      distributorId = distributorId,
      listener = object : HyprMXInitializationListener {
        override fun onInitialized(result: InitResult) {
          listener(result)
        }
      },
    )
  }

  /**
   * [context] The context
   * [distributorId] The distribution id provided to you by HyprMX.
   * [listener] The HyprMX Initialization Listener. HyprMX will callback to this when initialization is complete
   */
  fun initialize(
    context: Context,
    distributorId: String,
    listener: HyprMXInitializationListener,
  )

  /**
   * [context] The context
   * [distributorId] The distribution id provided to you by HyprMX.
   */
  suspend fun initialize(
    context: Context,
    distributorId: String,
  ): InitResult

  /**
   * Sets the user's age restriction.
   *
   * [enabled] Indicates if the user should be handled as Age Restricted.
   */
  fun setAgeRestrictedUser(enabled: Boolean)

  /**
   * Gets the placement for the given name.
   */
  fun getPlacement(placementName: String): Placement

  /**
   * Gets the HyprMX initialization state.
   */
  fun getInitializationState(): HyprMXState

  /**
   * Sets the GDPR consent status.
   */
  fun setConsentStatus(consentStatus: ConsentStatus)

  /**
   * Enables test inventory for the current application session.
   * This needs to be set BEFORE initialization in order to ensure inventory
   */
  fun enableTestMode()

  /**
   * Gets list of all available placements.
   */
  fun getPlacements(): Set<Placement>

  /**
   * Retrieves the bidding session token.
   * May return null if HyprMX is not initialized
   */
  fun sessionToken(): String?

  /**
   * [listener] Set the HyprMX Audio Ad Listener. HyprMX will callback to this anytime the Audio Ad state changes
   */
  fun setAudioAdListener(listener: HyprMXAudioAdListener?)

  /**
   * Extra data about the user.
   *
   * @param key The key
   * @param value The value of the key
   */
  fun setUserExtras(key: String, value: String?)

  /**
   * The initialization listener.
   *
   * When provided, HyprMX will callback to [onInitialized] when initialization has finished.
   * If there was a problem initializing, check the logs for more details on the failure.
   */
  @Keep
  interface HyprMXInitializationListener {

    /**
     *  The initialization process has finished.
     *
     *  @param result The initialization result whether successful or not.
     *  In case of failure, an error message may provided with the failure details.
     */
    fun onInitialized(result: InitResult)
  }

  /**
   * The Audio Ad listener.
   *
   * When provided, HyprMX will callback to [onAdAudioStart] when an Audio ad has requested the audio
   * stream. HyprMX will callback to [onAdAudioEnd] when the audio ad has completed audio.
   *
   * The application listening to these event should pause their MEDIA stream on [onAdAudioStart]
   * and resume on [onAdAudioEnd]
   */
  @Keep
  interface HyprMXAudioAdListener {

    /**
     *  The Audio Ad started playing.
     */
    fun onAdAudioStart()

    /**
     * The Audio Ad stopped playing.
     */
    fun onAdAudioEnd()
  }
}

interface HyprMXMediation {
  /**
   * Set the version of unity used to build application
   */
  fun setUnityVersion(version: String?)

  /**
   * Set the mediator version information
   *
   * @param mediator The mediator managing HyprMX.  Use one from HyprMXMediationProvider or provide your own
   * @param mediatorSDKVersion The version of the mediator SDK
   * @param adapterVersion The version of the mediation adapter
   */
  fun setMediationProvider(mediator: String?, mediatorSDKVersion: String?, adapterVersion: String?)
}

internal class HyprMXMediationAdapter : HyprMXMediation {
  internal var unityVersion: String? = null
    private set
  internal var mediator: String? = null
    private set
  internal var mediatorSDKVersion: String? = null
    private set
  internal var mediatorAdapterVersion: String? = null
    private set

  override fun setUnityVersion(version: String?) {
    unityVersion = version
  }

  override fun setMediationProvider(
    mediator: String?,
    mediatorSDKVersion: String?,
    adapterVersion: String?,
  ) {
    this.mediator = mediator
    this.mediatorSDKVersion = mediatorSDKVersion
    this.mediatorAdapterVersion = adapterVersion
  }
}
